package pl.redlink.push.service.repository

import pl.redlink.push.di.redlink
import pl.redlink.push.http.*
import pl.redlink.push.provider.RedlinkIdProvider
import pl.redlink.push.service.model.Device
import pl.redlink.push.service.model.Status
import pl.redlink.push.service.model.User
import java.net.URL

internal class PushRepositoryImpl(
    private val restClient: RestClient = redlink().restClient
) : PushRepository {

    override suspend fun lastPush(registrationId: String): Response {
        return restClient.call(
            Request(
                url = URL("$BASE_URL${lastPushEndpoint(registrationId)}"),
                method = Method.GET,
                headers = authorizationHeaders,
                contentType = HttpClient.CONTENT_TYPE_JSON
            )
        )
    }

    override suspend fun registerDevice(device: Device): Response {
        return restClient.asyncCallWithRetry(
            Request(
                url = URL("$BASE_URL$SEND_DEVICE_ENDPOINT"),
                body = device.toJson(),
                method = Method.POST,
                headers = authorizationHeaders,
                contentType = HttpClient.CONTENT_TYPE_JSON
            )
        )
    }

    override suspend fun updateUserData(registrationId: String, user: User): Response {
        return restClient.asyncCallWithRetry(
            Request(
                url = URL("$BASE_URL${updateUserData(registrationId)}"),
                body = user.toJson(),
                method = Method.PATCH,
                headers = authorizationHeaders,
                contentType = HttpClient.CONTENT_TYPE_JSON
            )
        )
    }

    override suspend fun pushStatus(status: Status): Response {
        return restClient.asyncCallWithRetry(
            Request(
                url = URL("$BASE_URL${reportsEndpoint(status.id)}"),
                body = status.toJson(),
                method = Method.POST,
                headers = authorizationHeaders,
                contentType = HttpClient.CONTENT_TYPE_JSON
            )
        )
    }

    companion object {

        private const val BASE_URL = "https://push.redlink.pl/v1/"
        private const val SEND_DEVICE_ENDPOINT = "devices/"
        private fun reportsEndpoint(id: String) = "pushes/$id/reports"
        private fun lastPushEndpoint(id: String) = "devices/$id/last_push"
        private fun updateUserData(id: String) = "devices/$id/user_data"
        private const val AUTHORIZATION = "AShIxRSOf5G015lYhPIXGfcgmSkVgrrwddK"
        private val authorizationHeaders
            get() = mapOf(
                "Authorization" to AUTHORIZATION,
                "X-Redlink-App-Token" to RedlinkIdProvider.getRedlinkToken().orEmpty(),
                "X-Redlink-App-Secret" to RedlinkIdProvider.getRedlinkSecret().orEmpty()
            )
    }

}
