package pl.redlink.push.manager.user

import android.annotation.SuppressLint
import android.content.Context
import pl.redlink.push.Logger
import pl.redlink.push.extension.isEmail
import pl.redlink.push.manager.PREFS_KEY_USER
import pl.redlink.push.manager.PreferencesManager
import pl.redlink.push.service.RegisterDeviceService
import pl.redlink.push.service.UserUpdateService
import java.util.*

@SuppressLint("StaticFieldLeak")
object RedlinkUser {

    private const val MAX_FIELD_SIZE = 64
    private var context: Context? = null
    private fun put(user: RedlinkUserData): Boolean {
        if (isValid(user)) {
            user.toJsonString()?.also {
                remove()
                PreferencesManager.put(PREFS_KEY_USER, it)
                return true
            }
        }
        return false
    }

    private fun isValid(user: RedlinkUserData): Boolean {
        var isValid = true
        if (!user.email.isBlank() && (!user.email.isEmail() || user.email.length > MAX_FIELD_SIZE)) {
            Logger.error("E-mail can be up to $MAX_FIELD_SIZE long and format have to be valid (${user.email})")
            isValid = isValid && false
        }
        if (user.companyName.length > MAX_FIELD_SIZE) {
            Logger.error("Company name can be up to $MAX_FIELD_SIZE long (${user.companyName})")
            isValid = isValid && false
        }
        if (user.firstName.length > MAX_FIELD_SIZE) {
            Logger.error("First name can be up to $MAX_FIELD_SIZE long (${user.firstName})")
            isValid = isValid && false
        }
        if (user.lastName.length > MAX_FIELD_SIZE) {
            Logger.error("Last name can be up to $MAX_FIELD_SIZE long (${user.lastName})")
            isValid = isValid && false
        }
        return isValid
    }

    @JvmSynthetic
    fun init(context: Context) {
        this.context = context
    }

    @JvmStatic
    fun get(): RedlinkUserData? =
        RedlinkUserData.fromJson(PreferencesManager.get(PREFS_KEY_USER) ?: "")
            ?: RedlinkUserData()

    @JvmStatic
    fun remove() {
        PreferencesManager.remove(PREFS_KEY_USER)
    }

    class Edit {

        private val user: RedlinkUserData = get() ?: RedlinkUserData()
        fun email(email: String): Edit {
            user.email = email
            return this
        }

        fun phone(phone: String): Edit {
            user.phone = phone
            return this
        }

        fun firstName(firstName: String): Edit {
            user.firstName = firstName
            return this
        }

        fun lastName(lastName: String): Edit {
            user.lastName = lastName
            return this
        }

        fun companyName(companyName: String): Edit {
            user.companyName = companyName
            return this
        }

        fun customValue(key: String, value: String): Edit {
            user.customStringValues[key] = value
            return this
        }

        fun customValue(key: String, value: Int): Edit {
            user.customIntValues[key] = value
            return this
        }

        fun customValue(key: String, value: Boolean): Edit {
            user.customBooleanValues[key] = value
            return this
        }

        fun customValue(key: String, value: Date): Edit {
            user.customDateValues[key] = (value.time / 1000).toInt()
            return this
        }

        fun removeCustomValues(): Edit {
            user.customStringValues.clear()
            user.customIntValues.clear()
            user.customBooleanValues.clear()
            user.customDateValues.clear()
            return this
        }

        fun save(): Boolean {
            return put(user).also {
                context?.let { UserUpdateService.enqueueRequest(it) }
            }
        }

    }
}
