package pl.redlink.push.lifecycle

import android.app.Activity
import android.app.AlertDialog
import android.content.Context
import pl.redlink.push.extension.isApplicationInForeground
import pl.redlink.push.fcm.NotificationActionBroadcast
import pl.redlink.push.fcm.PushMessage
import pl.redlink.push.fcm.action.Action
import pl.redlink.push.service.PushStatusService
import java.security.SecureRandom

open class DefaultInAppPushHandler : InAppPushHandler {

    override fun handleLastPush(activity: Activity, pushMessage: PushMessage) {
        if (!activity.isDestroyed && !activity.isFinishing && activity.isApplicationInForeground()) {
            showDialog(activity, pushMessage)
        }
    }

    private fun showDialog(activity: Activity, pushMessage: PushMessage) {
        AlertDialog.Builder(activity)
                .setTitle(pushMessage.title)
                .setMessage(pushMessage.body)
                .setOnCancelListener { sendCancelActionIntent(activity, pushMessage) }
                .apply {
                    if (pushMessage.actions?.actionButton.isNullOrEmpty()) {
                        val mainAction = pushMessage.action
                        if (mainAction != null) {
                            setPositiveButton(android.R.string.ok) { _, _ ->
                                sendActionIntent(context, pushMessage, mainAction)
                            }
                            setNegativeButton(android.R.string.cancel) { _, _ ->
                                sendCancelActionIntent(context, pushMessage)
                            }
                        } else {
                            setPositiveButton(android.R.string.ok) { _, _ ->
                                sendCancelActionIntent(context, pushMessage)
                            }
                        }
                    }
                }
                .apply {
                    pushMessage.actions?.actionButton?.also {
                        it.getOrNull(0)?.also { actionButton ->
                            setNegativeButton(actionButton.text) { _, _ ->
                                actionButton.action?.let { action -> sendActionIntent(activity, pushMessage, action) }
                            }
                        }

                        it.getOrNull(1)?.also { actionButton ->
                            setPositiveButton(actionButton.text) { _, _ ->
                                actionButton.action?.let { action -> sendActionIntent(activity, pushMessage, action) }
                            }
                        }
                    }
                }
                .show()
    }

    private fun sendActionIntent(context: Context, pushMessage: PushMessage, action: Action) {
        val random = SecureRandom()
        NotificationActionBroadcast.getActionPendingIntent(
                context = context,
                pushMessage = pushMessage,
                action = action,
                requestCode = random.nextInt()).send()

    }

    private fun sendCancelActionIntent(context: Context, pushMessage: PushMessage) {
        val random = SecureRandom()
        NotificationActionBroadcast.getCancelPendingIntent(context, pushMessage, random.nextInt(), PushStatusService.DETAILS_DIALOG_DISMISS_ACTION).send()
    }

}