package pl.redlink.push.http

import pl.redlink.push.http.interceptor.loggerInterceptor
import kotlinx.coroutines.*
import java.net.URL
import kotlin.coroutines.CoroutineContext

internal class RestClient(private val client: HttpClient) : CoroutineScope {

    override val coroutineContext: CoroutineContext
        get() = Dispatchers.Default + SupervisorJob()

    private val responseInterceptors: MutableList<(Request, Response) -> Unit> = mutableListOf(loggerInterceptor)

    fun get(url: String, responseCallback: (Response) -> Unit) = async {
        val response = request(url, method = Method.GET)
        responseCallback.invoke(response)
    }

    fun post(url: String, body: String?, responseCallback: (Response) -> Unit) = async {
        val response = request(url, body = body, method = Method.POST)
        responseCallback.invoke(response)
    }

    private fun request(url: String, body: String? = null, method: Method): Response {
        val request = Request(
                url = URL(url),
                method = method,
                body = body
        )
        return call(request)
    }

    fun call(request: Request): Response {
        return client.execute(request)
                .apply {
                    responseInterceptors.forEach { it.invoke(request, this) }
                }
    }

    suspend fun asyncCallWithRetry(request: Request, times: Int = 4): Response {
        return backOffRetry(times) {
            call(request)
        }
    }

    private suspend fun backOffRetry(
            times: Int = Int.MAX_VALUE,
            initialDelay: Long = 100, // 0.1 second
            maxDelay: Long = 1000,    // 1 second
            factor: Double = 2.0,
            block: suspend () -> Response): Response {
        var currentDelay = initialDelay
        repeat(times - 1) {
            val response = block()
            if (response.isSuccess()) {
                return response
            }
            println("Request failed, retry in $maxDelay ms")
            delay(currentDelay)
            currentDelay = (currentDelay * factor).toLong().coerceAtMost(maxDelay)
        }
        return block()
    }

}