package pl.redlink.push.service

import android.content.Context
import android.content.Intent
import androidx.core.app.JobIntentService
import pl.redlink.push.Logger
import pl.redlink.push.di.RedlinkModule.pushRepository
import pl.redlink.push.provider.FirebaseTokenProvider
import pl.redlink.push.provider.RedlinkIdProvider
import pl.redlink.push.service.model.Device
import pl.redlink.push.service.model.Status
import kotlinx.coroutines.runBlocking

internal class PushStatusService : JobIntentService() {

    override fun onHandleWork(intent: Intent) {
        runBlocking {
            RedlinkIdProvider.getFcmProjectId()?.let { fcmProjectId ->
                val fcmToken = FirebaseTokenProvider.getFcmToken(fcmProjectId)
                fcmToken?.let {
                    val pushId = intent.getStringExtra(EXTRA_PUSH_ID)
                    val occurredAt = intent.getLongExtra(EXTRA_OCCURRED_AT, System.currentTimeMillis() / 1000)
                    val status = intent.getSerializableExtra(EXTRA_STATUS) as Status.ReportPushStatus
                    val details = intent.getStringExtra(EXTRA_DETAILS)
                    val actionId = intent.getStringExtra(EXTRA_ACTION_ID)
                    val requestStatus = Status(
                            id = pushId,
                            status = status,
                            occurredAt = occurredAt,
                            type = Device.Type.ANDROID,
                            registrationId = fcmToken,
                            actionId = actionId,
                            details = details)
                    pushRepository.pushStatus(requestStatus)
                }
            }
        }
    }

    companion object {

        const val DETAILS_NOTIFICATION_CLICK_ACTION = "NOTIFICATION_CLICK_ACTION"
        const val DETAILS_NOTIFICATION_SWIPE_ACTION = "NOTIFICATION_SWIPE_ACTION"
        const val DETAILS_DIALOG_DISMISS_ACTION = "DIALOG_DISMISS_ACTION"

        private const val EXTRA_PUSH_ID = "redlink-extra-push-id"
        private const val EXTRA_STATUS = "redlink-extra-status"
        private const val EXTRA_OCCURRED_AT = "redlink-extra-occurred-at"
        private const val EXTRA_DETAILS = "redlink-extra-details"
        private const val EXTRA_ACTION_ID = "redlink-extra-action-id"

        private const val PUSH_STATUS_JOB_ID = 1000

        fun enqueueRequest(context: Context, pushId: String, status: Status.ReportPushStatus, occurredAt: Long, actionId: String? = null, details: String? = null) {
            val intent = Intent(context, PushStatusService::class.java).apply {
                putExtra(EXTRA_PUSH_ID, pushId)
                putExtra(EXTRA_STATUS, status)
                putExtra(EXTRA_OCCURRED_AT, occurredAt)
                putExtra(EXTRA_DETAILS, details)
                putExtra(EXTRA_ACTION_ID, actionId)
            }
            enqueueWork(context, PushStatusService::class.java, PUSH_STATUS_JOB_ID, intent)
        }

    }

}