package pl.redlink.push.fcm.notification

import android.content.Context
import androidx.core.app.NotificationManagerCompat
import pl.redlink.push.Logger
import pl.redlink.push.fcm.KEY_ID
import pl.redlink.push.fcm.KEY_TITLE
import pl.redlink.push.fcm.PushMessage
import pl.redlink.push.service.PushStatusService
import pl.redlink.push.service.model.Status
import com.google.firebase.messaging.RemoteMessage

class RedlinkNotification {

    companion object {

        const val DEFAULT_NOTIFICATION_ID = 0

        fun handlePushMessage(context: Context, pushMessage: PushMessage) {
            Logger.info(pushMessage.toString())
            updatePushStatus(context, pushMessage)
            if (pushMessage.silent == null || pushMessage.silent == false) {
                showNotification(context, pushMessage)
            }
        }

        fun providePushMessage(remoteMessage: RemoteMessage?): PushMessage? {
            if (canHandleRemoteMessage(remoteMessage)) {
                remoteMessage?.data?.let {
                    return PushMessage(remoteMessage.data)
                }
            }
            return null
        }

        private fun canHandleRemoteMessage(remoteMessage: RemoteMessage?): Boolean {
            return remoteMessage?.data?.let {
                val pushMessage = PushMessage(it)
                return pushMessage.has(KEY_ID) && pushMessage.has(KEY_TITLE)
            } ?: false
        }

        private fun showNotification(context: Context, pushMessage: PushMessage) {
            with(NotificationManagerCompat.from(context)) {
                val notificationId = pushMessage.id?.hashCode() ?: DEFAULT_NOTIFICATION_ID
                notify(notificationId, NotificationCreator.create(context, notificationId, pushMessage).build())
            }
        }

        private fun updatePushStatus(context: Context, pushMessage: PushMessage) {
            pushMessage.id?.let { id ->
                PushStatusService.enqueueRequest(
                        context = context,
                        pushId = id,
                        status = Status.ReportPushStatus.RECEIVED,
                        occurredAt = System.currentTimeMillis() / 1000)
            }
        }

    }

}