package pl.redlink.push.extension

import android.content.Context
import android.graphics.drawable.Drawable
import android.net.Uri
import android.os.Build
import androidx.core.content.ContextCompat
import androidx.vectordrawable.graphics.drawable.VectorDrawableCompat

private const val INVALID_RESOURCE_ID = 0
private const val PLATFORM_VD_CLAZZ = "android.graphics.drawable.VectorDrawable"

internal fun Context.getDrawableResourceIdByName(name: String): Int? {
    val identifier = resources.getIdentifier(name.substringBeforeLast("."), "drawable", packageName)
    return identifier.takeIf { it != INVALID_RESOURCE_ID && isDrawableProcessible(ContextCompat.getDrawable(this, identifier)) }
}

internal fun Context.getStringResourceByName(name: String): String? {
    val identifier = resources.getIdentifier(name, "string", packageName)
    return if (identifier != INVALID_RESOURCE_ID) getString(identifier) else null
}

internal fun Context.getRawResourceIdByName(name: String): Int? {
    val identifier = resources.getIdentifier(name.substringBeforeLast("."), "raw", packageName)
    return identifier.takeIf { it != INVALID_RESOURCE_ID }
}

internal fun Context.getRawResourceUriByName(name: String): Uri? =
    getRawResourceIdByName(name)?.let {
        Uri.parse("android.resource://" + packageName + "/raw/" + name.substringBeforeLast("."))
    }

internal fun Context.versionCode(): Long = try {
    val packageInfo = packageManager.getPackageInfo(packageName, 0)
    if (Build.VERSION.SDK_INT >= 28) {
        packageInfo.longVersionCode
    } else {
        packageInfo.versionCode.toLong()
    }
} catch (e: Exception) {
    INVALID_VERSION_CODE
}

private fun isVectorDrawable(drawable: Drawable?): Boolean {
    return drawable is VectorDrawableCompat || PLATFORM_VD_CLAZZ == drawable?.javaClass?.name
}

private fun isVectorSupport() = Build.VERSION.SDK_INT >= 21

private fun isDrawableProcessible(drawable: Drawable?): Boolean {
    if (isVectorSupport() || !isVectorDrawable(drawable)) {
        return true
    }
    return false
}

const val INVALID_VERSION_CODE = -1L