package pl.redlink.push.analytics

import android.annotation.SuppressLint
import android.content.Context
import pl.redlink.push.Logger
import pl.redlink.push.database.Event
import pl.redlink.push.database.EventsDatabaseHelper
import pl.redlink.push.extension.isJson
import pl.redlink.push.service.EventsService
import java.util.*

@SuppressLint("StaticFieldLeak")
object RedlinkAnalytics {

    private const val MAX_EVENT_NAME_LENGTH = 64
    private const val MAX_ADDITIONAL_KEY_NAME_LENGTH = 64
    private const val CHAR_SPACE = " "
    private const val TIME_DIVIDER = 1000
    private var context: Context? = null
    private var eventsDatabaseHelper: EventsDatabaseHelper? = null

    @JvmSynthetic
    internal fun init(context: Context, eventsDatabaseHelper: EventsDatabaseHelper) {
        this.context = context
        this.eventsDatabaseHelper = eventsDatabaseHelper
    }

    @JvmStatic
    fun trackEvent(name: String, params: Map<String, Any> = emptyMap(), userDataJson: String? = null) {

        if (!isEventNameValid(name)) {
            Logger.error("Invalid event name -> $name (Event names can be up to $MAX_EVENT_NAME_LENGTH, can't contain spaces")
            return
        }

        if (!areParamsValid(params)) {
            return
        }

        if (!isUserDataValid(userDataJson)) {
            Logger.error("Invalid JSON content in userData")
            return
        }

        eventsDatabaseHelper?.insertEvent(RedlinkAnalytics.prepareEvent(name, params, userDataJson))
        context?.let { EventsService.enqueueRequest(it) }
    }

    @JvmStatic
    fun trackEvent(name: String, key: String, value: String) {
        RedlinkAnalytics.trackEvent(name, mapOf(key to value))
    }

    @JvmStatic
    fun trackEvent(name: String, key: String, value: Int) {
        RedlinkAnalytics.trackEvent(name, mapOf(key to value))
    }

    @JvmStatic
    fun trackEvent(name: String, key: String, value: Boolean) {
        RedlinkAnalytics.trackEvent(name, mapOf(key to value))
    }

    @JvmStatic
    fun trackEvent(name: String, key: String, value: Date) {
        RedlinkAnalytics.trackEvent(name, mapOf(key to value))
    }

    internal fun prepareEvent(name: String, params: Map<String, Any>, userDataJson: String? = null): Event {
        val event = Event(
            name = name,
            timestamp = System.currentTimeMillis() / RedlinkAnalytics.TIME_DIVIDER,
            userData = userDataJson
        )
        for ((key, value) in params) {
            when (value) {
                is String -> {
                    event.stringParams[key] = value
                }
                is Int -> {
                    event.integerParams[key] = value
                }
                is Boolean -> {
                    event.booleanParams[key] = value
                }
                is Date -> {
                    event.dateParams[key] = (value.time / RedlinkAnalytics.TIME_DIVIDER).toInt()
                }
            }
        }
        return event
    }

    private fun isUserDataValid(userDataJson: String?): Boolean {
        return userDataJson == null || userDataJson.isJson() ?: false
    }

    private fun isEventNameValid(name: String): Boolean {
        return !name.contains(CHAR_SPACE) && name.length <= MAX_EVENT_NAME_LENGTH
    }

    private fun areParamsValid(params: Map<String, Any>): Boolean {
        var isValid = true
        params.forEach {
            if (it.key.length > MAX_ADDITIONAL_KEY_NAME_LENGTH) {
                Logger.error("Invalid event param key -> ${it.key} (Key names can be up to $MAX_ADDITIONAL_KEY_NAME_LENGTH long")
                isValid = isValid && false
            }
        }
        return isValid
    }

}
