package pl.redlink.push

import android.annotation.SuppressLint
import android.app.Application
import android.content.Context
import android.content.pm.ApplicationInfo
import androidx.browser.customtabs.CustomTabsIntent
import pl.redlink.push.di.redlink
import pl.redlink.push.extension.alsoIfTrue
import pl.redlink.push.extension.ifNull
import pl.redlink.push.fcm.RedlinkActionHandler
import pl.redlink.push.lifecycle.ActivityLifecycleHandler
import pl.redlink.push.lifecycle.InAppPushHandler
import pl.redlink.push.lifecycle.LifecycleController
import pl.redlink.push.manager.token.FcmTokenManager
import pl.redlink.push.manager.user.RedlinkUser
import pl.redlink.push.provider.RedlinkIdProvider
import com.google.firebase.FirebaseApp
import pl.redlink.push.analytics.RedlinkAnalytics
import pl.redlink.push.database.EventsDatabaseHelper
import pl.redlink.push.manager.PreferencesManager


@SuppressLint("StaticFieldLeak")
object RedlinkApp {

    private val activityLifecycleHandler: ActivityLifecycleHandler = redlink().activityLifeCycleHandler
    private var applicationContext: Context? = null

    /**
     * Enable debug mode (true to enable logger).
     */
    @JvmStatic
    fun setDebug(debug: Boolean) {
        Logger.enabled = debug
    }

    /**
     * Lifecycle controller that inform sdk about additional events like onNewIntent.
     */
    @JvmStatic
    val lifecycleController: LifecycleController = redlink().lifecycleController

    /**
     * Initialize redlink sdk
     */
    @JvmStatic fun initRedlinkApp(context: Context): Boolean {
        RedlinkIdProvider.context = context
        RedlinkApp.setDebug(0 != context.applicationInfo.flags and ApplicationInfo.FLAG_DEBUGGABLE)
        return RedlinkApp.checkRequirements().alsoIfTrue {
            RedlinkApp.applicationContext = context
            PreferencesManager.init(context)
            RedlinkApp.initUser(context)
            RedlinkApp.initAnalytics(context)
            RedlinkApp.initAppLifecycle(context)
        }
    }

    /**
     * CustomTabs builder for WEBVIEW actions
     */
    @JvmStatic fun customTabsBuilder(builder: CustomTabsIntent.Builder) {
        redlink().customTabProvider.customTabBuilder = builder
    }

    /**
     * Custom InAppPushManager handler
     */
    @JvmStatic fun customInAppPushHandler(inAppPushHandler: InAppPushHandler) {
        redlink().inAppPushManagerManager.inAppPushHandler = inAppPushHandler
    }

    /**
     * Custom Action handler
     */
    @JvmStatic fun customActionHandler(redlinkActionHandler: RedlinkActionHandler) {
        redlink().redlinkActionHandler = redlinkActionHandler
    }

    private fun checkRequirements(): Boolean {
        val redlinkAppId = RedlinkIdProvider.getRedlinkAppId().ifNull {
            Logger.error("Redlink project id has not been provided")
        }
        val redlinkToken = RedlinkIdProvider.getRedlinkToken().ifNull {
            Logger.error("Redlink token has not been provided")
        }
        val redlinkSecret = RedlinkIdProvider.getRedlinkSecret().ifNull {
            Logger.error("Redlink secret has not been provided")
        }
        val fcmProjectId = RedlinkIdProvider.getFcmProjectId().ifNull {
            Logger.error("Firebase project id has not been provided")
        }
        val redlinkEventsToken = RedlinkIdProvider.getRedlinkEventsToken().ifNull {
            Logger.error("Redlink events token has not been provided")
        }
        return !redlinkAppId.isNullOrEmpty() && !fcmProjectId.isNullOrEmpty()
                && !redlinkToken.isNullOrEmpty() && !redlinkSecret.isNullOrEmpty() &&
                !redlinkEventsToken.isNullOrEmpty()
    }

    private fun initUser(context: Context) {
        FirebaseApp.initializeApp(context)
        RedlinkUser.init(context)
        FcmTokenManager.init(context)
    }

    private fun initAnalytics(context: Context) {
        RedlinkAnalytics.init(context, EventsDatabaseHelper(context))
    }

    private fun initAppLifecycle(context: Context) {
        (context as? Application)?.apply {
            registerActivityLifecycleCallbacks(RedlinkApp.activityLifecycleHandler)
            registerComponentCallbacks(RedlinkApp.activityLifecycleHandler)
        }
    }

}
