package pl.redlink.push.fcm

import android.app.PendingIntent
import android.content.BroadcastReceiver
import android.content.Context
import android.content.Intent
import androidx.core.app.NotificationManagerCompat
import pl.redlink.push.di.redlink
import pl.redlink.push.fcm.action.Action
import pl.redlink.push.fcm.action.ActionProcessor
import pl.redlink.push.fcm.notification.RedlinkNotification
import pl.redlink.push.manager.PREFS_KEY_LAST_INTERACTED_PUSH
import pl.redlink.push.manager.PreferencesManager
import pl.redlink.push.service.PushStatusService
import pl.redlink.push.service.model.Status

internal class NotificationActionBroadcast : BroadcastReceiver() {

    private val actionProcessor: ActionProcessor = redlink().actionProcessor
    private val redlinkActionHandler: RedlinkActionHandler = redlink().redlinkActionHandler

    override fun onReceive(context: Context, intent: Intent?) {
        val notificationId = intent?.getIntExtra(EXTRA_NOTIFICATION_ID, RedlinkNotification.DEFAULT_NOTIFICATION_ID)
                ?: RedlinkNotification.DEFAULT_NOTIFICATION_ID
        NotificationManagerCompat.from(context).cancel(notificationId)

        val actionType = intent?.getStringExtra(EXTRA_ACTON_TYPE) ?: TYPE_CANCEL
        val action = intent?.getParcelableExtra<Action>(EXTRA_ACTION)

        if (intent?.hasExtra(EXTRA_PUSH_MESSAGE) == true) {
            val pushMessage: PushMessage = intent.getParcelableExtra(EXTRA_PUSH_MESSAGE)
            val details = if (intent.hasExtra(EXTRA_DETAILS)) intent.getStringExtra(EXTRA_DETAILS) else null
            pushMessage.id?.let { id ->
                PushStatusService.enqueueRequest(
                        context = context,
                        pushId = id,
                        status = Status.ReportPushStatus.REACTED_ON,
                        occurredAt = System.currentTimeMillis() / 1000,
                        actionId = action?.id,
                        details = details)
            }
            PreferencesManager.put(PREFS_KEY_LAST_INTERACTED_PUSH, pushMessage.id ?: "")
        }

        if (intent?.hasExtra(EXTRA_ACTION) == true && action != null) {
            redlinkActionHandler.handleAction(context, action)
        } else if (actionType == TYPE_ACTION) {
            val pushMessage: PushMessage? = intent?.getParcelableExtra(EXTRA_PUSH_MESSAGE)
            pushMessage?.action?.let {
                redlinkActionHandler.handleAction(context, it)
            } ?: actionProcessor.runApp(context)
        }

    }

    companion object {

        private const val EXTRA_ACTON_TYPE = "pl.redlink.push.fcm.extra-action-type"
        private const val EXTRA_DETAILS = "pl.redlink.push.fcm.extra-details"
        private const val TYPE_ACTION = "pl.redlink.push.fcm.type-action"
        private const val TYPE_CANCEL = "pl.redlink.push.fcm.type-cancel"

        const val EXTRA_ACTION = "pl.redlink.push.fcm.extra-action"
        const val EXTRA_PUSH_MESSAGE = "pl.redlink.push.fcm.extra-push-message"
        const val EXTRA_NOTIFICATION_ID = "pl.redlink.push.fcm.extra-notification-id"

        fun getPendingIntent(context: Context, pushMessage: PushMessage, requestCode: Int, details: String? = null): PendingIntent {
            val intent = Intent(context, NotificationActionBroadcast::class.java)
            intent.putExtra(EXTRA_PUSH_MESSAGE, pushMessage)
            intent.putExtra(EXTRA_ACTON_TYPE, TYPE_ACTION)
            intent.putExtra(EXTRA_DETAILS, details)
            return PendingIntent.getBroadcast(context, requestCode, intent, PendingIntent.FLAG_UPDATE_CURRENT)
        }

        fun getCancelPendingIntent(context: Context, pushMessage: PushMessage, requestCode: Int, details: String? = null): PendingIntent {
            val intent = Intent(context, NotificationActionBroadcast::class.java)
            intent.putExtra(EXTRA_PUSH_MESSAGE, pushMessage)
            intent.putExtra(EXTRA_ACTON_TYPE, TYPE_CANCEL)
            intent.putExtra(EXTRA_DETAILS, details)
            return PendingIntent.getBroadcast(context, requestCode, intent, PendingIntent.FLAG_UPDATE_CURRENT)
        }

        fun getActionPendingIntent(context: Context, pushMessage: PushMessage, action: Action, notificationId: Int? = null, requestCode: Int): PendingIntent {
            val intent = Intent(context, NotificationActionBroadcast::class.java).apply {
                setAction("action-${action.type}-${action.url}")
                notificationId?.let { putExtra(EXTRA_NOTIFICATION_ID, notificationId) }
                putExtra(EXTRA_ACTION, action)
                putExtra(EXTRA_ACTON_TYPE, TYPE_ACTION)
                putExtra(EXTRA_PUSH_MESSAGE, pushMessage)
            }
            return PendingIntent.getBroadcast(context, requestCode, intent, PendingIntent.FLAG_UPDATE_CURRENT)
        }

    }

}