package pl.redlink.push.ui

import android.app.Activity
import pl.redlink.push.di.redlink
import pl.redlink.push.extension.jsonToMap
import pl.redlink.push.fcm.PushMessage
import pl.redlink.push.lifecycle.DefaultInAppPushHandler
import pl.redlink.push.lifecycle.InAppPushHandler
import pl.redlink.push.manager.PREFS_KEY_LAST_INTERACTED_PUSH
import pl.redlink.push.manager.PreferencesManager
import pl.redlink.push.manager.token.FcmTokenManager
import pl.redlink.push.service.repository.PushRepository
import kotlinx.coroutines.*
import pl.redlink.push.Logger
import kotlin.coroutines.CoroutineContext

internal class InAppPushManager(
    var inAppPushHandler: InAppPushHandler? = DefaultInAppPushHandler(),
    private val pushRepository: PushRepository = redlink().pushRepository
) : CoroutineScope {

    override val coroutineContext: CoroutineContext
        get() = Dispatchers.Main + SupervisorJob()

    fun checkPushes(activity: Activity) = launch {
        FcmTokenManager.get()?.let {
            if (it.isNotEmpty()) {
                val pushMessage = fetchPushAsync(it).await()
                val lastInteractedPush = PreferencesManager.get(PREFS_KEY_LAST_INTERACTED_PUSH)
                if (pushMessage?.id != null && pushMessage.id != lastInteractedPush) {
                    inAppPushHandler?.handleLastPush(activity, pushMessage)
                }
            }
        }
    }

    private fun fetchPushAsync(fcmToken: String): Deferred<PushMessage?> {
        return async(context = Dispatchers.IO) {
            val response = pushRepository.lastPush(fcmToken)
            when {
                response.isSuccess() -> response.body?.let { body ->
                    val map = jsonToMap(body)
                    PushMessage(map)
                }
                response.status == 404 -> {
                    Logger.info("There is no in-app pushes available")
                    null
                }
                else -> null
            }
        }
    }
}