package pl.redlink.push.service

import android.content.Context
import android.content.Intent
import androidx.core.app.JobIntentService
import pl.redlink.push.di.redlink
import pl.redlink.push.manager.token.FcmTokenManager
import pl.redlink.push.manager.user.RedlinkUser
import pl.redlink.push.provider.RedlinkIdProvider
import pl.redlink.push.service.model.RedlinkEvent
import kotlinx.coroutines.Job
import kotlinx.coroutines.launch
import kotlinx.coroutines.runBlocking
import pl.redlink.push.Logger
import pl.redlink.push.database.EventsDatabaseHelper

class EventsService : JobIntentService() {

    private val jobs = mutableListOf<Job>()

    override fun onHandleWork(intent: Intent) {
        runBlocking {
            val eventsDatabaseHelper =
                EventsDatabaseHelper(applicationContext)
            val apiRepository = redlink().apiRepository
            val fcmToken = FcmTokenManager.get()
            val events = eventsDatabaseHelper.getEvents()
            if (events.isNotEmpty()) {
                fcmToken?.let { token ->
                    events.forEach { event ->
                        jobs += launch {
                            val appId = RedlinkIdProvider.getRedlinkAppId()
                            val user = RedlinkUser.get()
                            val redlinkEvent = RedlinkEvent(
                                    pushId = "$token|||$appId",
                                    eventName = event.name,
                                    createdAt = event.timestamp,
                                    email = user?.email ?: "",
                                    number = user?.phone ?: ""
                            ).also {
                                it.stringFields = event.stringParams
                                it.booleanFields = event.booleanParams
                                it.integerFields = event.integerParams
                                it.dateFields = event.dateParams
                            }
                            val response = apiRepository.sendEvent(redlinkEvent)
                            if (response.isSuccess()) {
                                Logger.info("Event has been deleted from DB: ${event.id}")
                                eventsDatabaseHelper.deleteEventsById(listOf(event.id))
                            }
                        }
                    }
                }
                jobs.forEach { it.join() }
            } else {
                stopSelf()
            }
        }
    }

    companion object {

        private const val EVENTS_JOB_ID = 3000

        fun enqueueRequest(context: Context) {
            val intent = Intent(context, EventsService::class.java).apply {
            }
            enqueueWork(context, EventsService::class.java, EVENTS_JOB_ID, intent)
        }

    }

}