package pl.redlink.push.fcm.notification

import android.app.NotificationChannel
import android.app.NotificationManager
import android.content.Context
import android.os.Build
import androidx.annotation.RequiresApi
import androidx.core.app.NotificationCompat
import pl.redlink.push.R
import pl.redlink.push.extension.toEnumOrDefault
import pl.redlink.push.fcm.NotificationActionBroadcast
import pl.redlink.push.fcm.PushMessage
import pl.redlink.push.fcm.notification.extender.ActionsExtender
import pl.redlink.push.fcm.notification.extender.ImageExtender
import pl.redlink.push.fcm.notification.extender.SoundExtender
import pl.redlink.push.fcm.notification.extender.TextExtender
import pl.redlink.push.service.PushStatusService
import pl.redlink.push.service.model.Visibility
import java.security.SecureRandom

internal class NotificationCreator {

    companion object {

        private const val DEFAULT_NOTIFICATION_CHANNEL_ID = "pl.redlink.push.default_channel_id"

        fun create(context: Context, notificationId: Int, pushMessage: PushMessage): NotificationCompat.Builder {
            val builder = NotificationCompat.Builder(context, DEFAULT_NOTIFICATION_CHANNEL_ID)
                    .setSmallIcon(R.drawable.redlink_default_icon)
                    .setContentTitle(pushMessage.title)
                    .setContentText(pushMessage.body)
                    .setAutoCancel(true)
                    .setPriority(NotificationCompat.PRIORITY_DEFAULT)
                    .setVisibility(toEnumOrDefault(Visibility::class.java, pushMessage.lockScreenVisibility, Visibility.PUBLIC).androidValue)
                    .extend(ActionsExtender(context, notificationId, pushMessage))
                    .extend(TextExtender(pushMessage))
                    .extend(ImageExtender(context, pushMessage))
                    .extend(SoundExtender(context, pushMessage))
                    .setDeleteIntent(NotificationActionBroadcast.getCancelPendingIntent(context, pushMessage, SecureRandom().nextInt(), PushStatusService.DETAILS_NOTIFICATION_SWIPE_ACTION))
                    .setContentIntent(NotificationActionBroadcast.getPendingIntent(context, pushMessage, SecureRandom().nextInt(), PushStatusService.DETAILS_NOTIFICATION_CLICK_ACTION))

            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
                createDefaultChannel(context)
            }
            return builder
        }

        @RequiresApi(Build.VERSION_CODES.O)
        private fun createDefaultChannel(context: Context) {
            val name = context.getString(R.string.redlink_default_channel_name)
            val descriptionText = context.getString(R.string.redlink_default_channel_description)
            val importance = NotificationManager.IMPORTANCE_DEFAULT
            val channel = NotificationChannel(DEFAULT_NOTIFICATION_CHANNEL_ID, name, importance).apply {
                description = descriptionText
            }
            val notificationManager = context.getSystemService(Context.NOTIFICATION_SERVICE) as? NotificationManager
            notificationManager?.createNotificationChannel(channel)
        }

    }

}