package pl.redlink.push.extension

import org.json.JSONArray
import org.json.JSONException
import org.json.JSONObject

internal fun JSONObject.getJSONObjectOrNull(key: String?): JSONObject? = getOrNull(key) { getJSONObject(key) }

internal fun JSONObject.getIntOrNull(key: String?): Int? = getOrNull(key) { getInt(key) }

internal fun JSONObject.getStringOrNull(key: String?): String? = getOrNull(key) { getString(key) }

internal fun JSONObject.getStringOrDefault(key: String?, default: String): String = getOrNull(key) { getString(key) }
        ?: default

internal fun JSONObject.getBoolOrNull(key: String?): Boolean? = getOrNull(key) { getBoolean(key) }

internal fun JSONObject.getStringMapOrNull(key: String?): MutableMap<String, String>? = getOrNull(key) { createStringMapOrNull(getJSONObject(key)) }

internal fun <A, B> JSONObject.getMapOrNull(key: String?): MutableMap<A, B>? = getOrNull(key) {
    val map: MutableMap<A, B>? = createMapOrNull(getJSONObject(key))
    return map
}


internal inline fun <reified T> JSONArray.forEach(action: (T?) -> Unit) {
    (0..length()).forEach { action(get(it) as? T) }
}

internal inline fun <reified T, R> JSONArray.map(transform: (T?) -> R): MutableList<R?> =
        mapTo(mutableListOf(), transform)

internal inline fun <reified T, R> JSONArray.mapTo(to: MutableList<R?>, transform: (T?) -> R): MutableList<R?> =
        to.apply {
            (0 until this@mapTo.length()).forEach { i -> (this@mapTo.get(i) as? T)?.let { add(transform(it)) } }
        }

internal inline fun <reified T, R> JSONArray.mapNotNull(transform: (T) -> R): MutableList<R> =
        mapNotNullTo(mutableListOf(), transform)

internal inline fun <reified T, R> JSONArray.mapNotNullTo(to: MutableList<R>, transform: (T) -> R): MutableList<R> =
        to.apply {
            (0 until this@mapNotNullTo.length()).forEach { i -> (this@mapNotNullTo.get(i) as? T)?.let { add(transform(it)) } }
        }

inline fun <T> JSONObject.getOrNull(key: String?, getter: JSONObject.() -> T) =
        if (key == null) null else {
            if (has(key) && !isNull(key)) getter(this) else null
        }

internal fun createJsonArrayOrNull(json: String?): JSONArray? {
    return try {
        JSONArray(json)
    } catch (e: Exception) {
        null
    }
}

@Throws(JSONException::class)
internal fun jsonToMap(json: String): HashMap<String, String> {
    val map = HashMap<String, String>()
    val jObject = JSONObject(json)
    val keys = jObject.keys()
    while (keys.hasNext()) {
        val key = keys.next() as String
        if (!jObject.isNull(key)) {
            val value = jObject.getString(key)
            map[key] = value
        }
    }
    return map
}

internal fun createJsonObjectOrNull(json: String?): JSONObject? {
    return try {
        JSONObject(json)
    } catch (e: Exception) {
        null
    }
}

internal fun createStringMapOrNull(jsonObject: JSONObject): MutableMap<String, String>? {
    return try {
        val map = mutableMapOf<String, String>()

        val keysIterator = jsonObject.keys().iterator()
        while (keysIterator.hasNext()) {
            val key = keysIterator.next()

            var value = jsonObject.get(key)
            if (value is JSONObject) {
                value = createStringMapOrNull(value)
            }

            map[key] = value as String
        }

        map
    } catch (e: Exception) {
        null
    }
}

internal fun <A, B> createMapOrNull(jsonObject: JSONObject): MutableMap<A, B>? {
    return try {
        val map = mutableMapOf<A, B>()
        val keysIterator = jsonObject.keys().iterator()
        while (keysIterator.hasNext()) {
            val key = keysIterator.next() as A
            var value = jsonObject.get(key.toString())
            if (value is JSONObject) {
                val newMap: MutableMap<A, B>? = createMapOrNull(value)
                value = newMap
            }
            map[key] = value as B
        }
        map
    } catch (e: Exception) {
        null
    }
}