package pl.redlink.push.database

import android.content.ContentValues
import android.content.Context
import android.database.sqlite.SQLiteDatabase
import android.database.sqlite.SQLiteOpenHelper
import pl.redlink.push.extension.toAny
import pl.redlink.push.extension.toByteArray

internal class EventsDatabaseHelper(context: Context) : SQLiteOpenHelper(context,
    DATABASE_NAME, null, DATABASE_VERSION) {

    override fun onCreate(db: SQLiteDatabase?) {
        createTableIfNotExist(db)
    }

    override fun onUpgrade(db: SQLiteDatabase?, oldVersion: Int, newVersion: Int) {
        dropTableIfExists(db)
        onCreate(db)
    }

    internal fun createTableIfNotExist(db: SQLiteDatabase?) {
        db?.execSQL(EventsDatabaseTable.QUERY_CREATE_TABLE)
    }

    internal fun dropTableIfExists(db: SQLiteDatabase?) {
        db?.execSQL(EventsDatabaseTable.QUERY_DROP_TABLE)
    }

    internal fun clearTable() {
        writableDatabase.execSQL(EventsDatabaseTable.QUERY_CLEAR_TABLE)
    }

    fun insertEvent(event: Event): Long {
        val contentValues = ContentValues().apply {
            put(EventsDatabaseTable.COLUMN_NAME_NAME, event.name)
            put(EventsDatabaseTable.COLUMN_NAME_TIMESTAMP, event.timestamp)
            event.stringParams.toByteArray()?.also { put(EventsDatabaseTable.COLUMN_NAME_STRING_PARAMETERS, it) }
            event.integerParams.toByteArray()?.also { put(EventsDatabaseTable.COLUMN_NAME_INTEGER_PARAMETERS, it) }
            event.booleanParams.toByteArray()?.also { put(EventsDatabaseTable.COLUMN_NAME_BOOLEAN_PARAMETERS, it) }
            event.dateParams.toByteArray()?.also { put(EventsDatabaseTable.COLUMN_NAME_DATE_PARAMETERS, it) }
        }
        return writableDatabase.insert(EventsDatabaseTable.TABLE_NAME, null, contentValues)
    }

    internal fun deleteEvents(events: List<Event>) {
        events.forEach {
            writableDatabase.delete(
                EventsDatabaseTable.TABLE_NAME,
                    EventsDatabaseTable.COLUMN_NAME_ID + " = " + it.id, null)
        }
    }

    internal fun deleteEventsById(ids: List<Int>) {
        ids.forEach {
            writableDatabase.delete(
                EventsDatabaseTable.TABLE_NAME,
                    EventsDatabaseTable.COLUMN_NAME_ID + " = " + it, null)
        }
    }

    fun getEventById(id: Int): pl.redlink.push.database.Event? {
        return getEvents(
                limit = 1,
                selection = "${EventsDatabaseTable.COLUMN_NAME_ID} = ?",
                selectionArgs = arrayOf(id.toString()))[0]
    }

    fun getEvents(limit: Int? = null, selection: String? = null, selectionArgs: Array<String>? = null): List<Event> {
        val cursor = readableDatabase.query(
            EventsDatabaseTable.TABLE_NAME,
            EventsDatabaseHelper.COLUMNS,
                selection,
                selectionArgs,
                null,
                null,
                null,
                limit?.toString()
        )

        val items = mutableListOf<Event>()
        cursor?.use {
            while (it.moveToNext()) {
                items += Event(
                    id = it.getInt(it.getColumnIndex(EventsDatabaseTable.COLUMN_NAME_ID)),
                    name = it.getString(it.getColumnIndex(EventsDatabaseTable.COLUMN_NAME_NAME)),
                    timestamp = it.getLong(it.getColumnIndex(EventsDatabaseTable.COLUMN_NAME_TIMESTAMP)),
                    stringParams = (it.getBlob(it.getColumnIndex(EventsDatabaseTable.COLUMN_NAME_STRING_PARAMETERS))
                        .toAny() as? MutableMap<String, String>) ?: mutableMapOf(),
                    booleanParams = (it.getBlob(it.getColumnIndex(EventsDatabaseTable.COLUMN_NAME_BOOLEAN_PARAMETERS))
                        .toAny() as? MutableMap<String, Boolean>) ?: mutableMapOf(),
                    integerParams = (it.getBlob(it.getColumnIndex(EventsDatabaseTable.COLUMN_NAME_INTEGER_PARAMETERS))
                        .toAny() as? MutableMap<String, Int>) ?: mutableMapOf(),
                    dateParams = (it.getBlob(it.getColumnIndex(EventsDatabaseTable.COLUMN_NAME_DATE_PARAMETERS))
                        .toAny() as? MutableMap<String, Int>) ?: mutableMapOf()
                )
            }
        }
        return items
    }

    companion object {
        private const val DATABASE_NAME = "Events.db"
        private const val DATABASE_VERSION = 2

        private val COLUMNS = arrayOf(
            EventsDatabaseTable.COLUMN_NAME_ID,
            EventsDatabaseTable.COLUMN_NAME_NAME,
            EventsDatabaseTable.COLUMN_NAME_TIMESTAMP,
            EventsDatabaseTable.COLUMN_NAME_STRING_PARAMETERS,
            EventsDatabaseTable.COLUMN_NAME_BOOLEAN_PARAMETERS,
            EventsDatabaseTable.COLUMN_NAME_INTEGER_PARAMETERS,
            EventsDatabaseTable.COLUMN_NAME_DATE_PARAMETERS
        )
    }

}
