package pl.redlink.push.analytics

import android.annotation.SuppressLint
import android.content.Context
import pl.redlink.push.database.Event
import pl.redlink.push.database.EventsDatabaseHelper
import pl.redlink.push.service.EventsService
import java.util.*

@SuppressLint("StaticFieldLeak")
object RedlinkAnalytics {

    private const val TIME_DIVIDER = 1000
    private var context: Context? = null
    private var eventsDatabaseHelper = lazy {
        RedlinkAnalytics.context?.let {
            EventsDatabaseHelper(it)
        }
    }

    @JvmSynthetic
    fun init(context: Context) {
        this.context = context
    }

    @JvmStatic
    fun trackEvent(name: String, params: Map<String, Any> = emptyMap()) {
        RedlinkAnalytics.eventsDatabaseHelper.value?.insertEvent(
            RedlinkAnalytics.prepareEvent(
                name,
                params
            )
        )
        RedlinkAnalytics.context?.let { EventsService.enqueueRequest(it) }
    }

    @JvmStatic
    fun trackEvent(name: String, key: String, value: String) {
        RedlinkAnalytics.trackEvent(name, mapOf(key to value))
    }

    @JvmStatic
    fun trackEvent(name: String, key: String, value: Int) {
        RedlinkAnalytics.trackEvent(name, mapOf(key to value))
    }

    @JvmStatic
    fun trackEvent(name: String, key: String, value: Boolean) {
        RedlinkAnalytics.trackEvent(name, mapOf(key to value))
    }

    @JvmStatic
    fun trackEvent(name: String, key: String, value: Date) {
        RedlinkAnalytics.trackEvent(name, mapOf(key to value))
    }

    internal fun prepareEvent(name: String, params: Map<String, Any>): Event {
        val event = Event(
            name = name,
            timestamp = System.currentTimeMillis() / RedlinkAnalytics.TIME_DIVIDER
        )
        for ((key, value) in params) {
            when (value) {
                is String -> {
                    event.stringParams[key] = value
                }
                is Int -> {
                    event.integerParams[key] = value
                }
                is Boolean -> {
                    event.booleanParams[key] = value
                }
                is Date -> {
                    event.dateParams[key] = (value.time / RedlinkAnalytics.TIME_DIVIDER).toInt()
                }
            }
        }
        return event
    }
}
