package pl.decerto.hyperon.persistence.validation;

import static java.lang.String.format;
import static java.util.Arrays.asList;

import java.util.HashSet;
import java.util.Set;
import java.util.TreeSet;
import java.util.stream.IntStream;

import org.apache.commons.lang3.StringUtils;

import pl.decerto.hyperon.persistence.exception.HyperonPersistenceMappingException;

public abstract class TableNameValidator {

	private static final Set<String> ANSI_SQL_KEYWORDS = new HashSet<>(asList(
		"as", "case", "create", "delete", "from", "having", "insert", "join", "merge", "null",
		"order by", "prepare", "select", "truncate", "union", "update", "where", "with"
	));

	private static final int ANSI_SQL_MAX_TABLENAME_LEN = 128;

	/**
	 * Whitelist of legal tablename characters: _ a-z A-Z 0-9
	 */
	private static final Set<Character> VALID_TABLENAME_CHARS = createLegalCharacters();

	private TableNameValidator() {
		throw new UnsupportedOperationException("util class");
	}

	/**
	 * If table name is invalid throws exception with proper message:
	 * - Table name is blank
	 * - Table name too long [name]. Max length: 128
	 * - Table name is a reserved keyword: [keyword]
	 * - Table name contains forbidden character: [name]
	 * @param table table name to validate
	 */
	public static void validateTableName(String table) {

		// Rule 1 - nazwa tabeli nie moze byc pusta

		if (StringUtils.isBlank(table)) {
			throw new HyperonPersistenceMappingException("Table name is blank");
		}

		// Rule 2 [ANSI SQL-92] - maksymalna liczba znakow: 128

		if (table.length() > ANSI_SQL_MAX_TABLENAME_LEN) {
			throw new HyperonPersistenceMappingException(
				format("Table name too long [%s]. Max length: %d", table, ANSI_SQL_MAX_TABLENAME_LEN));
		}

		// Rule 3 [ANSI SQL-92] - table name must not be an SQL-92 keyword

		if (ANSI_SQL_KEYWORDS.contains(table.toLowerCase())) {
			throw new HyperonPersistenceMappingException(
				format("Table name is a reserved keyword: [%s]", table)
			);
		}

		// Rule 4 - table name contains only allowed characters

		if (containsIllegalChar(table)) {
			throw new HyperonPersistenceMappingException("Table name contains forbidden character: [" + table + "]");
		}

	}

	private static Set<Character> createLegalCharacters() {

		Set<Character> set = new TreeSet<>();
		set.add('_');

		IntStream.range('a', 'z' + 1).forEach(c -> set.add((char) c));
		IntStream.range('A', 'Z' + 1).forEach(c -> set.add((char) c));
		IntStream.range('0', '9' + 1).forEach(c -> set.add((char) c));

		return set;
	}

	private static boolean isIllegalChar(char c) {
		return !VALID_TABLENAME_CHARS.contains(c);
	}

	private static boolean containsIllegalChar(String table) {
		for (int i = 0; i < table.length(); i++) {
			char c = table.charAt(i);
			if (isIllegalChar(c)) {
				return true;
			}
		}
		return false;
	}

}
