package pl.decerto.hyperon.persistence.model.def;

import static pl.decerto.hyperon.persistence.model.def.EntityTypeState.PERSISTENT;
import static pl.decerto.hyperon.persistence.model.def.EntityTypeState.TRANSIENT;

import java.util.HashMap;
import java.util.Map;
import java.util.Set;

import org.apache.commons.lang3.StringUtils;

/**
 * @author przemek hertel
 */
public class EntityType {

	protected final String name;
	protected String description;
	protected Map<String, PropertyDef> props = new HashMap<>();
	protected EntityTypeState state = PERSISTENT;

	public EntityType(String name) {
		this.name = name;
	}

	public EntityType add(String propertyName, PropertyDef prop) {
		props.put(propertyName, prop);
		return this;
	}

	public Map<String, PropertyDef> getProps() {
		return props;
	}

	public PropertyDef getProp(String propertyName) {
		return props.get(propertyName);
	}

	public int getPropCount() {
		return props.size();
	}

	public String getName() {
		return name;
	}

	public String getDescription() {
		return description;
	}

	public boolean hasDescription() {
		return StringUtils.isNotBlank(description);
	}

	public void setDescription(String description) {
		this.description = StringUtils.isNotBlank(description) ? description.trim() : null;
	}

	public boolean isRoot() {
		return false;
	}

	public boolean isPersistent() {
		return state == PERSISTENT;
	}

	public boolean isTransient() {
		return state == TRANSIENT;
	}

	public void setPersistent(boolean persistent) {
		setState(persistent ? PERSISTENT : TRANSIENT);
	}

	public void setState(EntityTypeState state) {
		this.state = state != null ? state : PERSISTENT;
	}

	protected void scanForEntityTypes(Set<EntityType> all) {
		scanForEntityTypes(all, true);
	}

	protected void scanForEntityTypes(Set<EntityType> all, boolean includeRoot) {

		// remember this type
		if (!isRoot() || includeRoot) {
			all.add(this);
		}

		// scan all children
		for (PropertyDef def : props.values()) {
			if (def.isEntityType()) {
				EntityType child = def.getEntityType();
				if (!all.contains(child)) {
					child.scanForEntityTypes(all, includeRoot);
				}
			}
		}
	}

	@Override
	public String toString() {
		return "EntityType[" + name + " : " + props.keySet() + "]";
	}

	@Override
	public boolean equals(Object o) {
		if (this == o) {
			return true;
		}
		if (!(o instanceof EntityType)) {
			return false;
		}

		EntityType that = (EntityType) o;
		return name.equals(that.name);
	}

	@Override
	public int hashCode() {
		return name.hashCode();
	}
}
