package pl.decerto.hyperon.persistence.model.def;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

import org.apache.commons.lang3.StringUtils;

import pl.decerto.hyperon.persistence.dao.TupleDef;

/**
 * @author przemek hertel
 */
public class BundleDef extends EntityType {

	public static final String BUNDLE_DEF_NAME = "Bundle";

	/**
	 * Root tuple definition (extra fields). May be null or empty.
	 */
	private TupleDef rootTupleDef;

	/**
	 * All non-root tuple definitions.
	 */
	private List<TupleDef> tupleDefs;

	/**
	 * All non-root tuple definitions mapped with key: typeName
	 */
	private Map<String, TupleDef> tupleDefMap;

	/**
	 * All non-root entity types (without root).
	 */
	private List<EntityType> entityTypes;

	/**
	 * All non-root entity types mapped with key: typeName
	 */
	private Map<String, EntityType> entityTypeMap;

	/**
	 * Construct with default root name: Bundle
	 */
	public BundleDef() {
		super(BUNDLE_DEF_NAME);
	}

	/**
	 * Construct with given root name
	 * @param name bundle def name
	 */
	public BundleDef(String name) {
		super(name);
	}

	@Override
	public BundleDef add(String propertyName, PropertyDef prop) {
		super.add(propertyName, prop);
		return this;
	}

	public List<TupleDef> getTupleDefs() {
		return tupleDefs;
	}

	public void setTupleDefs(Collection<TupleDef> tupleDefs) {
		this.tupleDefs = new ArrayList<>(tupleDefs);
	}

	public BundleDef done() {

		// find all non-root entities
		Set<EntityType> all = new HashSet<>();
		scanForEntityTypes(all, false);

		return done(all);
	}

	public BundleDef done(Collection<EntityType> values) {

		// all entities (with root)
		this.entityTypes = new ArrayList<>(values);

		// build entity type map
		entityTypeMap = new HashMap<>();
		for (EntityType type : entityTypes) {
			entityTypeMap.put(type.getName(), type);
		}

		// mapping
		if (tupleDefs == null) {
			tupleDefs = new LinkedList<>();
		}

		// build tuple definitions map
		tupleDefMap = new HashMap<>();
		for (TupleDef tdef : tupleDefs) {
			tupleDefMap.put(tdef.getEntityName(), tdef);
		}

		return this;
	}

	/**
	 * get all non-root entities
	 * @return list containing all entity types
	 * @see EntityType
	 */
	public List<EntityType> getAllEntityTypes() {
		return entityTypes;
	}

	public EntityType findType(String typeName) {
		return entityTypeMap.get(typeName);
	}

	public void setRootTupleDef(TupleDef rootTupleDef) {
		this.rootTupleDef = rootTupleDef;
	}

	public TupleDef getTupleDef(String typeName) {
		return getTupleDef(typeName, false);
	}

	public TupleDef getTupleDef(String typeName, boolean checkRoot) {
		if (checkRoot && Objects.nonNull(rootTupleDef) && StringUtils.equals(typeName, rootTupleDef.getEntityName())) {
			return rootTupleDef;
		} else {
			return tupleDefMap != null ? tupleDefMap.get(typeName) : null;
		}
	}

	public TupleDef getRootTupleDef() {
		return rootTupleDef;
	}

	@Override
	public boolean isRoot() {
		return true;
	}

	/**
	 * method checks if root entity has extra (simple) fields mapped to columns
	 * @return boolean indicating whether root entity has extra (simple) fields mapped to columns
	 */
	public boolean hasExtraFields() {
		return rootTupleDef != null && rootTupleDef.getFieldCount() > 0;
	}

	/**
	 * method counts number of extra (simple) fields mapped to columns
	 * @return number of extra (simple) fields mapped to columns
	 */
	public int getExtraFieldsCnt() {
		return rootTupleDef != null ? rootTupleDef.getFieldCount() : 0;
	}

	@Override
	public String toString() {
		return "BundleDef[" + props.keySet() + "]";
	}

}
