package org.xelevra.architecture.util;

import java.util.concurrent.atomic.AtomicBoolean;

import rx.Notification;
import rx.Observable;
import rx.functions.Action0;
import rx.functions.Action1;
import rx.functions.Func1;
import rx.subjects.BehaviorSubject;
import rx.subjects.PublishSubject;
import rx.subjects.Subject;

public class RefresherProvider<T> {
    private final Observable<T> refresher;
    private final Subject<Throwable, Throwable> errors;
    private Subject<T, T> subject;
    private AtomicBoolean isRefreshing;


    public RefresherProvider(Observable<T> coldRefreshObservable) {
        this(coldRefreshObservable, null);
    }

    public RefresherProvider(Observable<T> coldRefreshObservable, T defaultValue) {
        refresher = coldRefreshObservable;
        subject = defaultValue == null ? BehaviorSubject.<T>create().toSerialized() : BehaviorSubject.create(defaultValue).toSerialized();
        isRefreshing = new AtomicBoolean(false);
        errors = PublishSubject.<Throwable>create().toSerialized();
    }

    public void refresh() {
        if (isRefreshing.getAndSet(true)) return;
        refresher.materialize().doOnNext(new Action1<Notification<T>>() {
            @Override
            public void call(Notification<T> notification) {
                if (notification.isOnError()) errors.onNext(notification.getThrowable());
            }
        }).filter(new Func1<Notification<T>, Boolean>() {
            @Override
            public Boolean call(Notification<T> tNotification) {
                return tNotification.isOnNext();
            }
        }).<T>dematerialize().doOnTerminate(new Action0() {
            @Override
            public void call() {
                isRefreshing.set(false);
            }
        }).subscribe(new Action1<T>() {
            @Override
            public void call(T t) {
                subject.onNext(t);
            }
        });
    }

    public Observable<T> observe() {
        return subject.asObservable();
    }

    public Observable<Throwable> errors() {
        return errors.asObservable();
    }
}
