/*
 * Decompiled with CFR 0.152.
 */
package org.wso2.carbon.tools.converter.utils;

import java.io.IOException;
import java.io.OutputStream;
import java.net.URI;
import java.nio.charset.Charset;
import java.nio.file.CopyOption;
import java.nio.file.DirectoryStream;
import java.nio.file.FileSystem;
import java.nio.file.FileSystems;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileAttribute;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.jar.Attributes;
import java.util.jar.Manifest;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.stream.IntStream;
import org.wso2.carbon.tools.exception.CarbonToolException;

public class BundleGeneratorUtils {
    private static final Logger logger = Logger.getLogger(BundleGeneratorUtils.class.getName());

    private BundleGeneratorUtils() {
    }

    /*
     * Enabled aggressive block sorting
     */
    public static void convertFromJarToBundle(Path jarFile, Path targetDirectory, Manifest manifest, String extensionPrefix) throws IOException, CarbonToolException {
        if (!Files.isDirectory(targetDirectory, new LinkOption[0])) {
            String message = "Path target directory does not point to a directory";
            throw new CarbonToolException(message);
        }
        Path tempJarFilePathHolder = jarFile.getFileName();
        if (tempJarFilePathHolder == null) {
            String message = "Path representing the JAR file name has zero elements";
            throw new CarbonToolException(message);
        }
        String fileName = tempJarFilePathHolder.toString();
        if (!fileName.endsWith(".jar")) {
            String message = "Path jarFile does not point to a JAR file";
            throw new CarbonToolException(message);
        }
        if (BundleGeneratorUtils.isOSGiBundle(jarFile)) {
            logger.log(Level.WARNING, "Path " + jarFile.toString() + " refers to an OSGi bundle");
        }
        if (manifest == null) {
            manifest = new Manifest();
        }
        String exportedPackages = BundleGeneratorUtils.generateExportPackageList(BundleGeneratorUtils.listPackages(jarFile));
        fileName = fileName.replaceAll("-", "_");
        fileName = fileName.substring(0, fileName.length() - 4);
        String symbolicName = extensionPrefix + fileName;
        String pluginName = extensionPrefix + fileName + "_1.0.0" + ".jar";
        Path extensionBundle = Paths.get(targetDirectory.toString(), pluginName);
        logger.log(Level.FINEST, "Setting Manifest attributes");
        Attributes attributes = manifest.getMainAttributes();
        attributes.putValue("Manifest-Version", "1.0");
        attributes.putValue("Bundle-ManifestVersion", "2");
        attributes.putValue("Bundle-Name", fileName);
        attributes.putValue("Bundle-SymbolicName", symbolicName);
        attributes.putValue("Bundle-Version", "1.0.0");
        attributes.putValue("Export-Package", exportedPackages);
        attributes.putValue("Bundle-ClassPath", ".," + tempJarFilePathHolder.toString());
        attributes.putValue("DynamicImport-Package", "*");
        if (!Files.exists(extensionBundle, new LinkOption[0])) {
            logger.log(Level.FINE, "Creating the OSGi bundle for JAR file " + jarFile.toString());
            logger.log(Level.FINE, "Creating an OSGi bundle for JAR file " + tempJarFilePathHolder.toString() + ", at target directory " + extensionBundle.toString() + ".");
            BundleGeneratorUtils.createBundle(jarFile, extensionBundle, manifest);
            logger.log(Level.FINE, "Created an OSGi bundle for JAR file " + tempJarFilePathHolder.toString() + ", at target directory " + extensionBundle.toString());
            logger.log(Level.INFO, "Created the OSGi bundle " + pluginName + " for JAR file " + jarFile.toString());
            return;
        }
        logger.log(Level.INFO, "OSGi bundle " + pluginName + " already exists in the target directory");
    }

    private static String generateExportPackageList(List<String> packageNames) {
        StringBuilder exportedPackages = new StringBuilder();
        if (packageNames != null) {
            IntStream.range(0, packageNames.size()).forEach(packageCount -> {
                exportedPackages.append((String)packageNames.get(packageCount));
                if (packageCount != packageNames.size() - 1) {
                    exportedPackages.append(",");
                }
            });
        }
        return exportedPackages.toString();
    }

    public static boolean isOSGiBundle(Path jaFilePath) throws IOException, CarbonToolException {
        boolean hasVersion;
        boolean hasSymbolicName;
        try (FileSystem zipFileSystem = BundleGeneratorUtils.createZipFileSystem(jaFilePath, false);){
            Path manifestPath = zipFileSystem.getPath("META-INF", "MANIFEST.MF");
            Manifest manifest = new Manifest(Files.newInputStream(manifestPath, new OpenOption[0]));
            Attributes attributes = manifest.getMainAttributes();
            hasSymbolicName = attributes.getValue("Bundle-SymbolicName") != null;
            hasVersion = attributes.getValue("Bundle-Version") != null;
        }
        return hasSymbolicName && hasVersion;
    }

    public static void createBundle(Path jarFile, Path bundlePath, Manifest manifest) throws IOException, CarbonToolException {
        block39: {
            Path tempJarFilePathHolder = jarFile.getFileName();
            if (tempJarFilePathHolder != null) {
                if (manifest != null) {
                    Path tempBundleHolder = Files.createTempDirectory(Paths.get(System.getProperty("java.io.tmpdir"), new String[0]), "temp", new FileAttribute[0]);
                    tempBundleHolder.toFile().deleteOnExit();
                    Path manifestFile = Paths.get(tempBundleHolder.toString(), "MANIFEST.MF");
                    Path p2InfFile = Files.createTempFile(tempBundleHolder, "p2", ".inf", new FileAttribute[0]);
                    p2InfFile.toFile().deleteOnExit();
                    try (OutputStream manifestOutputStream = Files.newOutputStream(manifestFile, new OpenOption[0]);
                         OutputStream p2InfOutputStream = Files.newOutputStream(p2InfFile, new OpenOption[0]);
                         FileSystem zipFileSystem = BundleGeneratorUtils.createZipFileSystem(bundlePath, true);){
                        manifest.write(manifestOutputStream);
                        logger.log(Level.FINE, "Generated the OSGi bundlePath MANIFEST.MF for the JAR file " + jarFile.toString());
                        p2InfOutputStream.write("instructions.configure=markStarted(started:true);".getBytes(Charset.forName("UTF-8")));
                        p2InfOutputStream.flush();
                        logger.log(Level.FINE, "Generated the OSGi bundlePath p2.inf for the JAR file " + jarFile.toString());
                        Path manifestFolderPath = zipFileSystem.getPath("META-INF", new String[0]);
                        if (!Files.exists(manifestFolderPath, new LinkOption[0])) {
                            Files.createDirectories(manifestFolderPath, new FileAttribute[0]);
                        }
                        Path manifestPathInBundle = zipFileSystem.getPath("META-INF", "MANIFEST.MF");
                        Path p2InfPathInBundle = zipFileSystem.getPath("META-INF", "p2.inf");
                        Files.copy(jarFile, zipFileSystem.getPath(tempJarFilePathHolder.toString(), new String[0]), new CopyOption[0]);
                        Files.copy(manifestFile, manifestPathInBundle, new CopyOption[0]);
                        Files.copy(p2InfFile, p2InfPathInBundle, new CopyOption[0]);
                        BundleGeneratorUtils.delete(manifestFile);
                        break block39;
                    }
                }
                throw new CarbonToolException("Manifest cannot refer to null");
            }
            throw new CarbonToolException("Path representing the JAR file name has zero elements");
        }
    }

    public static boolean delete(Path path) throws IOException {
        List<Path> children;
        if (Files.isDirectory(path, new LinkOption[0]) && (children = BundleGeneratorUtils.listFiles(path)).size() > 0) {
            children.forEach(aChild -> {
                try {
                    BundleGeneratorUtils.delete(aChild);
                }
                catch (IOException e) {
                    logger.log(Level.SEVERE, e.getMessage(), e);
                }
            });
        }
        logger.log(Level.FINE, "Deleting " + path + ".");
        return Files.deleteIfExists(path);
    }

    public static List<Path> listFiles(Path directory) throws IOException {
        ArrayList<Path> files = new ArrayList<Path>();
        try (DirectoryStream<Path> directoryStream = Files.newDirectoryStream(directory);){
            directoryStream.forEach(files::add);
        }
        return files;
    }

    public static List<String> listPackages(Path jarFile) throws IOException, CarbonToolException {
        ArrayList<String> exportedPackagesList = new ArrayList<String>();
        List<Path> content = BundleGeneratorUtils.listZipFileContent(jarFile);
        content.forEach(zipChild -> {
            int index;
            String path = zipChild.toString();
            if (!path.endsWith("/") && path.endsWith(".class") && (index = path.lastIndexOf(47)) != -1) {
                path = path.substring(1, index);
                if (!exportedPackagesList.contains(path = path.replaceAll("/", "."))) {
                    exportedPackagesList.add(path);
                }
            }
        });
        logger.log(Level.FINE, "Returning a List<String> of packages from the JAR file " + jarFile.toString());
        return exportedPackagesList;
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public static List<Path> listZipFileContent(Path zipFilePath) throws IOException, CarbonToolException {
        final ArrayList<Path> zipFileContent = new ArrayList<Path>();
        if (!Files.exists(zipFilePath, new LinkOption[0])) {
            String message = "Path represented by the zipFilePath does not exist";
            throw new CarbonToolException(message);
        }
        Path zipFileName = zipFilePath.getFileName();
        if (zipFileName == null) return zipFileContent;
        if (!Files.isDirectory(zipFilePath, new LinkOption[0]) && (zipFileName.toString().endsWith(".zip") || zipFileName.toString().endsWith(".jar"))) {
            try (FileSystem zipFileSystem = BundleGeneratorUtils.createZipFileSystem(zipFilePath, false);){
                Path root = zipFileSystem.getPath("/", new String[0]);
                Files.walkFileTree(root, (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(){

                    @Override
                    public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
                        zipFileContent.add(file);
                        return FileVisitResult.CONTINUE;
                    }

                    @Override
                    public FileVisitResult preVisitDirectory(Path directory, BasicFileAttributes attrs) throws IOException {
                        zipFileContent.add(directory);
                        return FileVisitResult.CONTINUE;
                    }
                });
                return zipFileContent;
            }
        }
        String message = "Path zipFilePath may not exist or may not refer to a .zip or .jar file";
        throw new CarbonToolException(message);
    }

    public static FileSystem createZipFileSystem(Path zipFilePath, boolean create) throws IOException, CarbonToolException {
        Path zipFileName = zipFilePath.getFileName();
        if (zipFileName != null) {
            if (zipFileName.toString().endsWith(".zip") || zipFileName.toString().endsWith(".jar")) {
                HashMap<String, String> bundleJarProperties = new HashMap<String, String>();
                if (create) {
                    bundleJarProperties.put("create", "true");
                } else {
                    bundleJarProperties.put("create", "false");
                }
                bundleJarProperties.put("encoding", "UTF-8");
                URI zipFileIURI = URI.create("jar:" + Paths.get(zipFilePath.toString(), new String[0]).toUri());
                return FileSystems.newFileSystem(zipFileIURI, bundleJarProperties);
            }
            String message = "Path zipFilePath does not refer to a .zip or .jar file";
            throw new CarbonToolException(message);
        }
        String message = "Path representing the zip file name has zero elements";
        throw new CarbonToolException(message);
    }
}

