/*
 * Copyright (C) 2011-2026 Thomas Akehurst
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.github.tomakehurst.wiremock.client;

import static com.github.tomakehurst.wiremock.common.ContentTypes.CONTENT_TYPE;
import static com.github.tomakehurst.wiremock.common.ContentTypes.LOCATION;
import static com.github.tomakehurst.wiremock.http.RequestMethod.*;
import static com.github.tomakehurst.wiremock.matching.RequestPattern.thatMatch;
import static com.github.tomakehurst.wiremock.matching.RequestPatternBuilder.allRequests;

import com.github.tomakehurst.wiremock.admin.model.ListMessageChannelsResult;
import com.github.tomakehurst.wiremock.admin.model.ListMessageStubMappingsResult;
import com.github.tomakehurst.wiremock.admin.model.ListStubMappingsResult;
import com.github.tomakehurst.wiremock.admin.model.ServeEventQuery;
import com.github.tomakehurst.wiremock.admin.model.SingleStubMappingResult;
import com.github.tomakehurst.wiremock.common.FileSource;
import com.github.tomakehurst.wiremock.common.Json;
import com.github.tomakehurst.wiremock.common.SingleRootFileSource;
import com.github.tomakehurst.wiremock.core.Admin;
import com.github.tomakehurst.wiremock.extension.Parameters;
import com.github.tomakehurst.wiremock.global.GlobalSettings;
import com.github.tomakehurst.wiremock.http.DelayDistribution;
import com.github.tomakehurst.wiremock.http.Request;
import com.github.tomakehurst.wiremock.http.RequestMethod;
import com.github.tomakehurst.wiremock.matching.*;
import com.github.tomakehurst.wiremock.message.MessagePattern;
import com.github.tomakehurst.wiremock.message.MessageStubMapping;
import com.github.tomakehurst.wiremock.message.SendMessageActionBuilder;
import com.github.tomakehurst.wiremock.recording.RecordSpec;
import com.github.tomakehurst.wiremock.recording.RecordSpecBuilder;
import com.github.tomakehurst.wiremock.recording.RecordingStatusResult;
import com.github.tomakehurst.wiremock.recording.SnapshotRecordResult;
import com.github.tomakehurst.wiremock.standalone.RemoteMappingsLoader;
import com.github.tomakehurst.wiremock.store.InMemorySettingsStore;
import com.github.tomakehurst.wiremock.store.SettingsStore;
import com.github.tomakehurst.wiremock.stubbing.Scenario;
import com.github.tomakehurst.wiremock.stubbing.ServeEvent;
import com.github.tomakehurst.wiremock.stubbing.StubImport;
import com.github.tomakehurst.wiremock.stubbing.StubImportBuilder;
import com.github.tomakehurst.wiremock.stubbing.StubMapping;
import com.github.tomakehurst.wiremock.verification.FindMessageServeEventsResult;
import com.github.tomakehurst.wiremock.verification.FindNearMissesResult;
import com.github.tomakehurst.wiremock.verification.FindRequestsResult;
import com.github.tomakehurst.wiremock.verification.LoggedRequest;
import com.github.tomakehurst.wiremock.verification.MessageServeEvent;
import com.github.tomakehurst.wiremock.verification.NearMiss;
import com.github.tomakehurst.wiremock.verification.VerificationResult;
import com.github.tomakehurst.wiremock.verification.diff.Diff;
import com.networknt.schema.SpecVersion;
import java.io.File;
import java.time.Duration;
import java.time.LocalDateTime;
import java.time.ZonedDateTime;
import java.util.*;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.wiremock.url.Path;
import org.wiremock.url.PathAndQuery;

@SuppressWarnings("unused")
public class WireMock {

  private static final int DEFAULT_PORT = 8080;
  private static final String DEFAULT_HOST = "localhost";

  private final Admin admin;

  private final SettingsStore settingsStore = new InMemorySettingsStore();

  private static final InheritableThreadLocal<WireMock> defaultInstance =
      new InheritableThreadLocal<>() {
        @Override
        protected WireMock initialValue() {
          return WireMock.create().build();
        }
      };

  public static WireMockBuilder create() {
    return new WireMockBuilder();
  }

  public WireMock(Admin admin) {
    this.admin = admin;
  }

  public WireMock(int port) {
    this(WireMock.create().port(port).buildAdminClient());
  }

  public static StubMapping givenThat(MappingBuilder mappingBuilder) {
    return defaultInstance.get().register(mappingBuilder);
  }

  public static StubMapping stubFor(MappingBuilder mappingBuilder) {
    return givenThat(mappingBuilder);
  }

  public static void editStub(MappingBuilder mappingBuilder) {
    defaultInstance.get().editStubMapping(mappingBuilder);
  }

  public static void removeStub(MappingBuilder mappingBuilder) {
    defaultInstance.get().removeStubMapping(mappingBuilder);
  }

  public static void removeStub(StubMapping stubMapping) {
    defaultInstance.get().removeStubMapping(stubMapping);
  }

  public static void removeStub(UUID id) {
    defaultInstance.get().removeStubMapping(id);
  }

  public static ListStubMappingsResult listAllStubMappings() {
    return defaultInstance.get().allStubMappings();
  }

  public static StubMapping getSingleStubMapping(UUID id) {
    return defaultInstance.get().getStubMapping(id).getItem();
  }

  public static void configureFor(int port) {
    defaultInstance.set(WireMock.create().port(port).build());
  }

  public static void configureFor(String host, int port) {
    defaultInstance.set(WireMock.create().host(host).port(port).build());
  }

  public static void configureFor(String host, int port, String urlPathPrefix) {
    defaultInstance.set(
        WireMock.create().host(host).port(port).urlPathPrefix(urlPathPrefix).build());
  }

  public static void configureFor(String scheme, String host, int port, String urlPathPrefix) {
    defaultInstance.set(
        WireMock.create()
            .scheme(scheme)
            .host(host)
            .port(port)
            .urlPathPrefix(urlPathPrefix)
            .build());
  }

  public static void configureFor(String scheme, String host, int port) {
    defaultInstance.set(WireMock.create().scheme(scheme).host(host).port(port).build());
  }

  public static void configureFor(
      String scheme, String host, int port, String proxyHost, int proxyPort) {
    defaultInstance.set(
        WireMock.create()
            .scheme(scheme)
            .host(host)
            .port(port)
            .urlPathPrefix("")
            .hostHeader(null)
            .proxyHost(proxyHost)
            .proxyPort(proxyPort)
            .build());
  }

  public static void configureFor(WireMock client) {
    defaultInstance.set(client);
  }

  public static void configure() {
    defaultInstance.set(WireMock.create().build());
  }

  public static StringValuePattern equalTo(String value) {
    return new EqualToPattern(value);
  }

  public static StringValuePattern equalToNumber(Number value) {
    return new EqualToNumberPattern(value);
  }

  public static StringValuePattern greaterThanNumber(Number value) {
    return new GreaterThanNumberPattern(value);
  }

  public static StringValuePattern greaterThanEqualNumber(Number value) {
    return new GreaterThanEqualNumberPattern(value);
  }

  public static StringValuePattern lessThanNumber(Number value) {
    return new LessThanNumberPattern(value);
  }

  public static StringValuePattern lessThanEqualNumber(Number value) {
    return new LessThanEqualNumberPattern(value);
  }

  public static BinaryEqualToPattern binaryEqualTo(byte[] content) {
    return new BinaryEqualToPattern(content);
  }

  public static BinaryEqualToPattern binaryEqualTo(String content) {
    return new BinaryEqualToPattern(content);
  }

  public static StringValuePattern equalToIgnoreCase(String value) {
    return new EqualToPattern(value, true);
  }

  public static StringValuePattern equalToJson(String value) {
    return new EqualToJsonPattern(value, null, null);
  }

  public static StringValuePattern equalToJson(
      String value, boolean ignoreArrayOrder, boolean ignoreExtraElements) {
    return new EqualToJsonPattern(value, ignoreArrayOrder, ignoreExtraElements);
  }

  public static StringValuePattern matchingJsonPath(String value) {
    return new MatchesJsonPathPattern(value);
  }

  public static StringValuePattern matchingJsonPath(String value, StringValuePattern valuePattern) {
    return new MatchesJsonPathPattern(value, valuePattern);
  }

  public static StringValuePattern matchingJsonSchema(String schema) {
    return new MatchesJsonSchemaPattern(schema);
  }

  public static StringValuePattern matchingJsonSchema(
      String schema, JsonSchemaVersion jsonSchemaVersion) {
    return new MatchesJsonSchemaPattern(schema, jsonSchemaVersion);
  }

  public static EqualToXmlPattern equalToXml(String value) {
    return new EqualToXmlPattern(value);
  }

  public static EqualToXmlPattern equalToXml(
      String value, EqualToXmlPattern.NamespaceAwareness namespaceAwareness) {
    return new EqualToXmlPattern(value, null, null, null, null, null, namespaceAwareness);
  }

  public static EqualToXmlPattern equalToXml(String value, boolean enablePlaceholders) {
    return equalToXml(value, enablePlaceholders, false);
  }

  public static EqualToXmlPattern equalToXml(
      String value, boolean enablePlaceholders, boolean ignoreOrderOfSameNode) {
    return new EqualToXmlPattern(value, enablePlaceholders, ignoreOrderOfSameNode);
  }

  public static EqualToXmlPattern equalToXml(
      String value,
      boolean enablePlaceholders,
      String placeholderOpeningDelimiterRegex,
      String placeholderClosingDelimiterRegex) {
    return equalToXml(
        value,
        enablePlaceholders,
        placeholderOpeningDelimiterRegex,
        placeholderClosingDelimiterRegex,
        false);
  }

  public static EqualToXmlPattern equalToXml(
      String value,
      boolean enablePlaceholders,
      String placeholderOpeningDelimiterRegex,
      String placeholderClosingDelimiterRegex,
      boolean ignoreOrderOfSameNode) {
    return equalToXml(
        value,
        enablePlaceholders,
        placeholderOpeningDelimiterRegex,
        placeholderClosingDelimiterRegex,
        ignoreOrderOfSameNode,
        null);
  }

  public static EqualToXmlPattern equalToXml(
      String value,
      boolean enablePlaceholders,
      String placeholderOpeningDelimiterRegex,
      String placeholderClosingDelimiterRegex,
      boolean ignoreOrderOfSameNode,
      EqualToXmlPattern.NamespaceAwareness namespaceAwareness) {
    return new EqualToXmlPattern(
        value,
        enablePlaceholders,
        placeholderOpeningDelimiterRegex,
        placeholderClosingDelimiterRegex,
        null,
        ignoreOrderOfSameNode,
        namespaceAwareness);
  }

  public static MatchesXPathPattern matchingXPath(String value) {
    return new MatchesXPathPattern(value, Collections.emptyMap());
  }

  public static StringValuePattern matchingXPath(String value, Map<String, String> namespaces) {
    return new MatchesXPathPattern(value, namespaces);
  }

  public static StringValuePattern matchingXPath(String value, StringValuePattern valuePattern) {
    return new MatchesXPathPattern(value, valuePattern);
  }

  // Use this with the date/time matchers to avoid an explicit cast
  public static MatchesXPathPattern matchesXPathWithSubMatcher(
      String value, StringValuePattern valuePattern) {
    return new MatchesXPathPattern(value, valuePattern);
  }

  public static StringValuePattern containing(String value) {
    return new ContainsPattern(value);
  }

  public static StringValuePattern notContaining(String value) {
    return new NegativeContainsPattern(value);
  }

  public static StringValuePattern not(StringValuePattern unexpectedPattern) {
    return new NotPattern(unexpectedPattern);
  }

  public static StringValuePattern matching(String regex) {
    return new RegexPattern(regex);
  }

  public static StringValuePattern notMatching(String regex) {
    return new NegativeRegexPattern(regex);
  }

  public static BeforeDateTimePattern before(String dateTimeSpec) {
    return new BeforeDateTimePattern(dateTimeSpec);
  }

  public static BeforeDateTimePattern before(ZonedDateTime dateTime) {
    return new BeforeDateTimePattern(dateTime);
  }

  public static BeforeDateTimePattern before(LocalDateTime dateTime) {
    return new BeforeDateTimePattern(dateTime);
  }

  public static BeforeDateTimePattern beforeNow() {
    return new BeforeDateTimePattern("now");
  }

  public static EqualToDateTimePattern equalToDateTime(String dateTimeSpec) {
    return new EqualToDateTimePattern(dateTimeSpec);
  }

  public static EqualToDateTimePattern equalToDateTime(ZonedDateTime dateTime) {
    return new EqualToDateTimePattern(dateTime);
  }

  public static EqualToDateTimePattern equalToDateTime(LocalDateTime dateTime) {
    return new EqualToDateTimePattern(dateTime);
  }

  public static EqualToDateTimePattern isNow() {
    return new EqualToDateTimePattern("now");
  }

  public static AfterDateTimePattern after(String dateTimeSpec) {
    return new AfterDateTimePattern(dateTimeSpec);
  }

  public static AfterDateTimePattern after(ZonedDateTime dateTime) {
    return new AfterDateTimePattern(dateTime);
  }

  public static AfterDateTimePattern after(LocalDateTime dateTime) {
    return new AfterDateTimePattern(dateTime);
  }

  public static AfterDateTimePattern afterNow() {
    return new AfterDateTimePattern("now");
  }

  public static StringValuePattern absent() {
    return AbsentPattern.ABSENT;
  }

  public static StringValuePattern and(StringValuePattern... matchers) {
    return new LogicalAnd(matchers);
  }

  public static StringValuePattern or(StringValuePattern... matchers) {
    return new LogicalOr(matchers);
  }

  public void saveMappings() {
    admin.saveMappings();
  }

  public static void saveAllMappings() {
    defaultInstance.get().saveMappings();
  }

  public void removeMappings() {
    admin.resetMappings();
  }

  public static void removeAllMappings() {
    defaultInstance.get().removeMappings();
  }

  public void resetMappings() {
    admin.resetAll();
  }

  public static void reset() {
    defaultInstance.get().resetMappings();
  }

  public static void resetAllRequests() {
    defaultInstance.get().resetRequests();
  }

  public void resetRequests() {
    admin.resetRequests();
  }

  public void resetScenarios() {
    admin.resetScenarios();
  }

  public static void resetScenario(String name) {
    defaultInstance.get().resetScenarioState(name);
  }

  public void resetScenarioState(String name) {
    admin.resetScenario(name);
  }

  public static void setScenarioState(String name, String state) {
    defaultInstance.get().setSingleScenarioState(name, state);
  }

  public void setSingleScenarioState(String name, String state) {
    admin.setScenarioState(name, state);
  }

  public static List<Scenario> getAllScenarios() {
    return defaultInstance.get().getScenarios();
  }

  public List<Scenario> getScenarios() {
    return admin.getAllScenarios().getScenarios();
  }

  public static void resetAllScenarios() {
    defaultInstance.get().resetScenarios();
  }

  public void resetToDefaultMappings() {
    admin.resetToDefaultMappings();
  }

  public static void resetToDefault() {
    defaultInstance.get().resetToDefaultMappings();
  }

  public StubMapping register(MappingBuilder mappingBuilder) {
    StubMapping mapping = mappingBuilder.build();
    register(mapping);
    return mapping;
  }

  public void register(StubMapping mapping) {
    admin.addStubMapping(mapping);
  }

  public void editStubMapping(MappingBuilder mappingBuilder) {
    admin.editStubMapping(mappingBuilder.build());
  }

  public void removeStubMapping(MappingBuilder mappingBuilder) {
    admin.removeStubMapping(mappingBuilder.build());
  }

  public void removeStubMapping(StubMapping stubMapping) {
    admin.removeStubMapping(stubMapping);
  }

  public void removeStubMapping(UUID id) {
    admin.removeStubMapping(id);
  }

  public ListStubMappingsResult allStubMappings() {
    return admin.listAllStubMappings();
  }

  public SingleStubMappingResult getStubMapping(UUID id) {
    return admin.getStubMapping(id);
  }

  public static UrlPattern urlEqualTo(PathAndQuery testUrl) {
    return urlEqualTo(testUrl.toString());
  }

  public static UrlPattern urlEqualTo(String testUrl) {
    return new UrlPattern(equalTo(testUrl), false);
  }

  public static UrlPattern urlMatching(String urlRegex) {
    return new UrlPattern(matching(urlRegex), true);
  }

  public static UrlPathPattern urlPathEqualTo(Path testUrl) {
    return urlPathEqualTo(testUrl.toString());
  }

  public static UrlPathPattern urlPathEqualTo(String testUrl) {
    return new UrlPathPattern(equalTo(testUrl), false);
  }

  public static UrlPathPattern urlPathMatching(String urlRegex) {
    return new UrlPathPattern(matching(urlRegex), true);
  }

  public static UrlPathPattern urlPathTemplate(String pathTemplate) {
    return new UrlPathTemplatePattern(pathTemplate);
  }

  public static MultiValuePattern havingExactly(final StringValuePattern... valuePatterns) {
    if (valuePatterns.length == 0) {
      return noValues();
    }
    return new ExactMatchMultiValuePattern(Stream.of(valuePatterns).collect(Collectors.toList()));
  }

  public static MultiValuePattern havingExactly(String... values) {
    return havingExactly(Stream.of(values).map(EqualToPattern::new).toArray(EqualToPattern[]::new));
  }

  public static MultiValuePattern including(final StringValuePattern... valuePatterns) {
    if (valuePatterns.length == 0) {
      return noValues();
    }
    return new IncludesMatchMultiValuePattern(
        Stream.of(valuePatterns).collect(Collectors.toList()));
  }

  public static MultiValuePattern including(String... values) {
    return including(Stream.of(values).map(EqualToPattern::new).toArray(EqualToPattern[]::new));
  }

  public static MultiValuePattern noValues() {
    return MultiValuePattern.of(absent());
  }

  public static UrlPattern anyUrl() {
    return UrlPattern.ANY;
  }

  public static CountMatchingStrategy lessThan(int expected) {
    return new CountMatchingStrategy(CountMatchingStrategy.LESS_THAN, expected);
  }

  public static CountMatchingStrategy lessThanOrExactly(int expected) {
    return new CountMatchingStrategy(CountMatchingStrategy.LESS_THAN_OR_EQUAL, expected);
  }

  public static CountMatchingStrategy exactly(int expected) {
    return new CountMatchingStrategy(CountMatchingStrategy.EQUAL_TO, expected);
  }

  public static CountMatchingStrategy moreThanOrExactly(int expected) {
    return new CountMatchingStrategy(CountMatchingStrategy.GREATER_THAN_OR_EQUAL, expected);
  }

  public static CountMatchingStrategy moreThan(int expected) {
    return new CountMatchingStrategy(CountMatchingStrategy.GREATER_THAN, expected);
  }

  public static MappingBuilder get(UrlPattern urlPattern) {
    return new BasicMappingBuilder(GET, urlPattern);
  }

  public static MappingBuilder post(UrlPattern urlPattern) {
    return new BasicMappingBuilder(RequestMethod.POST, urlPattern);
  }

  public static MappingBuilder put(UrlPattern urlPattern) {
    return new BasicMappingBuilder(RequestMethod.PUT, urlPattern);
  }

  public static MappingBuilder delete(UrlPattern urlPattern) {
    return new BasicMappingBuilder(RequestMethod.DELETE, urlPattern);
  }

  public static MappingBuilder patch(UrlPattern urlPattern) {
    return new BasicMappingBuilder(RequestMethod.PATCH, urlPattern);
  }

  public static MappingBuilder head(UrlPattern urlPattern) {
    return new BasicMappingBuilder(RequestMethod.HEAD, urlPattern);
  }

  public static MappingBuilder options(UrlPattern urlPattern) {
    return new BasicMappingBuilder(RequestMethod.OPTIONS, urlPattern);
  }

  public static MappingBuilder trace(UrlPattern urlPattern) {
    return new BasicMappingBuilder(RequestMethod.TRACE, urlPattern);
  }

  public static MappingBuilder any(UrlPattern urlPattern) {
    return new BasicMappingBuilder(RequestMethod.ANY, urlPattern);
  }

  public static MappingBuilder query(UrlPattern urlPattern) {
    return new BasicMappingBuilder(RequestMethod.QUERY, urlPattern);
  }

  /**
   * A mapping builder that can be used for both GET and HEAD http method. Returns a response body
   * in case for GET and not in case of HEAD method. In case of tie the request is treated as a GET
   * request
   *
   * @param urlPattern for the specified method
   * @return a mapping builder for {@link RequestMethod#GET_OR_HEAD} http method
   */
  public static MappingBuilder getOrHead(UrlPattern urlPattern) {
    return request(isOneOf(GET, HEAD), urlPattern);
  }

  public static MappingBuilder request(String method, UrlPattern urlPattern) {
    return new BasicMappingBuilder(RequestMethod.fromString(method), urlPattern);
  }

  public static MappingBuilder request(RequestMethod method, UrlPattern urlPattern) {
    return new BasicMappingBuilder(method, urlPattern);
  }

  public static MappingBuilder requestMatching(String customRequestMatcherName) {
    return new BasicMappingBuilder(customRequestMatcherName, Parameters.empty());
  }

  public static MappingBuilder requestMatching(
      String customRequestMatcherName, Parameters parameters) {
    return new BasicMappingBuilder(customRequestMatcherName, parameters);
  }

  public static MappingBuilder requestMatching(ValueMatcher<Request> requestMatcher) {
    return new BasicMappingBuilder(requestMatcher);
  }

  public static ResponseDefinitionBuilder aResponse() {
    return new ResponseDefinitionBuilder();
  }

  public static ResponseDefinitionBuilder ok() {
    return aResponse().withStatus(200);
  }

  public static ResponseDefinitionBuilder ok(String body) {
    return aResponse().withStatus(200).withBody(body);
  }

  public static ResponseDefinitionBuilder okForContentType(String contentType, String body) {
    return aResponse().withStatus(200).withHeader(CONTENT_TYPE, contentType).withBody(body);
  }

  public static ResponseDefinitionBuilder okJson(String body) {
    return okForContentType("application/json", body);
  }

  public static ResponseDefinitionBuilder okXml(String body) {
    return okForContentType("application/xml", body);
  }

  public static ResponseDefinitionBuilder okTextXml(String body) {
    return okForContentType("text/xml", body);
  }

  public static ResponseDefinitionBuilder jsonResponse(String body, int status) {
    return aResponse()
        .withStatus(status)
        .withHeader(CONTENT_TYPE, "application/json")
        .withBody(body);
  }

  public static ResponseDefinitionBuilder jsonResponse(Object body, int status) {
    return jsonResponse(Json.write(body), status);
  }

  public static MappingBuilder proxyAllTo(String url) {
    return any(anyUrl()).willReturn(aResponse().proxiedFrom(url));
  }

  public static MappingBuilder get(String url) {
    return get(urlEqualTo(url));
  }

  public static MappingBuilder post(String url) {
    return post(urlEqualTo(url));
  }

  public static MappingBuilder put(String url) {
    return put(urlEqualTo(url));
  }

  public static MappingBuilder delete(String url) {
    return delete(urlEqualTo(url));
  }

  public static MappingBuilder patch(String url) {
    return patch(urlEqualTo(url));
  }

  public static MappingBuilder query(String url) {
    return query(urlEqualTo(url));
  }

  public static ResponseDefinitionBuilder created() {
    return aResponse().withStatus(201);
  }

  public static ResponseDefinitionBuilder noContent() {
    return aResponse().withStatus(204);
  }

  public static ResponseDefinitionBuilder permanentRedirect(String location) {
    return aResponse().withStatus(301).withHeader(LOCATION, location);
  }

  public static ResponseDefinitionBuilder temporaryRedirect(String location) {
    return aResponse().withStatus(302).withHeader(LOCATION, location);
  }

  public static ResponseDefinitionBuilder seeOther(String location) {
    return aResponse().withStatus(303).withHeader(LOCATION, location);
  }

  public static ResponseDefinitionBuilder badRequest() {
    return aResponse().withStatus(400);
  }

  public static ResponseDefinitionBuilder badRequestEntity() {
    return aResponse().withStatus(422);
  }

  public static ResponseDefinitionBuilder unauthorized() {
    return aResponse().withStatus(401);
  }

  public static ResponseDefinitionBuilder forbidden() {
    return aResponse().withStatus(403);
  }

  public static ResponseDefinitionBuilder notFound() {
    return aResponse().withStatus(404);
  }

  public static ResponseDefinitionBuilder serverError() {
    return aResponse().withStatus(500);
  }

  public static ResponseDefinitionBuilder serviceUnavailable() {
    return aResponse().withStatus(503);
  }

  public static ResponseDefinitionBuilder status(int status) {
    return aResponse().withStatus(status);
  }

  public void verifyThat(RequestPatternBuilder requestPatternBuilder) {
    verifyThat(moreThanOrExactly(1), requestPatternBuilder);
  }

  public void verifyThat(int expectedCount, RequestPatternBuilder requestPatternBuilder) {
    verifyThat(exactly(expectedCount), requestPatternBuilder);
  }

  public void verifyThat(
      CountMatchingStrategy expectedCount, RequestPatternBuilder requestPatternBuilder) {
    final RequestPattern requestPattern = requestPatternBuilder.build();

    int actualCount;
    if (requestPattern.hasInlineCustomMatcher()) {
      List<LoggedRequest> requests =
          admin.findRequestsMatching(RequestPattern.ANYTHING).getRequests();
      actualCount = (int) requests.stream().filter(thatMatch(requestPattern)).count();
    } else {
      VerificationResult result = admin.countRequestsMatching(requestPattern);
      result.assertRequestJournalEnabled();
      actualCount = result.getCount();
    }

    if (!expectedCount.match(actualCount)) {
      throw actualCount == 0
          ? verificationExceptionForNearMisses(requestPatternBuilder, requestPattern)
          : new VerificationException(requestPattern, expectedCount, actualCount);
    }
  }

  private VerificationException verificationExceptionForNearMisses(
      RequestPatternBuilder requestPatternBuilder, RequestPattern requestPattern) {
    List<NearMiss> nearMisses = findAllNearMissesFor(requestPatternBuilder);
    if (!nearMisses.isEmpty()) {
      Diff diff = new Diff(requestPattern, nearMisses.get(0).getRequest());
      return VerificationException.forUnmatchedRequestPattern(diff);
    }

    return new VerificationException(requestPattern, find(allRequests()));
  }

  public static void verify(RequestPatternBuilder requestPatternBuilder) {
    defaultInstance.get().verifyThat(requestPatternBuilder);
  }

  public static void verify(int count, RequestPatternBuilder requestPatternBuilder) {
    defaultInstance.get().verifyThat(count, requestPatternBuilder);
  }

  public static void verify(
      CountMatchingStrategy countMatchingStrategy, RequestPatternBuilder requestPatternBuilder) {
    defaultInstance.get().verifyThat(countMatchingStrategy, requestPatternBuilder);
  }

  public List<LoggedRequest> find(RequestPatternBuilder requestPatternBuilder) {
    FindRequestsResult result = admin.findRequestsMatching(requestPatternBuilder.build());
    result.assertRequestJournalEnabled();
    return result.getRequests();
  }

  public static List<LoggedRequest> findAll(RequestPatternBuilder requestPatternBuilder) {
    return defaultInstance.get().find(requestPatternBuilder);
  }

  public static List<ServeEvent> getAllServeEvents() {
    return defaultInstance.get().getServeEvents();
  }

  public List<ServeEvent> getServeEvents() {
    return admin.getServeEvents().getRequests();
  }

  public static List<ServeEvent> getAllServeEvents(ServeEventQuery query) {
    return defaultInstance.get().getServeEvents(query);
  }

  public List<ServeEvent> getServeEvents(ServeEventQuery query) {
    return admin.getServeEvents(query).getRequests();
  }

  public static void removeServeEvent(UUID eventId) {
    defaultInstance.get().removeEvent(eventId);
  }

  public void removeEvent(UUID eventId) {
    admin.removeServeEvent(eventId);
  }

  public List<ServeEvent> removeEvents(RequestPatternBuilder requestPatternBuilder) {
    return admin.removeServeEventsMatching(requestPatternBuilder.build()).getServeEvents();
  }

  public static List<ServeEvent> removeServeEvents(RequestPatternBuilder requestPatternBuilder) {
    return defaultInstance.get().removeEvents(requestPatternBuilder);
  }

  public static List<ServeEvent> removeEventsByStubMetadata(StringValuePattern pattern) {
    return defaultInstance.get().removeEventsByMetadata(pattern);
  }

  public List<ServeEvent> removeEventsByMetadata(StringValuePattern pattern) {
    return admin.removeServeEventsForStubsMatchingMetadata(pattern).getServeEvents();
  }

  public static RequestPatternBuilder getRequestedFor(UrlPattern urlPattern) {
    return new RequestPatternBuilder(GET, urlPattern);
  }

  public static RequestPatternBuilder postRequestedFor(UrlPattern urlPattern) {
    return new RequestPatternBuilder(RequestMethod.POST, urlPattern);
  }

  public static RequestPatternBuilder putRequestedFor(UrlPattern urlPattern) {
    return new RequestPatternBuilder(RequestMethod.PUT, urlPattern);
  }

  public static RequestPatternBuilder deleteRequestedFor(UrlPattern urlPattern) {
    return new RequestPatternBuilder(RequestMethod.DELETE, urlPattern);
  }

  public static RequestPatternBuilder patchRequestedFor(UrlPattern urlPattern) {
    return new RequestPatternBuilder(RequestMethod.PATCH, urlPattern);
  }

  public static RequestPatternBuilder headRequestedFor(UrlPattern urlPattern) {
    return new RequestPatternBuilder(RequestMethod.HEAD, urlPattern);
  }

  public static RequestPatternBuilder optionsRequestedFor(UrlPattern urlPattern) {
    return new RequestPatternBuilder(RequestMethod.OPTIONS, urlPattern);
  }

  public static RequestPatternBuilder traceRequestedFor(UrlPattern urlPattern) {
    return new RequestPatternBuilder(RequestMethod.TRACE, urlPattern);
  }

  public static RequestPatternBuilder anyRequestedFor(UrlPattern urlPattern) {
    return new RequestPatternBuilder(RequestMethod.ANY, urlPattern);
  }

  public static RequestPatternBuilder queryRequestedFor(UrlPattern urlPattern) {
    return new RequestPatternBuilder(RequestMethod.QUERY, urlPattern);
  }

  public static RequestPatternBuilder requestedFor(String method, UrlPattern urlPattern) {
    return new RequestPatternBuilder(RequestMethod.fromString(method), urlPattern);
  }

  public static RequestPatternBuilder requestMadeFor(
      String customMatcherName, Parameters parameters) {
    return RequestPatternBuilder.forCustomMatcher(customMatcherName, parameters);
  }

  public static RequestPatternBuilder requestMadeFor(ValueMatcher<Request> requestMatcher) {
    return RequestPatternBuilder.forCustomMatcher(requestMatcher);
  }

  public static void setGlobalFixedDelay(int milliseconds) {
    defaultInstance.get().setGlobalFixedDelayVariable(milliseconds);
  }

  public void setGlobalFixedDelayVariable(int milliseconds) {
    GlobalSettings settings = settingsStore.get().copy().fixedDelay(milliseconds).build();
    updateGlobalSettings(settings);
  }

  public static void setGlobalRandomDelay(DelayDistribution distribution) {
    defaultInstance.get().setGlobalRandomDelayVariable(distribution);
  }

  public void setGlobalRandomDelayVariable(DelayDistribution distribution) {
    GlobalSettings settings = settingsStore.get().copy().delayDistribution(distribution).build();
    updateGlobalSettings(settings);
  }

  public static void updateSettings(GlobalSettings settings) {
    defaultInstance.get().updateGlobalSettings(settings);
  }

  public void updateGlobalSettings(GlobalSettings settings) {
    settingsStore.set(settings);
    admin.updateGlobalSettings(settings);
  }

  public void shutdown() {
    admin.shutdownServer();
  }

  public static void shutdownServer() {
    defaultInstance.get().shutdown();
  }

  public static List<NearMiss> findNearMissesForAllUnmatched() {
    return defaultInstance.get().findNearMissesForAllUnmatchedRequests();
  }

  public List<NearMiss> findNearMissesForAllUnmatchedRequests() {
    FindNearMissesResult nearMissesResult = admin.findNearMissesForUnmatchedRequests();
    return nearMissesResult.getNearMisses();
  }

  public static List<LoggedRequest> findUnmatchedRequests() {
    return defaultInstance.get().findAllUnmatchedRequests();
  }

  public List<LoggedRequest> findAllUnmatchedRequests() {
    FindRequestsResult unmatchedResult = admin.findUnmatchedRequests();
    return unmatchedResult.getRequests();
  }

  public static List<NearMiss> findNearMissesFor(LoggedRequest loggedRequest) {
    return defaultInstance.get().findTopNearMissesFor(loggedRequest);
  }

  public List<NearMiss> findTopNearMissesFor(LoggedRequest loggedRequest) {
    FindNearMissesResult nearMissesResult = admin.findTopNearMissesFor(loggedRequest);
    return nearMissesResult.getNearMisses();
  }

  public static List<NearMiss> findNearMissesFor(RequestPatternBuilder requestPatternBuilder) {
    return defaultInstance.get().findAllNearMissesFor(requestPatternBuilder);
  }

  public List<NearMiss> findAllNearMissesFor(RequestPatternBuilder requestPatternBuilder) {
    FindNearMissesResult nearMissesResult =
        admin.findTopNearMissesFor(requestPatternBuilder.build());
    return nearMissesResult.getNearMisses();
  }

  public void loadMappingsFrom(String rootDir) {
    loadMappingsFrom(new File(rootDir));
  }

  public void loadMappingsFrom(File rootDir) {
    FileSource mappingsSource = new SingleRootFileSource(rootDir);
    new RemoteMappingsLoader(mappingsSource, this).load();
  }

  public static List<StubMapping> snapshotRecord() {
    return defaultInstance.get().takeSnapshotRecording();
  }

  public static List<StubMapping> snapshotRecord(RecordSpecBuilder spec) {
    return defaultInstance.get().takeSnapshotRecording(spec);
  }

  public List<StubMapping> takeSnapshotRecording() {
    return admin.snapshotRecord().getStubMappings();
  }

  public List<StubMapping> takeSnapshotRecording(RecordSpecBuilder spec) {
    return admin.snapshotRecord(spec.build()).getStubMappings();
  }

  public static MultipartValuePatternBuilder aMultipart() {
    return new MultipartValuePatternBuilder();
  }

  public static MultipartValuePatternBuilder aMultipart(String name) {
    return new MultipartValuePatternBuilder(name);
  }

  public static void startRecording(String targetBaseUrl) {
    defaultInstance.get().startStubRecording(targetBaseUrl);
  }

  public static void startRecording() {
    defaultInstance.get().startStubRecording();
  }

  public static void startRecording(RecordSpecBuilder spec) {
    defaultInstance.get().startStubRecording(spec);
  }

  public void startStubRecording(String targetBaseUrl) {
    admin.startRecording(targetBaseUrl);
  }

  public void startStubRecording() {
    admin.startRecording(RecordSpec.DEFAULTS);
  }

  public void startStubRecording(RecordSpecBuilder spec) {
    admin.startRecording(spec.build());
  }

  public static SnapshotRecordResult stopRecording() {
    return defaultInstance.get().stopStubRecording();
  }

  public SnapshotRecordResult stopStubRecording() {
    return admin.stopRecording();
  }

  public static RecordingStatusResult getRecordingStatus() {
    return defaultInstance.get().getStubRecordingStatus();
  }

  public RecordingStatusResult getStubRecordingStatus() {
    return admin.getRecordingStatus();
  }

  public static RecordSpecBuilder recordSpec() {
    return new RecordSpecBuilder();
  }

  public List<StubMapping> findAllStubsByMetadata(StringValuePattern pattern) {
    return admin.findAllStubsByMetadata(pattern).getMappings();
  }

  public static List<StubMapping> findStubsByMetadata(StringValuePattern pattern) {
    return defaultInstance.get().findAllStubsByMetadata(pattern);
  }

  public void removeStubsByMetadataPattern(StringValuePattern pattern) {
    admin.removeStubsByMetadata(pattern);
  }

  public static void removeStubsByMetadata(StringValuePattern pattern) {
    defaultInstance.get().removeStubsByMetadataPattern(pattern);
  }

  public void importStubMappings(StubImport stubImport) {
    admin.importStubs(stubImport);
  }

  public void importStubMappings(StubImportBuilder stubImport) {
    importStubMappings(stubImport.build());
  }

  public static void importStubs(StubImportBuilder stubImport) {
    importStubs(stubImport.build());
  }

  public static void importStubs(StubImport stubImport) {
    defaultInstance.get().importStubMappings(stubImport);
  }

  public void removeStubMappings(List<StubMapping> stubs) {
    admin.removeStubMappings(stubs);
  }

  public static void removeStubs(List<StubMapping> stubs) {
    defaultInstance.get().removeStubMappings(stubs);
  }

  public GlobalSettings getGlobalSettings() {
    return admin.getGlobalSettings().getSettings();
  }

  public static GlobalSettings getSettings() {
    return defaultInstance.get().getGlobalSettings();
  }

  // Message stub mapping DSL methods

  public static MessageStubMapping.Builder message() {
    return MessageStubMapping.builder();
  }

  public static SendMessageActionBuilder sendMessage() {
    return new SendMessageActionBuilder();
  }

  public static SendMessageActionBuilder sendMessage(String message) {
    return new SendMessageActionBuilder().withBody(message);
  }

  public static MessageStubMapping messageStubFor(MessageStubMappingBuilder builder) {
    return defaultInstance.get().registerMessageStub(builder);
  }

  public MessageStubMapping registerMessageStub(MessageStubMappingBuilder builder) {
    MessageStubMapping mapping = builder.build();
    admin.addMessageStubMapping(mapping);
    return mapping;
  }

  public static MessageStubMapping messageStubFor(MessageStubMapping messageStubMapping) {
    return defaultInstance.get().registerMessageStub(messageStubMapping);
  }

  public MessageStubMapping registerMessageStub(MessageStubMapping messageStubMapping) {
    admin.addMessageStubMapping(messageStubMapping);
    return messageStubMapping;
  }

  public static void removeMessageStub(UUID id) {
    defaultInstance.get().deleteMessageStub(id);
  }

  public void deleteMessageStub(UUID id) {
    admin.removeMessageStubMapping(id);
  }

  public static void resetMessageStubs() {
    defaultInstance.get().resetAllMessageStubs();
  }

  public void resetAllMessageStubs() {
    admin.resetMessageStubMappings();
  }

  public static List<MessageStubMapping> findMessageStubsByMetadata(StringValuePattern pattern) {
    return defaultInstance.get().findAllMessageStubsByMetadata(pattern);
  }

  public List<MessageStubMapping> findAllMessageStubsByMetadata(StringValuePattern pattern) {
    return admin.findAllMessageStubsByMetadata(pattern).getMessageMappings();
  }

  public static void removeMessageStubsByMetadata(StringValuePattern pattern) {
    defaultInstance.get().removeAllMessageStubsByMetadata(pattern);
  }

  public void removeAllMessageStubsByMetadata(StringValuePattern pattern) {
    admin.removeMessageStubsByMetadata(pattern);
  }

  public static ListMessageStubMappingsResult listAllMessageStubMappings() {
    return defaultInstance.get().allMessageStubMappings();
  }

  public ListMessageStubMappingsResult allMessageStubMappings() {
    return admin.listAllMessageStubMappings();
  }

  public static ListMessageChannelsResult listAllMessageChannels() {
    return defaultInstance.get().allMessageChannels();
  }

  public ListMessageChannelsResult allMessageChannels() {
    return admin.listAllMessageChannels();
  }

  // Message journal verification methods

  public static List<MessageServeEvent> getAllMessageServeEvents() {
    return defaultInstance.get().getMessageServeEvents();
  }

  public List<MessageServeEvent> getMessageServeEvents() {
    return admin.getMessageServeEvents().getMessageServeEvents();
  }

  public static MessageServeEvent getMessageServeEvent(UUID id) {
    return defaultInstance.get().getMessageServeEventById(id);
  }

  public MessageServeEvent getMessageServeEventById(UUID id) {
    return admin.getMessageServeEvent(id).getItem();
  }

  public static void removeMessageServeEvent(UUID eventId) {
    defaultInstance.get().deleteMessageServeEvent(eventId);
  }

  public void deleteMessageServeEvent(UUID eventId) {
    admin.removeMessageServeEvent(eventId);
  }

  public static List<MessageServeEvent> findAllMessageEvents(MessagePattern pattern) {
    return defaultInstance.get().findMessageEvents(pattern);
  }

  public List<MessageServeEvent> findMessageEvents(MessagePattern pattern) {
    return admin.findMessageEventsMatching(pattern);
  }

  public static void verifyMessageEvent(MessagePattern pattern) {
    defaultInstance.get().verifyThatMessageEvent(pattern);
  }

  public void verifyThatMessageEvent(MessagePattern pattern) {
    verifyThatMessageEvent(moreThanOrExactly(1), pattern);
  }

  public static void verifyMessageEvent(int expectedCount, MessagePattern pattern) {
    defaultInstance.get().verifyThatMessageEvent(expectedCount, pattern);
  }

  public void verifyThatMessageEvent(int expectedCount, MessagePattern pattern) {
    verifyThatMessageEvent(exactly(expectedCount), pattern);
  }

  public static void verifyMessageEvent(
      CountMatchingStrategy expectedCount, MessagePattern pattern) {
    defaultInstance.get().verifyThatMessageEvent(expectedCount, pattern);
  }

  public void verifyThatMessageEvent(CountMatchingStrategy expectedCount, MessagePattern pattern) {
    int actualCount = admin.countMessageEventsMatching(pattern);
    if (!expectedCount.match(actualCount)) {
      throw new VerificationException(
          "Expected "
              + expectedCount
              + " message events matching pattern but found "
              + actualCount);
    }
  }

  public static Optional<MessageServeEvent> waitForMessageEvent(
      MessagePattern pattern, Duration maxWait) {
    return defaultInstance.get().waitForMessage(pattern, maxWait);
  }

  public Optional<MessageServeEvent> waitForMessage(MessagePattern pattern, Duration maxWait) {
    return admin.waitForMessageEvent(pattern, maxWait);
  }

  public static List<MessageServeEvent> waitForMessageEvents(
      MessagePattern pattern, int count, Duration maxWait) {
    return defaultInstance.get().waitForMessages(pattern, count, maxWait);
  }

  public List<MessageServeEvent> waitForMessages(
      MessagePattern pattern, int count, Duration maxWait) {
    return admin.waitForMessageEvents(pattern, count, maxWait);
  }

  public static void resetMessageJournal() {
    defaultInstance.get().resetMessages();
  }

  public void resetMessages() {
    admin.resetMessageJournal();
  }

  public static FindMessageServeEventsResult removeMessageServeEventsMatching(
      MessagePattern pattern) {
    return defaultInstance.get().removeMessageEvents(pattern);
  }

  public FindMessageServeEventsResult removeMessageEvents(MessagePattern pattern) {
    return admin.removeMessageServeEventsMatching(pattern);
  }

  public static FindMessageServeEventsResult removeMessageServeEventsForStubsMatchingMetadata(
      StringValuePattern pattern) {
    return defaultInstance.get().removeMessageEventsForStubsMatchingMetadata(pattern);
  }

  public FindMessageServeEventsResult removeMessageEventsForStubsMatchingMetadata(
      StringValuePattern pattern) {
    return admin.removeMessageServeEventsForStubsMatchingMetadata(pattern);
  }

  public enum JsonSchemaVersion {
    V4,
    V6,
    V7,
    V201909,
    V202012;

    public static final JsonSchemaVersion DEFAULT = V202012;

    public SpecVersion.VersionFlag toVersionFlag() {
      return switch (this) {
        case V4 -> SpecVersion.VersionFlag.V4;
        case V6 -> SpecVersion.VersionFlag.V6;
        case V7 -> SpecVersion.VersionFlag.V7;
        case V201909 -> SpecVersion.VersionFlag.V201909;
        case V202012 -> SpecVersion.VersionFlag.V202012;
      };
    }
  }
}
