/*
 * Decompiled with CFR 0.152.
 */
package org.apache.activemq.artemis.core.server.impl.quorum;

import java.util.Objects;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import org.apache.activemq.artemis.api.core.ActiveMQException;
import org.apache.activemq.artemis.core.server.NodeManager;
import org.apache.activemq.artemis.lockmanager.DistributedLock;
import org.apache.activemq.artemis.lockmanager.DistributedLockManager;
import org.apache.activemq.artemis.lockmanager.MutableLong;
import org.apache.activemq.artemis.lockmanager.UnavailableStateException;
import org.slf4j.Logger;

public final class ActivationSequenceStateMachine {
    private static final long CHECK_ACTIVATION_SEQUENCE_WAIT_MILLIS = 200L;
    private static final long PRIMARY_LOCK_ACQUIRE_TIMEOUT_MILLIS = 2000L;

    private ActivationSequenceStateMachine() {
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public static DistributedLock tryActivate(NodeManager nodeManager, DistributedLockManager distributedManager, Logger logger) throws InterruptedException, ExecutionException, TimeoutException, UnavailableStateException {
        Objects.requireNonNull(nodeManager);
        Objects.requireNonNull(distributedManager);
        Objects.requireNonNull(logger);
        Objects.requireNonNull(nodeManager.getNodeId());
        String nodeId = nodeManager.getNodeId().toString();
        long nodeActivationSequence = nodeManager.getNodeActivationSequence();
        if (nodeActivationSequence < 0L) {
            throw new IllegalArgumentException("nodeActivationSequence must be > 0");
        }
        DistributedLock activationLock = distributedManager.getDistributedLock(nodeId);
        try (MutableLong coordinatedNodeSequence = distributedManager.getMutableLong(nodeId);){
            while (true) {
                long timeout = 0L;
                switch (ActivationSequenceStateMachine.validateActivationSequence(coordinatedNodeSequence, activationLock, nodeId, nodeActivationSequence, logger)) {
                    case Stale: {
                        activationLock.close();
                        DistributedLock distributedLock = null;
                        return distributedLock;
                    }
                    case InSync: {
                        timeout = 2000L;
                        break;
                    }
                }
                if (!activationLock.tryLock(timeout, TimeUnit.MILLISECONDS)) {
                    logger.debug("Candidate for Node ID = {}, with local activation sequence: {}, cannot acquire primary lock within {}; retrying", new Object[]{nodeId, nodeActivationSequence, timeout});
                    if (timeout != 0L) continue;
                    Thread.sleep(200L);
                    continue;
                }
                switch (ActivationSequenceStateMachine.validateActivationSequence(coordinatedNodeSequence, activationLock, nodeId, nodeActivationSequence, logger)) {
                    case Stale: {
                        activationLock.close();
                        DistributedLock distributedLock = null;
                        return distributedLock;
                    }
                    case InSync: {
                        logger.info("Assuming active role for NodeID = {}, local activation sequence {} matches current coordinated activation sequence {}", new Object[]{nodeId, nodeActivationSequence, nodeActivationSequence});
                        DistributedLock distributedLock = activationLock;
                        return distributedLock;
                    }
                    case SelfRepair: {
                        logger.info("Assuming active role for NodeID = {}: local activation sequence {} matches claimed coordinated activation sequence {}. Repairing sequence", new Object[]{nodeId, nodeActivationSequence, nodeActivationSequence});
                        try {
                            ActivationSequenceStateMachine.repairActivationSequence(nodeManager, coordinatedNodeSequence, nodeActivationSequence, true);
                            DistributedLock distributedLock = activationLock;
                            return distributedLock;
                        }
                        catch (NodeManager.NodeManagerException | UnavailableStateException ex) {
                            activationLock.close();
                            throw ex;
                        }
                    }
                    case MaybeInSync: {
                        logger.warn("Assuming active role for NodeID = {}: repairing claimed activation sequence", (Object)nodeId);
                        try {
                            ActivationSequenceStateMachine.repairActivationSequence(nodeManager, coordinatedNodeSequence, nodeActivationSequence, false);
                            DistributedLock ex = activationLock;
                            return ex;
                        }
                        catch (NodeManager.NodeManagerException | UnavailableStateException ex) {
                            activationLock.close();
                            throw ex;
                        }
                    }
                }
            }
        }
    }

    private static void repairActivationSequence(NodeManager nodeManager, MutableLong coordinatedNodeSequence, long nodeActivationSequence, boolean selfHeal) throws UnavailableStateException {
        long coordinatedNodeActivationSequence;
        long l = coordinatedNodeActivationSequence = selfHeal ? nodeActivationSequence : nodeActivationSequence + 1L;
        if (!selfHeal) {
            nodeManager.writeNodeActivationSequence(coordinatedNodeActivationSequence);
        }
        coordinatedNodeSequence.set(coordinatedNodeActivationSequence);
    }

    private static ValidationResult validateActivationSequence(MutableLong coordinatedNodeSequence, DistributedLock activationLock, String lockAndLongId, long nodeActivationSequence, Logger logger) throws UnavailableStateException {
        assert (coordinatedNodeSequence.getMutableLongId().equals(lockAndLongId));
        assert (activationLock.getLockId().equals(lockAndLongId));
        long currentCoordinatedNodeSequence = coordinatedNodeSequence.get();
        if (nodeActivationSequence == currentCoordinatedNodeSequence) {
            return ValidationResult.InSync;
        }
        if (currentCoordinatedNodeSequence > 0L) {
            logger.info("Not a candidate for NodeID = {} activation, local activation sequence {} does not match coordinated activation sequence {}", new Object[]{lockAndLongId, nodeActivationSequence, currentCoordinatedNodeSequence});
            return ValidationResult.Stale;
        }
        long claimedCoordinatedNodeSequence = -currentCoordinatedNodeSequence;
        long sequenceGap = claimedCoordinatedNodeSequence - nodeActivationSequence;
        if (sequenceGap == 0L) {
            return ValidationResult.SelfRepair;
        }
        if (sequenceGap == 1L) {
            return ValidationResult.MaybeInSync;
        }
        assert (sequenceGap > 1L);
        logger.info("Not a candidate for NodeID = {} activation, local activation sequence {} does not match coordinated activation sequence {}", new Object[]{lockAndLongId, nodeActivationSequence, claimedCoordinatedNodeSequence});
        return ValidationResult.Stale;
    }

    public static boolean awaitNextCommittedActivationSequence(DistributedLockManager distributedManager, String coordinatedLockAndNodeId, long activationSequence, long timeoutMills, Logger logger) throws ExecutionException, InterruptedException, TimeoutException, UnavailableStateException {
        long elapsedNs;
        Objects.requireNonNull(distributedManager);
        Objects.requireNonNull(logger);
        Objects.requireNonNull(coordinatedLockAndNodeId);
        if (activationSequence < 0L) {
            throw new IllegalArgumentException("activationSequence must be >= 0, while is " + activationSequence);
        }
        if (!distributedManager.isStarted()) {
            throw new IllegalStateException("manager must be started");
        }
        MutableLong coordinatedActivationSequence = distributedManager.getMutableLong(coordinatedLockAndNodeId);
        boolean anyNext = false;
        long timeoutNs = TimeUnit.MILLISECONDS.toNanos(timeoutMills);
        long started = System.nanoTime();
        do {
            long coordinatedValue;
            if ((coordinatedValue = coordinatedActivationSequence.get()) > activationSequence) {
                logger.info("Detected a new activation sequence with NodeID = {}: and sequence: {}", (Object)coordinatedLockAndNodeId, (Object)coordinatedValue);
                anyNext = true;
                break;
            }
            if (coordinatedValue < 0L) {
                long claimedSequence = -coordinatedValue;
                long activationsGap = claimedSequence - activationSequence;
                if (activationsGap > 1L) {
                    logger.info("Detected furthers sequential server activations from sequence {}, with NodeID = {}: and claimed sequence: {}", new Object[]{activationSequence, coordinatedLockAndNodeId, claimedSequence});
                    anyNext = true;
                    break;
                }
                logger.debug("Detected claiming of activation sequence = {} for NodeID = {}", (Object)claimedSequence, (Object)coordinatedLockAndNodeId);
            }
            try {
                TimeUnit.MILLISECONDS.sleep(200L);
            }
            catch (InterruptedException interruptedException) {
                // empty catch block
            }
        } while ((elapsedNs = System.nanoTime() - started) < timeoutNs);
        return anyNext;
    }

    public static void ensureSequentialAccessToNodeData(String serverDescription, NodeManager nodeManager, DistributedLockManager distributedManager, Logger logger) throws ActiveMQException, InterruptedException, UnavailableStateException, ExecutionException, TimeoutException {
        long nextActivationSequence;
        Objects.requireNonNull(serverDescription);
        Objects.requireNonNull(nodeManager);
        Objects.requireNonNull(distributedManager);
        Objects.requireNonNull(logger);
        String lockAndLongId = Objects.toString(nodeManager.getNodeId(), null);
        Objects.requireNonNull(lockAndLongId);
        long nodeActivationSequence = nodeManager.getNodeActivationSequence();
        if (nodeActivationSequence < 0L) {
            throw new IllegalArgumentException("nodeActivationSequence must be >= 0");
        }
        DistributedLock primaryLock = distributedManager.getDistributedLock(lockAndLongId);
        if (!primaryLock.isHeldByCaller()) {
            String message = String.format("Server [%s] primary lock for NodeID = %s not held. Activation sequence cannot be safely changed", serverDescription, lockAndLongId);
            logger.info(message);
            throw new UnavailableStateException(message);
        }
        MutableLong coordinatedNodeActivationSequence = distributedManager.getMutableLong(lockAndLongId);
        if (!coordinatedNodeActivationSequence.compareAndSet(nodeActivationSequence, -(nextActivationSequence = nodeActivationSequence + 1L))) {
            String message = String.format("Server [%s], cannot assume active role for NodeID = %s, activation sequence claim failed, local activation sequence %d no longer matches current coordinated sequence %d", serverDescription, lockAndLongId, nodeActivationSequence, coordinatedNodeActivationSequence.get());
            logger.info(message);
            throw new ActiveMQException(message);
        }
        try {
            nodeManager.writeNodeActivationSequence(nextActivationSequence);
        }
        catch (NodeManager.NodeManagerException fatal) {
            logger.error("Server [{}] failed to set local activation sequence to: {} for NodeId ={}. Cannot continue committing coordinated activation sequence: REQUIRES ADMIN INTERVENTION", new Object[]{serverDescription, nextActivationSequence, lockAndLongId});
            throw new UnavailableStateException((Throwable)fatal);
        }
        logger.info("Server [{}], incremented local activation sequence to: {} for NodeId = {}", new Object[]{serverDescription, nextActivationSequence, lockAndLongId});
        if (!coordinatedNodeActivationSequence.compareAndSet(-nextActivationSequence, nextActivationSequence)) {
            String message = String.format("Server [%s], cannot assume active role for NodeID = %s, activation sequence commit failed, local activation sequence %d no longer matches current coordinated sequence %d", serverDescription, lockAndLongId, nodeActivationSequence, coordinatedNodeActivationSequence.get());
            logger.info(message);
            throw new ActiveMQException(message);
        }
        logger.info("Server [{}], incremented coordinated activation sequence to: {} for NodeId = {}", new Object[]{serverDescription, nextActivationSequence, lockAndLongId});
    }

    private static enum ValidationResult {
        Stale,
        InSync,
        MaybeInSync,
        SelfRepair;

    }
}

