/*
 * Decompiled with CFR 0.152.
 */
package io.opentelemetry.sdk.logs.export;

import io.opentelemetry.api.common.AttributeKey;
import io.opentelemetry.api.common.Attributes;
import io.opentelemetry.api.metrics.LongCounter;
import io.opentelemetry.api.metrics.Meter;
import io.opentelemetry.api.metrics.MeterProvider;
import io.opentelemetry.context.Context;
import io.opentelemetry.sdk.common.CompletableResultCode;
import io.opentelemetry.sdk.internal.DaemonThreadFactory;
import io.opentelemetry.sdk.logs.LogRecordProcessor;
import io.opentelemetry.sdk.logs.ReadWriteLogRecord;
import io.opentelemetry.sdk.logs.data.LogRecordData;
import io.opentelemetry.sdk.logs.export.BatchLogRecordProcessorBuilder;
import io.opentelemetry.sdk.logs.export.LogRecordExporter;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Queue;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicReference;
import java.util.logging.Level;
import java.util.logging.Logger;

public final class BatchLogRecordProcessor
implements LogRecordProcessor {
    private static final String WORKER_THREAD_NAME = BatchLogRecordProcessor.class.getSimpleName() + "_WorkerThread";
    private static final AttributeKey<String> LOG_RECORD_PROCESSOR_TYPE_LABEL = AttributeKey.stringKey((String)"logRecordProcessorType");
    private static final AttributeKey<Boolean> LOG_RECORD_PROCESSOR_DROPPED_LABEL = AttributeKey.booleanKey((String)"dropped");
    private static final String LOG_RECORD_PROCESSOR_TYPE_VALUE = BatchLogRecordProcessor.class.getSimpleName();
    private final Worker worker;
    private final AtomicBoolean isShutdown = new AtomicBoolean(false);

    public static BatchLogRecordProcessorBuilder builder(LogRecordExporter logRecordExporter) {
        return new BatchLogRecordProcessorBuilder(logRecordExporter);
    }

    BatchLogRecordProcessor(LogRecordExporter logRecordExporter, MeterProvider meterProvider, long scheduleDelayNanos, int maxQueueSize, int maxExportBatchSize, long exporterTimeoutNanos) {
        this.worker = new Worker(logRecordExporter, meterProvider, scheduleDelayNanos, maxExportBatchSize, exporterTimeoutNanos, new ArrayBlockingQueue(maxQueueSize));
        Thread workerThread = new DaemonThreadFactory(WORKER_THREAD_NAME).newThread((Runnable)this.worker);
        workerThread.start();
    }

    @Override
    public void onEmit(Context context, ReadWriteLogRecord logRecord) {
        if (logRecord == null) {
            return;
        }
        this.worker.addLog(logRecord);
    }

    @Override
    public CompletableResultCode shutdown() {
        if (this.isShutdown.getAndSet(true)) {
            return CompletableResultCode.ofSuccess();
        }
        return this.worker.shutdown();
    }

    @Override
    public CompletableResultCode forceFlush() {
        return this.worker.forceFlush();
    }

    ArrayList<LogRecordData> getBatch() {
        return this.worker.batch;
    }

    private static final class Worker
    implements Runnable {
        private static final Logger logger = Logger.getLogger(Worker.class.getName());
        private final LongCounter processedLogsCounter;
        private final Attributes droppedAttrs;
        private final Attributes exportedAttrs;
        private final LogRecordExporter logRecordExporter;
        private final long scheduleDelayNanos;
        private final int maxExportBatchSize;
        private final long exporterTimeoutNanos;
        private long nextExportTime;
        private final Queue<ReadWriteLogRecord> queue;
        private final AtomicInteger logsNeeded = new AtomicInteger(Integer.MAX_VALUE);
        private final BlockingQueue<Boolean> signal;
        private final AtomicReference<CompletableResultCode> flushRequested = new AtomicReference();
        private volatile boolean continueWork = true;
        private final ArrayList<LogRecordData> batch;

        private Worker(LogRecordExporter logRecordExporter, MeterProvider meterProvider, long scheduleDelayNanos, int maxExportBatchSize, long exporterTimeoutNanos, Queue<ReadWriteLogRecord> queue) {
            this.logRecordExporter = logRecordExporter;
            this.scheduleDelayNanos = scheduleDelayNanos;
            this.maxExportBatchSize = maxExportBatchSize;
            this.exporterTimeoutNanos = exporterTimeoutNanos;
            this.queue = queue;
            this.signal = new ArrayBlockingQueue<Boolean>(1);
            Meter meter = meterProvider.meterBuilder("io.opentelemetry.sdk.logs").build();
            meter.gaugeBuilder("queueSize").ofLongs().setDescription("The number of logs queued").setUnit("1").buildWithCallback(result -> result.record((long)queue.size(), Attributes.of((AttributeKey)LOG_RECORD_PROCESSOR_TYPE_LABEL, (Object)LOG_RECORD_PROCESSOR_TYPE_VALUE)));
            this.processedLogsCounter = meter.counterBuilder("processedLogs").setUnit("1").setDescription("The number of logs processed by the BatchLogRecordProcessor. [dropped=true if they were dropped due to high throughput]").build();
            this.droppedAttrs = Attributes.of((AttributeKey)LOG_RECORD_PROCESSOR_TYPE_LABEL, (Object)LOG_RECORD_PROCESSOR_TYPE_VALUE, (AttributeKey)LOG_RECORD_PROCESSOR_DROPPED_LABEL, (Object)true);
            this.exportedAttrs = Attributes.of((AttributeKey)LOG_RECORD_PROCESSOR_TYPE_LABEL, (Object)LOG_RECORD_PROCESSOR_TYPE_VALUE, (AttributeKey)LOG_RECORD_PROCESSOR_DROPPED_LABEL, (Object)false);
            this.batch = new ArrayList(this.maxExportBatchSize);
        }

        private void addLog(ReadWriteLogRecord logData) {
            if (!this.queue.offer(logData)) {
                this.processedLogsCounter.add(1L, this.droppedAttrs);
            } else if (this.queue.size() >= this.logsNeeded.get()) {
                this.signal.offer(true);
            }
        }

        @Override
        public void run() {
            this.updateNextExportTime();
            while (this.continueWork) {
                if (this.flushRequested.get() != null) {
                    this.flush();
                }
                while (!this.queue.isEmpty() && this.batch.size() < this.maxExportBatchSize) {
                    this.batch.add(this.queue.poll().toLogRecordData());
                }
                if (this.batch.size() >= this.maxExportBatchSize || System.nanoTime() >= this.nextExportTime) {
                    this.exportCurrentBatch();
                    this.updateNextExportTime();
                }
                if (!this.queue.isEmpty()) continue;
                try {
                    long pollWaitTime = this.nextExportTime - System.nanoTime();
                    if (pollWaitTime <= 0L) continue;
                    this.logsNeeded.set(this.maxExportBatchSize - this.batch.size());
                    this.signal.poll(pollWaitTime, TimeUnit.NANOSECONDS);
                    this.logsNeeded.set(Integer.MAX_VALUE);
                }
                catch (InterruptedException e) {
                    Thread.currentThread().interrupt();
                    return;
                }
            }
        }

        private void flush() {
            for (int logsToFlush = this.queue.size(); logsToFlush > 0; --logsToFlush) {
                ReadWriteLogRecord logRecord = this.queue.poll();
                assert (logRecord != null);
                this.batch.add(logRecord.toLogRecordData());
                if (this.batch.size() < this.maxExportBatchSize) continue;
                this.exportCurrentBatch();
            }
            this.exportCurrentBatch();
            CompletableResultCode flushResult = this.flushRequested.get();
            if (flushResult != null) {
                flushResult.succeed();
                this.flushRequested.set(null);
            }
        }

        private void updateNextExportTime() {
            this.nextExportTime = System.nanoTime() + this.scheduleDelayNanos;
        }

        private CompletableResultCode shutdown() {
            CompletableResultCode result = new CompletableResultCode();
            CompletableResultCode flushResult = this.forceFlush();
            flushResult.whenComplete(() -> {
                this.continueWork = false;
                CompletableResultCode shutdownResult = this.logRecordExporter.shutdown();
                shutdownResult.whenComplete(() -> {
                    if (!flushResult.isSuccess() || !shutdownResult.isSuccess()) {
                        result.fail();
                    } else {
                        result.succeed();
                    }
                });
            });
            return result;
        }

        private CompletableResultCode forceFlush() {
            CompletableResultCode possibleResult;
            CompletableResultCode flushResult = new CompletableResultCode();
            if (this.flushRequested.compareAndSet(null, flushResult)) {
                this.signal.offer(true);
            }
            return (possibleResult = this.flushRequested.get()) == null ? CompletableResultCode.ofSuccess() : possibleResult;
        }

        private void exportCurrentBatch() {
            if (this.batch.isEmpty()) {
                return;
            }
            try {
                CompletableResultCode result = this.logRecordExporter.export(Collections.unmodifiableList(this.batch));
                result.join(this.exporterTimeoutNanos, TimeUnit.NANOSECONDS);
                if (result.isSuccess()) {
                    this.processedLogsCounter.add((long)this.batch.size(), this.exportedAttrs);
                } else {
                    logger.log(Level.FINE, "Exporter failed");
                }
            }
            catch (RuntimeException e) {
                logger.log(Level.WARNING, "Exporter threw an Exception", e);
            }
            finally {
                this.batch.clear();
            }
        }
    }
}

