/*
 * Decompiled with CFR 0.152.
 */
package org.apache.activemq.artemis.tests.integration.stomp;

import jakarta.jms.BytesMessage;
import jakarta.jms.Destination;
import jakarta.jms.Message;
import jakarta.jms.MessageConsumer;
import jakarta.jms.MessageListener;
import jakarta.jms.MessageProducer;
import jakarta.jms.TextMessage;
import java.io.ByteArrayOutputStream;
import java.net.URI;
import java.nio.charset.StandardCharsets;
import java.util.HashSet;
import java.util.UUID;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.apache.activemq.artemis.api.config.ActiveMQDefaultConfiguration;
import org.apache.activemq.artemis.api.core.QueueConfiguration;
import org.apache.activemq.artemis.api.core.RoutingType;
import org.apache.activemq.artemis.api.core.SimpleString;
import org.apache.activemq.artemis.api.core.client.ActiveMQClient;
import org.apache.activemq.artemis.api.core.client.ClientMessage;
import org.apache.activemq.artemis.api.core.client.ClientProducer;
import org.apache.activemq.artemis.api.core.client.ClientSession;
import org.apache.activemq.artemis.api.core.client.ClientSessionFactory;
import org.apache.activemq.artemis.api.core.client.ServerLocator;
import org.apache.activemq.artemis.api.core.management.ManagementHelper;
import org.apache.activemq.artemis.api.jms.ActiveMQJMSClient;
import org.apache.activemq.artemis.core.management.impl.ActiveMQServerControlImpl;
import org.apache.activemq.artemis.core.postoffice.Binding;
import org.apache.activemq.artemis.core.postoffice.QueueBinding;
import org.apache.activemq.artemis.core.postoffice.impl.LocalQueueBinding;
import org.apache.activemq.artemis.core.server.ActiveMQServer;
import org.apache.activemq.artemis.core.server.Queue;
import org.apache.activemq.artemis.core.server.impl.ActiveMQServerImpl;
import org.apache.activemq.artemis.core.server.impl.AddressInfo;
import org.apache.activemq.artemis.core.settings.impl.AddressSettings;
import org.apache.activemq.artemis.logs.AssertionLoggerHandler;
import org.apache.activemq.artemis.tests.integration.mqtt.imported.FuseMQTTClientProvider;
import org.apache.activemq.artemis.tests.integration.stomp.StompTestBase;
import org.apache.activemq.artemis.tests.integration.stomp.util.ClientStompFrame;
import org.apache.activemq.artemis.tests.integration.stomp.util.StompClientConnection;
import org.apache.activemq.artemis.tests.integration.stomp.util.StompClientConnectionFactory;
import org.apache.activemq.artemis.tests.util.Wait;
import org.apache.activemq.artemis.utils.RandomUtil;
import org.apache.activemq.artemis.utils.collections.IterableStream;
import org.jboss.logging.Logger;
import org.junit.After;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

@RunWith(value=Parameterized.class)
public class StompTest
extends StompTestBase {
    private static final Logger log = Logger.getLogger(StompTest.class);
    protected StompClientConnection conn;

    @Override
    protected ActiveMQServer createServer() throws Exception {
        ActiveMQServer server = super.createServer();
        server.getConfiguration().setAddressQueueScanPeriod(100L);
        return server;
    }

    @Override
    @Before
    public void setUp() throws Exception {
        super.setUp();
        this.conn = StompClientConnectionFactory.createClientConnection(this.uri);
    }

    @Override
    @After
    public void tearDown() throws Exception {
        try {
            boolean connected = this.conn != null && this.conn.isConnected();
            log.debug((Object)("Connection 1.0 connected: " + connected));
            if (connected) {
                try {
                    this.conn.disconnect();
                }
                catch (Exception exception) {
                    // empty catch block
                }
            }
        }
        finally {
            super.tearDown();
            this.conn.closeTransport();
        }
    }

    @Test
    public void testConnectionTTL() throws Exception {
        int port = 61614;
        URI uri = StompTest.createStompClientUri(this.scheme, this.hostname, port);
        this.server.getRemotingService().createAcceptor("test", "tcp://127.0.0.1:" + port + "?connectionTtl=1000").start();
        StompClientConnection conn = StompClientConnectionFactory.createClientConnection(uri);
        conn.connect("brianm", "wombats");
        Thread.sleep(5000L);
        ClientStompFrame frame = conn.receiveFrame();
        StompTest.assertEquals((Object)"ERROR", (Object)frame.getCommand());
        StompTest.assertFalse((boolean)conn.isConnected());
    }

    @Test
    public void testSendManyMessages() throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        MessageConsumer consumer = this.session.createConsumer((Destination)this.queue);
        int count = 1000;
        final CountDownLatch latch = new CountDownLatch(count);
        consumer.setMessageListener(new MessageListener(){

            public void onMessage(Message arg0) {
                latch.countDown();
            }
        });
        for (int i = 1; i <= count; ++i) {
            StompTest.send(this.conn, StompTest.getQueuePrefix() + StompTest.getQueueName(), null, "Hello World!");
        }
        StompTest.assertTrue((boolean)latch.await(60L, TimeUnit.SECONDS));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testSendOverDiskFull() throws Exception {
        AssertionLoggerHandler.startCapture();
        try {
            MessageConsumer consumer = this.session.createConsumer((Destination)this.queue);
            this.conn.connect(this.defUser, this.defPass);
            int count = 1000;
            final CountDownLatch latch = new CountDownLatch(count);
            consumer.setMessageListener(new MessageListener(){

                public void onMessage(Message arg0) {
                    latch.countDown();
                }
            });
            ((ActiveMQServerImpl)this.server).getMonitor().setMaxUsage(0.0).tick();
            Exception e = null;
            try {
                for (int i = 1; i <= count; ++i) {
                    StompTest.send(this.conn, StompTest.getQueuePrefix() + StompTest.getQueueName(), null, "Hello World!");
                }
            }
            catch (Exception se) {
                e = se;
            }
            StompTest.assertNotNull((Object)e);
            AssertionLoggerHandler.findText((String[])new String[]{"AMQ119119"});
        }
        finally {
            AssertionLoggerHandler.clear();
            AssertionLoggerHandler.stopCapture();
        }
    }

    @Test
    public void testConnect() throws Exception {
        ClientStompFrame frame = this.conn.createFrame("CONNECT").addHeader("login", this.defUser).addHeader("passcode", this.defPass).addHeader("request-id", "1");
        ClientStompFrame response = this.conn.sendFrame(frame);
        Assert.assertTrue((boolean)response.getCommand().equals("CONNECTED"));
        Assert.assertTrue((boolean)response.getHeader("response-id").equals("1"));
    }

    @Test
    public void testDisconnectAndError() throws Exception {
        ClientStompFrame frame = this.conn.createFrame("CONNECT").addHeader("login", this.defUser).addHeader("passcode", this.defPass).addHeader("request-id", "1");
        ClientStompFrame response = this.conn.sendFrame(frame);
        Assert.assertTrue((boolean)response.getCommand().equals("CONNECTED"));
        Assert.assertTrue((boolean)response.getHeader("response-id").equals("1"));
        this.conn.disconnect();
        try {
            StompTest.send(this.conn, StompTest.getQueuePrefix() + StompTest.getQueueName(), null, "Hello World!");
            StompTest.fail((String)"Should have thrown an exception since connection is disconnected");
        }
        catch (Exception exception) {
            // empty catch block
        }
    }

    @Test
    public void testSendMessage() throws Exception {
        MessageConsumer consumer = this.session.createConsumer((Destination)this.queue);
        this.conn.connect(this.defUser, this.defPass);
        StompTest.send(this.conn, StompTest.getQueuePrefix() + StompTest.getQueueName(), null, "Hello World");
        TextMessage message = (TextMessage)consumer.receive(1000L);
        Assert.assertNotNull((Object)message);
        Assert.assertEquals((Object)"Hello World", (Object)message.getText());
        Assert.assertEquals((String)"getJMSPriority", (long)4L, (long)message.getJMSPriority());
        long tnow = System.currentTimeMillis();
        long tmsg = message.getJMSTimestamp();
        Assert.assertTrue((Math.abs(tnow - tmsg) < 1000L ? (byte)1 : 0) != 0);
    }

    @Test
    public void sendSTOMPReceiveMQTT() throws Exception {
        FuseMQTTClientProvider clientProvider = new FuseMQTTClientProvider();
        clientProvider.connect("tcp://" + this.hostname + ":" + 61613);
        clientProvider.subscribe(StompTest.getTopicPrefix() + StompTest.getTopicName(), 0);
        String stompPayload = "This is a test message";
        this.conn.connect(this.defUser, this.defPass);
        StompTest.send(this.conn, StompTest.getTopicPrefix() + StompTest.getTopicName(), null, stompPayload);
        byte[] mqttPayload = clientProvider.receive(10000);
        clientProvider.disconnect();
        StompTest.assertEquals((Object)stompPayload, (Object)new String(mqttPayload, StandardCharsets.UTF_8));
        clientProvider.disconnect();
    }

    @Test
    public void sendMQTTReceiveSTOMP() throws Exception {
        String payload = "This is a test message";
        this.conn.connect(this.defUser, this.defPass);
        StompTest.subscribe(this.conn, null, "auto");
        FuseMQTTClientProvider clientProvider = new FuseMQTTClientProvider();
        clientProvider.connect("tcp://" + this.hostname + ":" + 61613);
        clientProvider.publish(StompTest.getQueuePrefix() + StompTest.getQueueName(), payload.getBytes(), 0);
        clientProvider.disconnect();
        ClientStompFrame frame = this.conn.receiveFrame();
        StompTest.assertTrue((boolean)frame.getBody().contains(payload));
    }

    public void sendMessageToNonExistentQueue(String queuePrefix, String queue, RoutingType routingType) throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        StompTest.send(this.conn, queuePrefix + queue, null, "Hello World", true, routingType);
        MessageConsumer consumer = this.session.createConsumer((Destination)ActiveMQJMSClient.createQueue((String)queue));
        TextMessage message = (TextMessage)consumer.receive(1000L);
        Assert.assertNotNull((Object)message);
        Assert.assertEquals((Object)"Hello World", (Object)message.getText());
        Assert.assertEquals((String)"getJMSPriority", (long)4L, (long)message.getJMSPriority());
        long tnow = System.currentTimeMillis();
        long tmsg = message.getJMSTimestamp();
        Assert.assertTrue((Math.abs(tnow - tmsg) < 1500L ? (byte)1 : 0) != 0);
        StompTest.assertNotNull((Object)this.server.getPostOffice().getBinding(new SimpleString(queue)));
        consumer.close();
        Wait.assertTrue(() -> this.server.getPostOffice().getBinding(new SimpleString(queue)) == null);
    }

    @Test
    public void testSendMessageToNonExistentQueue() throws Exception {
        this.sendMessageToNonExistentQueue(StompTest.getQueuePrefix(), RandomUtil.randomString(), RoutingType.ANYCAST);
    }

    @Test
    public void testSendMessageToNonExistentQueueUsingExplicitDefaultRouting() throws Exception {
        String nonExistentQueue = RandomUtil.randomString();
        this.server.getAddressSettingsRepository().addMatch(nonExistentQueue, (Object)new AddressSettings().setDefaultAddressRoutingType(RoutingType.ANYCAST).setDefaultQueueRoutingType(RoutingType.ANYCAST));
        this.sendMessageToNonExistentQueue(StompTest.getQueuePrefix(), nonExistentQueue, null);
    }

    public void sendMessageToNonExistentTopic(String topicPrefix, String topic, RoutingType routingType) throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        StompTest.send(this.conn, topicPrefix + topic, null, "Hello World", true, routingType);
        MessageConsumer consumer = this.session.createConsumer((Destination)ActiveMQJMSClient.createTopic((String)topic));
        StompTest.send(this.conn, topicPrefix + topic, null, "Hello World", true, routingType);
        TextMessage message = (TextMessage)consumer.receive(1000L);
        Assert.assertNotNull((Object)message);
        Assert.assertEquals((Object)"Hello World", (Object)message.getText());
        Assert.assertEquals((String)"getJMSPriority", (long)4L, (long)message.getJMSPriority());
        long tnow = System.currentTimeMillis();
        long tmsg = message.getJMSTimestamp();
        Assert.assertTrue((Math.abs(tnow - tmsg) < 1500L ? (byte)1 : 0) != 0);
        StompTest.assertNotNull((Object)this.server.getAddressInfo(new SimpleString(topic)));
        consumer.close();
        Wait.assertTrue(() -> this.server.getAddressInfo(new SimpleString(topic)) == null);
    }

    @Test
    public void testSendMessageToNonExistentTopic() throws Exception {
        this.sendMessageToNonExistentTopic(StompTest.getTopicPrefix(), RandomUtil.randomString(), RoutingType.MULTICAST);
    }

    @Test
    public void testSendMessageToNonExistentTopicUsingExplicitDefaultRouting() throws Exception {
        String nonExistentTopic = RandomUtil.randomString();
        this.server.getAddressSettingsRepository().addMatch(nonExistentTopic, (Object)new AddressSettings().setDefaultAddressRoutingType(RoutingType.MULTICAST).setDefaultQueueRoutingType(RoutingType.MULTICAST));
        this.sendMessageToNonExistentTopic(StompTest.getTopicPrefix(), nonExistentTopic, null);
    }

    @Test
    public void testSendMessageToNonExistentTopicUsingImplicitDefaultRouting() throws Exception {
        this.sendMessageToNonExistentTopic(StompTest.getTopicPrefix(), RandomUtil.randomString(), null);
    }

    @Test
    public void testSendMessageWithLeadingNewLine() throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        ClientStompFrame frame = this.conn.createFrame("SEND").addHeader("destination", StompTest.getQueuePrefix() + StompTest.getQueueName()).setBody("Hello World");
        this.conn.sendWickedFrame(frame);
        MessageConsumer consumer = this.session.createConsumer((Destination)this.queue);
        TextMessage message = (TextMessage)consumer.receive(1000L);
        Assert.assertNotNull((Object)message);
        Assert.assertEquals((Object)"Hello World", (Object)message.getText());
        StompTest.send(this.conn, StompTest.getQueuePrefix() + StompTest.getQueueName(), null, "Hello World");
        message = (TextMessage)consumer.receive(1000L);
        Assert.assertNotNull((Object)message);
        Assert.assertEquals((Object)"Hello World", (Object)message.getText());
        long tnow = System.currentTimeMillis();
        long tmsg = message.getJMSTimestamp();
        Assert.assertTrue((Math.abs(tnow - tmsg) < 1000L ? (byte)1 : 0) != 0);
    }

    @Test
    public void testSendMessageWithReceipt() throws Exception {
        MessageConsumer consumer = this.session.createConsumer((Destination)this.queue);
        this.conn.connect(this.defUser, this.defPass);
        StompTest.send(this.conn, StompTest.getQueuePrefix() + StompTest.getQueueName(), null, "Hello World", true);
        TextMessage message = (TextMessage)consumer.receive(1000L);
        Assert.assertNotNull((Object)message);
        Assert.assertEquals((Object)"Hello World", (Object)message.getText());
        long tnow = System.currentTimeMillis();
        long tmsg = message.getJMSTimestamp();
        Assert.assertTrue((Math.abs(tnow - tmsg) < 1000L ? (byte)1 : 0) != 0);
    }

    @Test
    public void testSendMessageWithContentLength() throws Exception {
        MessageConsumer consumer = this.session.createConsumer((Destination)this.queue);
        this.conn.connect(this.defUser, this.defPass);
        byte[] data = new byte[]{1, 0, 0, 4};
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        baos.write(data);
        baos.flush();
        ClientStompFrame frame = this.conn.createFrame("SEND").addHeader("destination", StompTest.getQueuePrefix() + StompTest.getQueueName()).addHeader("content-length", Integer.toString(data.length)).setBody(new String(baos.toByteArray()));
        this.conn.sendFrame(frame);
        BytesMessage message = (BytesMessage)consumer.receive(10000L);
        Assert.assertNotNull((Object)message);
        StompTest.assertEquals((long)data.length, (long)message.getBodyLength());
        StompTest.assertEquals((long)data[0], (long)message.readByte());
        StompTest.assertEquals((long)data[1], (long)message.readByte());
        StompTest.assertEquals((long)data[2], (long)message.readByte());
        StompTest.assertEquals((long)data[3], (long)message.readByte());
    }

    @Test
    public void testJMSXGroupIdCanBeSet() throws Exception {
        String jmsxGroupID = "JMSXGroupID";
        MessageConsumer consumer = this.session.createConsumer((Destination)this.queue);
        this.conn.connect(this.defUser, this.defPass);
        ClientStompFrame frame = this.conn.createFrame("SEND").addHeader("destination", StompTest.getQueuePrefix() + StompTest.getQueueName()).addHeader("JMSXGroupID", "JMSXGroupID").setBody("Hello World");
        this.conn.sendFrame(frame);
        TextMessage message = (TextMessage)consumer.receive(1000L);
        Assert.assertNotNull((Object)message);
        Assert.assertEquals((Object)"Hello World", (Object)message.getText());
        Assert.assertEquals((Object)"JMSXGroupID", (Object)message.getStringProperty("JMSXGroupID"));
    }

    @Test
    public void testSendMessageWithCustomHeadersAndSelector() throws Exception {
        MessageConsumer consumer = this.session.createConsumer((Destination)this.queue, "foo = 'abc'");
        this.conn.connect(this.defUser, this.defPass);
        ClientStompFrame frame = this.conn.createFrame("SEND").addHeader("destination", StompTest.getQueuePrefix() + StompTest.getQueueName()).addHeader("foo", "abc").addHeader("bar", "123").setBody("Hello World");
        this.conn.sendFrame(frame);
        TextMessage message = (TextMessage)consumer.receive(1000L);
        Assert.assertNotNull((Object)message);
        Assert.assertEquals((Object)"Hello World", (Object)message.getText());
        Assert.assertEquals((String)"foo", (Object)"abc", (Object)message.getStringProperty("foo"));
        Assert.assertEquals((String)"bar", (Object)"123", (Object)message.getStringProperty("bar"));
    }

    @Test
    public void testSendMessageWithCustomHeadersAndHyphenatedSelector() throws Exception {
        MessageConsumer consumer = this.session.createConsumer((Destination)this.queue, "hyphenated_props:b-ar = '123'");
        this.conn.connect(this.defUser, this.defPass);
        ClientStompFrame frame = this.conn.createFrame("SEND").addHeader("destination", StompTest.getQueuePrefix() + StompTest.getQueueName()).addHeader("foo", "abc").addHeader("b-ar", "123").setBody("Hello World");
        this.conn.sendFrame(frame);
        TextMessage message = (TextMessage)consumer.receive(1000L);
        Assert.assertNotNull((Object)message);
        Assert.assertEquals((Object)"Hello World", (Object)message.getText());
        Assert.assertEquals((String)"foo", (Object)"abc", (Object)message.getStringProperty("foo"));
        Assert.assertEquals((String)"b-ar", (Object)"123", (Object)message.getStringProperty("b-ar"));
    }

    @Test
    public void testSendMessageWithStandardHeaders() throws Exception {
        MessageConsumer consumer = this.session.createConsumer((Destination)this.queue);
        this.conn.connect(this.defUser, this.defPass);
        ClientStompFrame frame = this.conn.createFrame("SEND").addHeader("destination", StompTest.getQueuePrefix() + StompTest.getQueueName()).addHeader("foo", "abc").addHeader("bar", "123").addHeader("correlation-id", "c123").addHeader("persistent", "true").addHeader("type", "t345").addHeader("JMSXGroupID", "abc").addHeader("priority", "3").setBody("Hello World");
        this.conn.sendFrame(frame);
        TextMessage message = (TextMessage)consumer.receive(1000L);
        Assert.assertNotNull((Object)message);
        Assert.assertEquals((Object)"Hello World", (Object)message.getText());
        Assert.assertEquals((String)"JMSCorrelationID", (Object)"c123", (Object)message.getJMSCorrelationID());
        Assert.assertEquals((String)"getJMSType", (Object)"t345", (Object)message.getJMSType());
        Assert.assertEquals((String)"getJMSPriority", (long)3L, (long)message.getJMSPriority());
        Assert.assertEquals((long)2L, (long)message.getJMSDeliveryMode());
        Assert.assertEquals((String)"foo", (Object)"abc", (Object)message.getStringProperty("foo"));
        Assert.assertEquals((String)"bar", (Object)"123", (Object)message.getStringProperty("bar"));
        Assert.assertEquals((String)"JMSXGroupID", (Object)"abc", (Object)message.getStringProperty("JMSXGroupID"));
    }

    @Test
    public void testSendMessageWithLongHeaders() throws Exception {
        MessageConsumer consumer = this.session.createConsumer((Destination)this.queue);
        this.conn.connect(this.defUser, this.defPass);
        StringBuffer buffer = new StringBuffer();
        for (int i = 0; i < 1024; ++i) {
            buffer.append("a");
        }
        ClientStompFrame frame = this.conn.createFrame("SEND").addHeader("destination", StompTest.getQueuePrefix() + StompTest.getQueueName()).addHeader("foo", "abc").addHeader("bar", "123").addHeader("correlation-id", "c123").addHeader("persistent", "true").addHeader("type", "t345").addHeader("JMSXGroupID", "abc").addHeader("priority", "3").addHeader("longHeader", buffer.toString()).setBody("Hello World");
        this.conn.sendFrame(frame);
        TextMessage message = (TextMessage)consumer.receive(1000L);
        Assert.assertNotNull((Object)message);
        Assert.assertEquals((Object)"Hello World", (Object)message.getText());
        Assert.assertEquals((String)"JMSCorrelationID", (Object)"c123", (Object)message.getJMSCorrelationID());
        Assert.assertEquals((String)"getJMSType", (Object)"t345", (Object)message.getJMSType());
        Assert.assertEquals((String)"getJMSPriority", (long)3L, (long)message.getJMSPriority());
        Assert.assertEquals((long)2L, (long)message.getJMSDeliveryMode());
        Assert.assertEquals((String)"foo", (Object)"abc", (Object)message.getStringProperty("foo"));
        Assert.assertEquals((String)"longHeader", (long)1024L, (long)message.getStringProperty("longHeader").length());
        Assert.assertEquals((String)"JMSXGroupID", (Object)"abc", (Object)message.getStringProperty("JMSXGroupID"));
    }

    @Test
    public void testSendMessageWithDelay() throws Exception {
        MessageConsumer consumer = this.session.createConsumer((Destination)this.queue);
        this.conn.connect(this.defUser, this.defPass);
        ClientStompFrame frame = this.conn.createFrame("SEND").addHeader("destination", StompTest.getQueuePrefix() + StompTest.getQueueName()).addHeader("foo", "abc").addHeader("bar", "123").addHeader("correlation-id", "c123").addHeader("persistent", "true").addHeader("type", "t345").addHeader("JMSXGroupID", "abc").addHeader("priority", "3").addHeader("AMQ_SCHEDULED_DELAY", "2000").setBody("Hello World");
        this.conn.sendFrame(frame);
        StompTest.assertNull((String)"Should not receive message yet", (Object)consumer.receive(100L));
        TextMessage message = (TextMessage)consumer.receive(4000L);
        Assert.assertNotNull((Object)message);
        Assert.assertEquals((Object)"Hello World", (Object)message.getText());
        Assert.assertEquals((String)"JMSCorrelationID", (Object)"c123", (Object)message.getJMSCorrelationID());
        Assert.assertEquals((String)"getJMSType", (Object)"t345", (Object)message.getJMSType());
        Assert.assertEquals((String)"getJMSPriority", (long)3L, (long)message.getJMSPriority());
        Assert.assertEquals((long)2L, (long)message.getJMSDeliveryMode());
        Assert.assertEquals((String)"foo", (Object)"abc", (Object)message.getStringProperty("foo"));
        Assert.assertEquals((String)"JMSXGroupID", (Object)"abc", (Object)message.getStringProperty("JMSXGroupID"));
    }

    @Test
    public void testSendMessageWithDeliveryTime() throws Exception {
        MessageConsumer consumer = this.session.createConsumer((Destination)this.queue);
        this.conn.connect(this.defUser, this.defPass);
        ClientStompFrame frame = this.conn.createFrame("SEND").addHeader("destination", StompTest.getQueuePrefix() + StompTest.getQueueName()).addHeader("foo", "abc").addHeader("bar", "123").addHeader("correlation-id", "c123").addHeader("persistent", "true").addHeader("type", "t345").addHeader("JMSXGroupID", "abc").addHeader("priority", "3").addHeader("AMQ_SCHEDULED_TIME", Long.toString(System.currentTimeMillis() + 2000L)).setBody("Hello World");
        this.conn.sendFrame(frame);
        StompTest.assertNull((String)"Should not receive message yet", (Object)consumer.receive(100L));
        TextMessage message = (TextMessage)consumer.receive(4000L);
        Assert.assertNotNull((Object)message);
        Assert.assertEquals((Object)"Hello World", (Object)message.getText());
        Assert.assertEquals((String)"JMSCorrelationID", (Object)"c123", (Object)message.getJMSCorrelationID());
        Assert.assertEquals((String)"getJMSType", (Object)"t345", (Object)message.getJMSType());
        Assert.assertEquals((String)"getJMSPriority", (long)3L, (long)message.getJMSPriority());
        Assert.assertEquals((long)2L, (long)message.getJMSDeliveryMode());
        Assert.assertEquals((String)"foo", (Object)"abc", (Object)message.getStringProperty("foo"));
        Assert.assertEquals((String)"JMSXGroupID", (Object)"abc", (Object)message.getStringProperty("JMSXGroupID"));
    }

    @Test
    public void testSendMessageWithDelayWithBadValue() throws Exception {
        MessageConsumer consumer = this.session.createConsumer((Destination)this.queue);
        this.conn.connect(this.defUser, this.defPass);
        ClientStompFrame frame = this.conn.createFrame("SEND").addHeader("destination", StompTest.getQueuePrefix() + StompTest.getQueueName()).addHeader("AMQ_SCHEDULED_DELAY", "foo").setBody("Hello World");
        this.conn.sendFrame(frame);
        StompTest.assertNull((String)"Should not receive message yet", (Object)consumer.receive(100L));
        ClientStompFrame error = this.conn.receiveFrame();
        Assert.assertNotNull((Object)error);
        Assert.assertTrue((boolean)error.getCommand().equals("ERROR"));
    }

    @Test
    public void testSendMessageWithDeliveryTimeWithBadValue() throws Exception {
        MessageConsumer consumer = this.session.createConsumer((Destination)this.queue);
        this.conn.connect(this.defUser, this.defPass);
        ClientStompFrame frame = this.conn.createFrame("SEND").addHeader("destination", StompTest.getQueuePrefix() + StompTest.getQueueName()).addHeader("AMQ_SCHEDULED_TIME", "foo").setBody("Hello World");
        this.conn.sendFrame(frame);
        StompTest.assertNull((String)"Should not receive message yet", (Object)consumer.receive(100L));
        ClientStompFrame error = this.conn.receiveFrame();
        Assert.assertNotNull((Object)error);
        Assert.assertTrue((boolean)error.getCommand().equals("ERROR"));
    }

    @Test
    public void testSubscribeWithAutoAck() throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        StompTest.subscribe(this.conn, null, "auto");
        this.sendJmsMessage(this.getName());
        ClientStompFrame frame = this.conn.receiveFrame(10000L);
        Assert.assertEquals((Object)"MESSAGE", (Object)frame.getCommand());
        Assert.assertEquals((Object)(StompTest.getQueuePrefix() + StompTest.getQueueName()), (Object)frame.getHeader("destination"));
        Assert.assertEquals((Object)this.getName(), (Object)frame.getBody());
        Pattern cl = Pattern.compile("content-length:\\s*(\\d+)", 2);
        Matcher cl_matcher = cl.matcher(frame.toString());
        Assert.assertFalse((boolean)cl_matcher.find());
        this.conn.disconnect();
        MessageConsumer consumer = this.session.createConsumer((Destination)this.queue);
        Message message = consumer.receive(100L);
        Assert.assertNull((Object)message);
    }

    @Test
    public void testIngressTimestamp() throws Exception {
        this.server.getAddressSettingsRepository().addMatch("#", (Object)new AddressSettings().setEnableIngressTimestamp(true));
        this.conn.connect(this.defUser, this.defPass);
        StompTest.subscribe(this.conn, null, "auto");
        long beforeSend = System.currentTimeMillis();
        this.sendJmsMessage(this.getName());
        long afterSend = System.currentTimeMillis();
        ClientStompFrame frame = this.conn.receiveFrame(10000L);
        Assert.assertEquals((Object)"MESSAGE", (Object)frame.getCommand());
        Assert.assertEquals((Object)(StompTest.getQueuePrefix() + StompTest.getQueueName()), (Object)frame.getHeader("destination"));
        String ingressTimestampHeader = frame.getHeader("ingress-timestamp");
        Assert.assertNotNull((Object)ingressTimestampHeader);
        long ingressTimestamp = Long.parseLong(ingressTimestampHeader);
        StompTest.assertTrue((String)("Ingress timstamp " + ingressTimestamp + " should be >= " + beforeSend + " and <= " + afterSend), (ingressTimestamp >= beforeSend && ingressTimestamp <= afterSend ? (byte)1 : 0) != 0);
        this.conn.disconnect();
    }

    @Test
    public void testAnycastDestinationTypeMessageProperty() throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        StompTest.subscribe(this.conn, null, "auto");
        StompTest.send(this.conn, StompTest.getQueuePrefix() + StompTest.getQueueName(), null, this.getName(), true, RoutingType.ANYCAST);
        ClientStompFrame frame = this.conn.receiveFrame(10000L);
        Assert.assertEquals((Object)"MESSAGE", (Object)frame.getCommand());
        Assert.assertEquals((Object)(StompTest.getQueuePrefix() + StompTest.getQueueName()), (Object)frame.getHeader("destination"));
        Assert.assertEquals((Object)RoutingType.ANYCAST.toString(), (Object)frame.getHeader("destination-type"));
        Assert.assertTrue((frame.getHeader(org.apache.activemq.artemis.api.core.Message.HDR_ROUTING_TYPE.toString()) == null ? (byte)1 : 0) != 0);
        Assert.assertEquals((Object)this.getName(), (Object)frame.getBody());
        this.conn.disconnect();
    }

    @Test
    public void testMulticastDestinationTypeMessageProperty() throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        StompTest.subscribeTopic(this.conn, null, null, null);
        StompTest.send(this.conn, StompTest.getTopicPrefix() + StompTest.getTopicName(), null, this.getName(), true, RoutingType.MULTICAST);
        ClientStompFrame frame = this.conn.receiveFrame(10000L);
        Assert.assertEquals((Object)"MESSAGE", (Object)frame.getCommand());
        Assert.assertEquals((Object)(StompTest.getTopicPrefix() + StompTest.getTopicName()), (Object)frame.getHeader("destination"));
        Assert.assertEquals((Object)RoutingType.MULTICAST.toString(), (Object)frame.getHeader("destination-type"));
        Assert.assertTrue((frame.getHeader(org.apache.activemq.artemis.api.core.Message.HDR_ROUTING_TYPE.toString()) == null ? (byte)1 : 0) != 0);
        Assert.assertEquals((Object)this.getName(), (Object)frame.getBody());
        this.conn.disconnect();
    }

    @Test
    public void testSubscribeWithAutoAckAndBytesMessage() throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        StompTest.subscribe(this.conn, null, "auto");
        byte[] payload = new byte[]{1, 2, 3, 4, 5};
        this.sendJmsMessage(payload, (Destination)this.queue);
        ClientStompFrame frame = this.conn.receiveFrame(10000L);
        Assert.assertEquals((Object)"MESSAGE", (Object)frame.getCommand());
        Pattern cl = Pattern.compile("content-length:\\s*(\\d+)", 2);
        Matcher cl_matcher = cl.matcher(frame.toString());
        Assert.assertTrue((boolean)cl_matcher.find());
        Assert.assertEquals((Object)"5", (Object)cl_matcher.group(1));
        Assert.assertFalse((boolean)Pattern.compile("type:\\s*null", 2).matcher(frame.toString()).find());
        Assert.assertTrue((frame.getBody().toString().indexOf(new String(payload)) > -1 ? (byte)1 : 0) != 0);
        this.conn.disconnect();
    }

    @Test
    public void testSubscribeWithMessageSentWithProperties() throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        StompTest.subscribe(this.conn, null, "auto");
        MessageProducer producer = this.session.createProducer((Destination)this.queue);
        BytesMessage message = this.session.createBytesMessage();
        message.setStringProperty("S", "value");
        message.setBooleanProperty("n", false);
        message.setByteProperty("byte", (byte)9);
        message.setDoubleProperty("d", 2.0);
        message.setFloatProperty("f", 6.0f);
        message.setIntProperty("i", 10);
        message.setLongProperty("l", 121L);
        message.setShortProperty("s", (short)12);
        message.writeBytes("Hello World".getBytes(StandardCharsets.UTF_8));
        producer.send((Message)message);
        ClientStompFrame frame = this.conn.receiveFrame(10000L);
        Assert.assertNotNull((Object)frame);
        Assert.assertEquals((Object)"MESSAGE", (Object)frame.getCommand());
        Assert.assertEquals((Object)"value", (Object)frame.getHeader("S"));
        Assert.assertEquals((Object)"false", (Object)frame.getHeader("n"));
        Assert.assertEquals((Object)"9", (Object)frame.getHeader("byte"));
        Assert.assertEquals((Object)"2.0", (Object)frame.getHeader("d"));
        Assert.assertEquals((Object)"6.0", (Object)frame.getHeader("f"));
        Assert.assertEquals((Object)"10", (Object)frame.getHeader("i"));
        Assert.assertEquals((Object)"121", (Object)frame.getHeader("l"));
        Assert.assertEquals((Object)"12", (Object)frame.getHeader("s"));
        Assert.assertEquals((Object)"Hello World", (Object)frame.getBody());
        this.conn.disconnect();
    }

    @Test
    public void testSubscribeWithID() throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        StompTest.subscribe(this.conn, "mysubid", "auto");
        this.sendJmsMessage(this.getName());
        ClientStompFrame frame = this.conn.receiveFrame(10000L);
        Assert.assertEquals((Object)"MESSAGE", (Object)frame.getCommand());
        Assert.assertEquals((Object)(StompTest.getQueuePrefix() + StompTest.getQueueName()), (Object)frame.getHeader("destination"));
        Assert.assertEquals((Object)"mysubid", (Object)frame.getHeader("subscription"));
        Assert.assertEquals((Object)this.getName(), (Object)frame.getBody());
        this.conn.disconnect();
    }

    @Test
    public void testBodyWithUTF8() throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        StompTest.subscribe(this.conn, null, "auto");
        String text = "A\u00ea\u00f1\u00fcC";
        log.debug((Object)text);
        this.sendJmsMessage(text);
        ClientStompFrame frame = this.conn.receiveFrame(10000L);
        log.debug((Object)frame);
        Assert.assertEquals((Object)"MESSAGE", (Object)frame.getCommand());
        Assert.assertEquals((Object)(StompTest.getQueuePrefix() + StompTest.getQueueName()), (Object)frame.getHeader("destination"));
        Assert.assertEquals((Object)text, (Object)frame.getBody());
        this.conn.disconnect();
    }

    @Test
    public void testMessagesAreInOrder() throws Exception {
        ClientStompFrame frame;
        int i;
        int ctr = 10;
        String[] data = new String[ctr];
        this.conn.connect(this.defUser, this.defPass);
        StompTest.subscribe(this.conn, null, "auto");
        for (i = 0; i < ctr; ++i) {
            data[i] = this.getName() + i;
            this.sendJmsMessage(data[i]);
        }
        for (i = 0; i < ctr; ++i) {
            frame = this.conn.receiveFrame(1000L);
            Assert.assertTrue((String)"Message not in order", (boolean)frame.getBody().equals(data[i]));
        }
        Thread.sleep(200L);
        for (i = 0; i < ctr; ++i) {
            data[i] = this.getName() + ":" + "second:" + i;
            this.sendJmsMessage(data[i]);
        }
        for (i = 0; i < ctr; ++i) {
            frame = this.conn.receiveFrame(1000L);
            Assert.assertTrue((String)"Message not in order", (boolean)frame.getBody().equals(data[i]));
        }
        this.conn.disconnect();
    }

    @Test
    public void testSubscribeWithAutoAckAndSelector() throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        StompTest.subscribe(this.conn, null, "auto", null, "foo = 'zzz'");
        this.sendJmsMessage("Ignored message", "foo", "1234");
        this.sendJmsMessage("Real message", "foo", "zzz");
        ClientStompFrame frame = this.conn.receiveFrame(10000L);
        Assert.assertEquals((Object)"MESSAGE", (Object)frame.getCommand());
        Assert.assertTrue((String)("Should have received the real message but got: " + frame), (boolean)frame.getBody().equals("Real message"));
        this.conn.disconnect();
    }

    @Test
    public void testSubscribeWithAutoAckAndHyphenatedSelector() throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        StompTest.subscribe(this.conn, null, "auto", null, "hyphenated_props:foo-bar = 'zzz'");
        ServerLocator serverLocator = this.addServerLocator(ActiveMQClient.createServerLocator((String)"vm://0"));
        ClientSessionFactory clientSessionFactory = serverLocator.createSessionFactory();
        ClientSession clientSession = clientSessionFactory.createSession(true, true);
        ClientProducer producer = clientSession.createProducer(StompTest.getQueuePrefix() + StompTest.getQueueName());
        ClientMessage ignoredMessage = clientSession.createMessage(false);
        ignoredMessage.putStringProperty("foo-bar", "1234");
        ignoredMessage.getBodyBuffer().writeNullableSimpleString(SimpleString.toSimpleString((String)"Ignored message"));
        ClientMessage realMessage = clientSession.createMessage(false);
        realMessage.putStringProperty("foo-bar", "zzz");
        realMessage.getBodyBuffer().writeNullableSimpleString(SimpleString.toSimpleString((String)"Real message"));
        producer.send((org.apache.activemq.artemis.api.core.Message)ignoredMessage);
        producer.send((org.apache.activemq.artemis.api.core.Message)realMessage);
        ClientStompFrame frame = this.conn.receiveFrame(10000L);
        Assert.assertEquals((Object)"MESSAGE", (Object)frame.getCommand());
        Assert.assertTrue((String)("Should have received the real message but got: " + frame), (boolean)frame.getBody().equals("Real message"));
        this.conn.disconnect();
    }

    @Test
    public void testSubscribeWithAutoAckAndXpathSelector() throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        StompTest.subscribe(this.conn, null, "auto", null, "XPATH 'root/a'");
        this.sendJmsMessage("<root><b key='first' num='1'/><c key='second' num='2'>c</c></root>");
        this.sendJmsMessage("<root><a key='first' num='1'/><b key='second' num='2'>b</b></root>");
        ClientStompFrame frame = this.conn.receiveFrame(10000L);
        Assert.assertEquals((Object)"MESSAGE", (Object)frame.getCommand());
        Assert.assertTrue((String)("Should have received the real message but got: " + frame), (boolean)frame.getBody().equals("<root><a key='first' num='1'/><b key='second' num='2'>b</b></root>"));
        this.conn.disconnect();
    }

    @Test
    public void testSubscribeWithClientAck() throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        StompTest.subscribe(this.conn, null, "client");
        this.sendJmsMessage(this.getName());
        ClientStompFrame frame = this.conn.receiveFrame(10000L);
        Assert.assertEquals((Object)"MESSAGE", (Object)frame.getCommand());
        Assert.assertNotNull((Object)frame.getHeader("message-id"));
        StompTest.ack(this.conn, null, frame);
        this.conn.disconnect();
        MessageConsumer consumer = this.session.createConsumer((Destination)this.queue);
        Message message = consumer.receive(100L);
        Assert.assertNull((Object)message);
    }

    @Test
    public void testRedeliveryWithClientAck() throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        StompTest.subscribe(this.conn, null, "client");
        this.sendJmsMessage(this.getName());
        ClientStompFrame frame = this.conn.receiveFrame(10000L);
        Assert.assertEquals((Object)"MESSAGE", (Object)frame.getCommand());
        this.conn.disconnect();
        MessageConsumer consumer = this.session.createConsumer((Destination)this.queue);
        Message message = consumer.receive(1000L);
        Assert.assertNotNull((Object)message);
        Assert.assertTrue((boolean)message.getJMSRedelivered());
    }

    @Test
    public void testSubscribeWithClientAckThenConsumingAgainWithAutoAckWithNoDisconnectFrame() throws Exception {
        this.assertSubscribeWithClientAckThenConsumeWithAutoAck(false);
    }

    @Test
    public void testSubscribeWithClientAckThenConsumingAgainWithAutoAckWithExplicitDisconnect() throws Exception {
        this.assertSubscribeWithClientAckThenConsumeWithAutoAck(true);
    }

    protected void assertSubscribeWithClientAckThenConsumeWithAutoAck(boolean sendDisconnect) throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        StompTest.subscribe(this.conn, null, "client");
        this.sendJmsMessage(this.getName());
        ClientStompFrame frame = this.conn.receiveFrame(10000L);
        Assert.assertEquals((Object)"MESSAGE", (Object)frame.getCommand());
        log.debug((Object)"Reconnecting!");
        if (sendDisconnect) {
            this.conn.disconnect();
            this.conn.destroy();
            this.conn = StompClientConnectionFactory.createClientConnection(this.uri);
        } else {
            this.conn.destroy();
            this.conn = StompClientConnectionFactory.createClientConnection(this.uri);
        }
        this.conn.connect(this.defUser, this.defPass);
        StompTest.subscribe(this.conn, null);
        frame = this.conn.receiveFrame(10000L);
        Assert.assertEquals((Object)"MESSAGE", (Object)frame.getCommand());
        this.conn.disconnect();
        this.conn.destroy();
        this.conn = StompClientConnectionFactory.createClientConnection(this.uri);
        this.conn.connect(this.defUser, this.defPass);
        StompTest.subscribe(this.conn, null, "auto", null, true);
        this.sendJmsMessage("shouldBeNextMessage");
        frame = this.conn.receiveFrame(10000L);
        Assert.assertEquals((Object)"MESSAGE", (Object)frame.getCommand());
        Assert.assertEquals((Object)"shouldBeNextMessage", (Object)frame.getBody());
    }

    @Test
    public void testUnsubscribe() throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        StompTest.subscribe(this.conn, null, "auto");
        this.sendJmsMessage("first message");
        ClientStompFrame frame = this.conn.receiveFrame(10000L);
        Assert.assertEquals((Object)"MESSAGE", (Object)frame.getCommand());
        StompTest.unsubscribe(this.conn, null, StompTest.getQueuePrefix() + StompTest.getQueueName(), true, false);
        this.sendJmsMessage("second message");
        frame = this.conn.receiveFrame(100L);
        log.debug((Object)("Received frame: " + frame));
        Assert.assertNull((String)"No message should have been received since subscription was removed", (Object)frame);
    }

    @Test
    public void testUnsubscribeWithID() throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        StompTest.subscribe(this.conn, "mysubid", "auto");
        this.sendJmsMessage("first message");
        ClientStompFrame frame = this.conn.receiveFrame(10000L);
        Assert.assertEquals((Object)"MESSAGE", (Object)frame.getCommand());
        StompTest.unsubscribe(this.conn, "mysubid", null, true, false);
        this.sendJmsMessage("second message");
        frame = this.conn.receiveFrame(100L);
        log.debug((Object)("Received frame: " + frame));
        Assert.assertNull((String)"No message should have been received since subscription was removed", (Object)frame);
    }

    @Test
    public void testTransactionCommit() throws Exception {
        MessageConsumer consumer = this.session.createConsumer((Destination)this.queue);
        this.conn.connect(this.defUser, this.defPass);
        StompTest.beginTransaction(this.conn, "tx1");
        StompTest.send(this.conn, StompTest.getQueuePrefix() + StompTest.getQueueName(), null, "Hello World", true, null, "tx1");
        StompTest.assertNull((Object)consumer.receive(100L));
        StompTest.commitTransaction(this.conn, "tx1", true);
        Message message = consumer.receive(1000L);
        Assert.assertNotNull((String)"Should have received a message", (Object)message);
    }

    @Test
    public void testSuccessiveTransactionsWithSameID() throws Exception {
        MessageConsumer consumer = this.session.createConsumer((Destination)this.queue);
        this.conn.connect(this.defUser, this.defPass);
        StompTest.beginTransaction(this.conn, "tx1");
        StompTest.send(this.conn, StompTest.getQueuePrefix() + StompTest.getQueueName(), null, "Hello World", true, null, "tx1");
        StompTest.commitTransaction(this.conn, "tx1");
        Message message = consumer.receive(1000L);
        Assert.assertNotNull((String)"Should have received a message", (Object)message);
        StompTest.beginTransaction(this.conn, "tx1");
        StompTest.send(this.conn, StompTest.getQueuePrefix() + StompTest.getQueueName(), null, "Hello World", true, null, "tx1");
        StompTest.commitTransaction(this.conn, "tx1");
        message = consumer.receive(1000L);
        Assert.assertNotNull((String)"Should have received a message", (Object)message);
    }

    @Test
    public void testBeginSameTransactionTwice() throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        StompTest.beginTransaction(this.conn, "tx1");
        StompTest.beginTransaction(this.conn, "tx1");
        ClientStompFrame frame = this.conn.receiveFrame(1000L);
        Assert.assertEquals((Object)"ERROR", (Object)frame.getCommand());
    }

    @Test
    public void testTransactionRollback() throws Exception {
        MessageConsumer consumer = this.session.createConsumer((Destination)this.queue);
        String txId = "tx1";
        this.conn.connect(this.defUser, this.defPass);
        StompTest.beginTransaction(this.conn, txId);
        StompTest.send(this.conn, StompTest.getQueuePrefix() + StompTest.getQueueName(), null, "first message", true, null, txId);
        StompTest.abortTransaction(this.conn, txId);
        StompTest.beginTransaction(this.conn, txId);
        StompTest.send(this.conn, StompTest.getQueuePrefix() + StompTest.getQueueName(), null, "second message", true, null, txId);
        StompTest.commitTransaction(this.conn, txId);
        TextMessage message = (TextMessage)consumer.receive(1000L);
        Assert.assertNotNull((Object)message);
        Assert.assertEquals((Object)"second message", (Object)message.getText());
    }

    @Test
    public void testSubscribeToTopic() throws Exception {
        int baselineQueueCount = this.server.getActiveMQServerControl().getQueueNames().length;
        this.conn.connect(this.defUser, this.defPass);
        StompTest.subscribeTopic(this.conn, null, null, null, true);
        Wait.assertTrue((String)"Subscription queue should be created here", () -> {
            int length = this.server.getActiveMQServerControl().getQueueNames().length;
            if (length - baselineQueueCount == 1) {
                return true;
            }
            log.debug((Object)("Queue count: " + (length - baselineQueueCount)));
            return false;
        });
        this.sendJmsMessage(this.getName(), (Destination)this.topic);
        ClientStompFrame frame = this.conn.receiveFrame(1000L);
        Assert.assertEquals((Object)"MESSAGE", (Object)frame.getCommand());
        Assert.assertEquals((Object)(StompTest.getTopicPrefix() + StompTest.getTopicName()), (Object)frame.getHeader("destination"));
        Assert.assertEquals((Object)this.getName(), (Object)frame.getBody());
        StompTest.unsubscribe(this.conn, null, StompTest.getTopicPrefix() + StompTest.getTopicName(), true, false);
        this.sendJmsMessage(this.getName(), (Destination)this.topic);
        frame = this.conn.receiveFrame(100L);
        log.debug((Object)("Received frame: " + frame));
        Assert.assertNull((String)"No message should have been received since subscription was removed", (Object)frame);
        StompTest.assertEquals((String)"Subscription queue should be deleted", (long)0L, (long)(this.server.getActiveMQServerControl().getQueueNames().length - baselineQueueCount));
        this.conn.disconnect();
    }

    @Test
    public void testSubscribeToQueue() throws Exception {
        int baselineQueueCount = this.server.getActiveMQServerControl().getQueueNames().length;
        this.conn.connect(this.defUser, this.defPass);
        StompTest.subscribe(this.conn, null, null, null, true);
        Wait.assertFalse((String)"Queue should not be created here", () -> this.server.getActiveMQServerControl().getQueueNames().length - baselineQueueCount == 1);
        this.sendJmsMessage(this.getName(), (Destination)this.queue);
        ClientStompFrame frame = this.conn.receiveFrame(1000L);
        Assert.assertEquals((Object)"MESSAGE", (Object)frame.getCommand());
        Assert.assertEquals((Object)(StompTest.getQueuePrefix() + StompTest.getQueueName()), (Object)frame.getHeader("destination"));
        Assert.assertEquals((Object)this.getName(), (Object)frame.getBody());
        StompTest.unsubscribe(this.conn, null, StompTest.getQueuePrefix() + StompTest.getQueueName(), true, false);
        this.sendJmsMessage(this.getName(), (Destination)this.queue);
        frame = this.conn.receiveFrame(100L);
        log.debug((Object)("Received frame: " + frame));
        Assert.assertNull((String)"No message should have been received since subscription was removed", (Object)frame);
        StompTest.assertEquals((String)"Subscription queue should not be deleted", (long)baselineQueueCount, (long)this.server.getActiveMQServerControl().getQueueNames().length);
        this.conn.disconnect();
    }

    @Test
    public void testSubscribeToNonExistentQueue() throws Exception {
        String nonExistentQueue = RandomUtil.randomString();
        this.conn.connect(this.defUser, this.defPass);
        StompTest.subscribe(this.conn, null, null, null, null, StompTest.getQueuePrefix() + nonExistentQueue, true);
        this.sendJmsMessage(this.getName(), (Destination)ActiveMQJMSClient.createQueue((String)nonExistentQueue));
        ClientStompFrame frame = this.conn.receiveFrame(1000L);
        Assert.assertEquals((Object)"MESSAGE", (Object)frame.getCommand());
        Assert.assertEquals((Object)(StompTest.getQueuePrefix() + nonExistentQueue), (Object)frame.getHeader("destination"));
        Assert.assertEquals((Object)this.getName(), (Object)frame.getBody());
        StompTest.assertNotNull((Object)this.server.getPostOffice().getBinding(new SimpleString(nonExistentQueue)));
        Queue subscription = ((LocalQueueBinding)this.server.getPostOffice().getBinding(new SimpleString(nonExistentQueue))).getQueue();
        Wait.assertEquals((long)0L, () -> ((Queue)subscription).getMessageCount());
        StompTest.unsubscribe(this.conn, null, StompTest.getQueuePrefix() + nonExistentQueue, true, false);
        Wait.assertTrue(() -> this.server.getPostOffice().getBinding(new SimpleString(nonExistentQueue)) == null);
        this.sendJmsMessage(this.getName(), (Destination)ActiveMQJMSClient.createQueue((String)nonExistentQueue));
        frame = this.conn.receiveFrame(100L);
        log.debug((Object)("Received frame: " + frame));
        Assert.assertNull((String)"No message should have been received since subscription was removed", (Object)frame);
        this.conn.disconnect();
    }

    @Test
    public void testDurableSubscriberWithReconnection() throws Exception {
        this.conn.connect(this.defUser, this.defPass, "myclientid");
        StompTest.subscribeTopic(this.conn, null, null, this.getName());
        this.conn.disconnect();
        Thread.sleep(500L);
        this.sendJmsMessage(this.getName(), (Destination)this.topic);
        this.conn.destroy();
        this.conn = StompClientConnectionFactory.createClientConnection(this.uri);
        this.conn.connect(this.defUser, this.defPass, "myclientid");
        StompTest.subscribeTopic(this.conn, null, null, this.getName());
        ClientStompFrame frame = this.conn.receiveFrame(3000L);
        StompTest.assertNotNull((String)"Should have received a message from the durable subscription", (Object)frame);
        Assert.assertEquals((Object)"MESSAGE", (Object)frame.getCommand());
        Assert.assertEquals((Object)(StompTest.getTopicPrefix() + StompTest.getTopicName()), (Object)frame.getHeader("destination"));
        Assert.assertEquals((Object)this.getName(), (Object)frame.getBody());
        StompTest.unsubscribe(this.conn, null, StompTest.getTopicPrefix() + StompTest.getTopicName(), true, true);
        this.conn.disconnect();
    }

    @Test
    public void testDurableSubscriberWithReconnectionLegacy() throws Exception {
        this.conn.connect(this.defUser, this.defPass, "myclientid");
        StompTest.subscribeTopicLegacyActiveMQ(this.conn, null, null, this.getName(), true, false);
        this.conn.disconnect();
        Thread.sleep(500L);
        this.sendJmsMessage(this.getName(), (Destination)this.topic);
        this.conn.destroy();
        this.conn = StompClientConnectionFactory.createClientConnection(this.uri);
        this.conn.connect(this.defUser, this.defPass, "myclientid");
        StompTest.subscribeTopicLegacyActiveMQ(this.conn, null, null, this.getName(), true, false);
        ClientStompFrame frame = this.conn.receiveFrame(3000L);
        StompTest.assertNotNull((String)"Should have received a message from the durable subscription", (Object)frame);
        Assert.assertEquals((Object)"MESSAGE", (Object)frame.getCommand());
        Assert.assertEquals((Object)(StompTest.getTopicPrefix() + StompTest.getTopicName()), (Object)frame.getHeader("destination"));
        Assert.assertEquals((Object)this.getName(), (Object)frame.getBody());
        StompTest.unsubscribeLegacyActiveMQ(this.conn, null, StompTest.getTopicPrefix() + StompTest.getTopicName(), true, true);
        this.conn.disconnect();
    }

    @Test
    public void testDurableSubscriber() throws Exception {
        this.conn.connect(this.defUser, this.defPass, "myclientid");
        StompTest.subscribeTopic(this.conn, null, null, this.getName(), true);
        ClientStompFrame response = StompTest.subscribeTopic(this.conn, null, null, this.getName(), true);
        Assert.assertEquals((Object)"ERROR", (Object)response.getCommand());
        this.conn.disconnect();
    }

    @Test
    public void testDurableSubscriberLegacySubscriptionHeader() throws Exception {
        this.conn.connect(this.defUser, this.defPass, "myclientid");
        StompTest.subscribeTopicLegacyActiveMQ(this.conn, null, null, this.getName(), true, false);
        ClientStompFrame response = StompTest.subscribeTopicLegacyActiveMQ(this.conn, null, null, this.getName(), true, false);
        Assert.assertEquals((Object)"ERROR", (Object)response.getCommand());
        this.conn.disconnect();
    }

    @Test
    public void testDurableUnSubscribe() throws Exception {
        this.conn.connect(this.defUser, this.defPass, "myclientid");
        StompTest.subscribeTopic(this.conn, null, null, this.getName(), true);
        this.conn.disconnect();
        Thread.sleep(500L);
        StompTest.assertNotNull((Object)this.server.locateQueue(SimpleString.toSimpleString((String)("myclientid." + this.getName()))));
        this.conn.destroy();
        this.conn = StompClientConnectionFactory.createClientConnection(this.uri);
        this.conn.connect(this.defUser, this.defPass, "myclientid");
        StompTest.unsubscribe(this.conn, this.getName(), StompTest.getTopicPrefix() + StompTest.getTopicName(), false, true);
        this.conn.disconnect();
        Thread.sleep(500L);
        Wait.assertTrue(() -> this.server.locateQueue(SimpleString.toSimpleString((String)("myclientid." + this.getName()))) == null);
    }

    @Test
    public void testDurableUnSubscribeWithoutDurableSubName() throws Exception {
        this.server.getConfiguration().getWildcardConfiguration().setDelimiter('/');
        this.server.getAddressSettingsRepository().addMatch("/topic/#", (Object)new AddressSettings().setDefaultAddressRoutingType(RoutingType.MULTICAST).setDefaultQueueRoutingType(RoutingType.MULTICAST));
        this.conn.connect(this.defUser, this.defPass, "myclientid");
        String subId = UUID.randomUUID().toString();
        String durableSubName = UUID.randomUUID().toString();
        String receipt = UUID.randomUUID().toString();
        ClientStompFrame frame = this.conn.createFrame("SUBSCRIBE").addHeader("destination", "/topic/test.foo").addHeader("id", subId).addHeader("ack", "client-individual").addHeader("durable-subscription-name", durableSubName).addHeader("receipt", receipt);
        frame = this.conn.sendFrame(frame);
        StompTest.assertEquals((Object)receipt, (Object)frame.getHeader("receipt-id"));
        Wait.assertTrue(() -> this.server.locateQueue(SimpleString.toSimpleString((String)("myclientid." + durableSubName))) != null);
        receipt = UUID.randomUUID().toString();
        frame = this.conn.createFrame("UNSUBSCRIBE").addHeader("id", subId).addHeader("receipt", receipt);
        frame = this.conn.sendFrame(frame);
        StompTest.assertEquals((Object)receipt, (Object)frame.getHeader("receipt-id"));
        this.conn.disconnect();
        Wait.assertTrue(() -> this.server.locateQueue(SimpleString.toSimpleString((String)("myclientid." + durableSubName))) != null);
    }

    @Test
    public void testDurableUnSubscribeLegacySubscriptionHeader() throws Exception {
        this.conn.connect(this.defUser, this.defPass, "myclientid");
        StompTest.subscribeTopicLegacyActiveMQ(this.conn, null, null, this.getName(), true, false);
        this.conn.disconnect();
        Thread.sleep(500L);
        StompTest.assertNotNull((Object)this.server.locateQueue(SimpleString.toSimpleString((String)("myclientid." + this.getName()))));
        this.conn.destroy();
        this.conn = StompClientConnectionFactory.createClientConnection(this.uri);
        this.conn.connect(this.defUser, this.defPass, "myclientid");
        StompTest.unsubscribeLegacyActiveMQ(this.conn, this.getName(), StompTest.getTopicPrefix() + StompTest.getTopicName(), false, true);
        this.conn.disconnect();
        Thread.sleep(500L);
        Wait.assertTrue(() -> this.server.locateQueue(SimpleString.toSimpleString((String)("myclientid." + this.getName()))) == null);
    }

    @Test
    public void testSubscribeToTopicWithNoLocal() throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        StompTest.subscribeTopic(this.conn, null, null, null, true, true);
        StompTest.send(this.conn, StompTest.getTopicPrefix() + StompTest.getTopicName(), null, "Hello World");
        ClientStompFrame frame = this.conn.receiveFrame(100L);
        log.debug((Object)("Received frame: " + frame));
        Assert.assertNull((String)"No message should have been received since subscription was removed", (Object)frame);
        this.sendJmsMessage(this.getName(), (Destination)this.topic);
        frame = this.conn.receiveFrame(10000L);
        Assert.assertEquals((Object)"MESSAGE", (Object)frame.getCommand());
        Assert.assertEquals((Object)(StompTest.getTopicPrefix() + StompTest.getTopicName()), (Object)frame.getHeader("destination"));
        Assert.assertEquals((Object)this.getName(), (Object)frame.getBody());
        this.conn.disconnect();
    }

    @Test
    public void testTopicExistsAfterNoUnsubscribeDisconnect() throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        StompTest.subscribeTopic(this.conn, null, null, null, true);
        this.conn.disconnect();
        Thread.sleep(500L);
        this.conn.destroy();
        this.conn = StompClientConnectionFactory.createClientConnection(this.uri);
        this.conn.connect(this.defUser, this.defPass);
        ClientStompFrame response = StompTest.send(this.conn, StompTest.getTopicPrefix() + StompTest.getTopicName(), null, "Hello World", true);
        StompTest.assertEquals((Object)"RECEIPT", (Object)response.getCommand());
        ClientStompFrame frame = this.conn.receiveFrame(100L);
        log.debug((Object)("Received frame: " + frame));
        Assert.assertNull((Object)frame);
        this.conn.disconnect();
    }

    @Test
    public void testClientAckNotPartOfTransaction() throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        StompTest.subscribe(this.conn, null, "client");
        this.sendJmsMessage(this.getName());
        ClientStompFrame frame = this.conn.receiveFrame(10000L);
        Assert.assertEquals((Object)"MESSAGE", (Object)frame.getCommand());
        Assert.assertEquals((Object)(StompTest.getQueuePrefix() + StompTest.getQueueName()), (Object)frame.getHeader("destination"));
        String messageID = frame.getHeader("message-id");
        Assert.assertNotNull((Object)messageID);
        Assert.assertEquals((Object)this.getName(), (Object)frame.getBody());
        StompTest.beginTransaction(this.conn, "tx1");
        StompTest.ack(this.conn, null, messageID, "tx1");
        StompTest.abortTransaction(this.conn, "tx1");
        frame = this.conn.receiveFrame(100L);
        Assert.assertNull((String)"No message should have been received as the message was acked even though the transaction has been aborted", (Object)frame);
        StompTest.unsubscribe(this.conn, null, StompTest.getQueuePrefix() + StompTest.getQueueName(), false, false);
        this.conn.disconnect();
    }

    @Test
    public void testMultiProtocolConsumers() throws Exception {
        int TIME_OUT = 2000;
        int NEGATIVE_TIME_OUT = 100;
        int count = 1000;
        MessageConsumer consumer1 = this.session.createConsumer((Destination)this.topic);
        MessageConsumer consumer2 = this.session.createConsumer((Destination)this.topic);
        this.conn.connect(this.defUser, this.defPass);
        StompTest.subscribeTopic(this.conn, null, null, null, true);
        MessageProducer producer = this.session.createProducer((Destination)this.topic);
        TextMessage message = this.session.createTextMessage(this.getName());
        for (int i = 1; i <= count; ++i) {
            producer.send((Message)message);
            Assert.assertNotNull((Object)consumer1.receive(2000L));
            Assert.assertNotNull((Object)consumer2.receive(2000L));
            ClientStompFrame frame = this.conn.receiveFrame(2000L);
            Assert.assertEquals((Object)"MESSAGE", (Object)frame.getCommand());
            Assert.assertEquals((Object)(StompTest.getTopicPrefix() + StompTest.getTopicName()), (Object)frame.getHeader("destination"));
            Assert.assertEquals((Object)this.getName(), (Object)frame.getBody());
        }
        consumer1.close();
        consumer2.close();
        StompTest.unsubscribe(this.conn, null, StompTest.getTopicPrefix() + StompTest.getTopicName(), true, false);
        this.sendJmsMessage(this.getName(), (Destination)this.topic);
        ClientStompFrame frame = this.conn.receiveFrame(100L);
        log.debug((Object)("Received frame: " + frame));
        Assert.assertNull((String)"No message should have been received since subscription was removed", (Object)frame);
        this.conn.disconnect();
    }

    @Test
    public void testUnexpectedAck() throws Exception {
        String messageID = "888888";
        this.conn.connect(this.defUser, this.defPass);
        StompTest.ack(this.conn, null, messageID, null);
        ClientStompFrame frame = this.conn.receiveFrame(1000L);
        StompTest.assertNotNull((Object)frame);
        StompTest.assertEquals((Object)"ERROR", (Object)frame.getCommand());
        this.conn.disconnect();
    }

    @Test
    public void testDotAnycastPrefixOnSend() throws Exception {
        this.testPrefix("jms.queue.", RoutingType.ANYCAST, true);
    }

    @Test
    public void testDotMulticastPrefixOnSend() throws Exception {
        this.testPrefix("jms.topic.", RoutingType.MULTICAST, true);
    }

    @Test
    public void testDotAnycastPrefixOnSubscribe() throws Exception {
        this.testPrefix("jms.queue.", RoutingType.ANYCAST, false);
    }

    @Test
    public void testDotMulticastPrefixOnSubscribe() throws Exception {
        this.testPrefix("jms.topic.", RoutingType.MULTICAST, false);
    }

    @Test
    public void testSlashAnycastPrefixOnSend() throws Exception {
        this.testPrefix("/queue/", RoutingType.ANYCAST, true);
    }

    @Test
    public void testSlashMulticastPrefixOnSend() throws Exception {
        this.testPrefix("/topic/", RoutingType.MULTICAST, true);
    }

    @Test
    public void testSlashAnycastPrefixOnSubscribe() throws Exception {
        this.testPrefix("/queue/", RoutingType.ANYCAST, false);
    }

    @Test
    public void testSlashMulticastPrefixOnSubscribe() throws Exception {
        this.testPrefix("/topic/", RoutingType.MULTICAST, false);
    }

    public void testPrefix(String prefix, RoutingType routingType, boolean send) throws Exception {
        int port = 61614;
        URI uri = StompTest.createStompClientUri(this.scheme, this.hostname, port);
        String ADDRESS = UUID.randomUUID().toString();
        String PREFIXED_ADDRESS = prefix + ADDRESS;
        String param = routingType.toString();
        String urlParam = param.toLowerCase() + "Prefix";
        this.server.getRemotingService().createAcceptor("test", "tcp://" + this.hostname + ":" + port + "?protocols=" + "STOMP" + "&" + urlParam + "=" + prefix).start();
        StompClientConnection conn = StompClientConnectionFactory.createClientConnection(uri);
        conn.connect(this.defUser, this.defPass);
        if (send) {
            StompTest.send(conn, PREFIXED_ADDRESS, null, "Hello World", true);
        } else {
            String uuid = UUID.randomUUID().toString();
            ClientStompFrame frame = conn.createFrame("SUBSCRIBE").addHeader("destination", PREFIXED_ADDRESS).addHeader("receipt", uuid);
            frame = conn.sendFrame(frame);
            StompTest.assertEquals((Object)uuid, (Object)frame.getHeader("receipt-id"));
        }
        AddressInfo addressInfo = this.server.getAddressInfo(SimpleString.toSimpleString((String)ADDRESS));
        StompTest.assertNotNull((String)("No address was created with the name " + ADDRESS), (Object)addressInfo);
        HashSet<RoutingType> routingTypes = new HashSet<RoutingType>();
        routingTypes.add(RoutingType.valueOf((String)param));
        StompTest.assertEquals(routingTypes, (Object)addressInfo.getRoutingTypes());
        conn.disconnect();
    }

    @Test
    public void testPrefixedAutoCreatedAnycastAndMulticastWithSameName() throws Exception {
        int port = 61614;
        URI uri = StompTest.createStompClientUri(this.scheme, this.hostname, port);
        String ADDRESS = UUID.randomUUID().toString();
        this.server.getRemotingService().createAcceptor("test", "tcp://" + this.hostname + ":" + port + "?protocols=STOMP&anycastPrefix=/queue/&multicastPrefix=/topic/").start();
        StompClientConnection conn = StompClientConnectionFactory.createClientConnection(uri);
        conn.connect(this.defUser, this.defPass);
        String uuid = UUID.randomUUID().toString();
        ClientStompFrame frame = conn.createFrame("SUBSCRIBE").addHeader("destination", "/queue/" + ADDRESS).addHeader("receipt", uuid);
        frame = conn.sendFrame(frame);
        StompTest.assertEquals((Object)uuid, (Object)frame.getHeader("receipt-id"));
        AddressInfo addressInfo = this.server.getAddressInfo(SimpleString.toSimpleString((String)ADDRESS));
        StompTest.assertNotNull((String)("No address was created with the name " + ADDRESS), (Object)addressInfo);
        StompTest.assertTrue((boolean)addressInfo.getRoutingTypes().contains(RoutingType.ANYCAST));
        StompTest.assertFalse((boolean)addressInfo.getRoutingTypes().contains(RoutingType.MULTICAST));
        StompTest.assertNotNull((Object)this.server.locateQueue(SimpleString.toSimpleString((String)ADDRESS)));
        frame = StompTest.send(conn, "/topic/" + ADDRESS, null, "Hello World 1", true);
        StompTest.assertFalse((boolean)frame.getCommand().equals("ERROR"));
        addressInfo = this.server.getAddressInfo(SimpleString.toSimpleString((String)ADDRESS));
        StompTest.assertTrue((boolean)addressInfo.getRoutingTypes().contains(RoutingType.ANYCAST));
        StompTest.assertTrue((boolean)addressInfo.getRoutingTypes().contains(RoutingType.MULTICAST));
        frame = conn.receiveFrame(100L);
        Assert.assertNull((Object)frame);
        frame = StompTest.send(conn, "/queue/" + ADDRESS, null, "Hello World 2", true);
        StompTest.assertFalse((boolean)frame.getCommand().equals("ERROR"));
        frame = conn.receiveFrame(1000L);
        Assert.assertEquals((Object)"MESSAGE", (Object)frame.getCommand());
        Assert.assertEquals((Object)"Hello World 2", (Object)frame.getBody());
        Assert.assertEquals((Object)RoutingType.ANYCAST.toString(), (Object)frame.getHeader("destination-type"));
        Assert.assertEquals((Object)("/queue/" + ADDRESS), (Object)frame.getHeader("destination"));
        frame = conn.receiveFrame(100L);
        Assert.assertNull((Object)frame);
        StompTest.unsubscribe(conn, null, "/queue/" + ADDRESS, true, false);
        Wait.assertTrue(() -> this.server.getAddressInfo(SimpleString.toSimpleString((String)ADDRESS)) == null);
        uuid = UUID.randomUUID().toString();
        frame = conn.createFrame("SUBSCRIBE").addHeader("destination", "/topic/" + ADDRESS).addHeader("receipt", uuid);
        frame = conn.sendFrame(frame);
        StompTest.assertEquals((Object)uuid, (Object)frame.getHeader("receipt-id"));
        frame = StompTest.send(conn, "/topic/" + ADDRESS, null, "Hello World 3", true);
        StompTest.assertFalse((boolean)frame.getCommand().equals("ERROR"));
        frame = conn.receiveFrame(1000L);
        Assert.assertEquals((Object)"MESSAGE", (Object)frame.getCommand());
        Assert.assertEquals((Object)"Hello World 3", (Object)frame.getBody());
        Assert.assertEquals((Object)RoutingType.MULTICAST.toString(), (Object)frame.getHeader("destination-type"));
        Assert.assertEquals((Object)("/topic/" + ADDRESS), (Object)frame.getHeader("destination"));
        frame = conn.receiveFrame(100L);
        Assert.assertNull((Object)frame);
        StompTest.unsubscribe(conn, null, "/topic/" + ADDRESS, true, false);
        conn.disconnect();
    }

    @Test
    public void testPrefixedAutoCreatedMulticastAndAnycastWithSameName() throws Exception {
        int port = 61614;
        URI uri = StompTest.createStompClientUri(this.scheme, this.hostname, port);
        String ADDRESS = UUID.randomUUID().toString();
        this.server.getRemotingService().createAcceptor("test", "tcp://" + this.hostname + ":" + port + "?protocols=STOMP&anycastPrefix=/queue/&multicastPrefix=/topic/").start();
        StompClientConnection conn = StompClientConnectionFactory.createClientConnection(uri);
        conn.connect(this.defUser, this.defPass);
        String uuid = UUID.randomUUID().toString();
        ClientStompFrame frame = conn.createFrame("SUBSCRIBE").addHeader("destination", "/topic/" + ADDRESS).addHeader("receipt", uuid);
        frame = conn.sendFrame(frame);
        StompTest.assertEquals((Object)uuid, (Object)frame.getHeader("receipt-id"));
        AddressInfo addressInfo = this.server.getAddressInfo(SimpleString.toSimpleString((String)ADDRESS));
        StompTest.assertNotNull((String)("No address was created with the name " + ADDRESS), (Object)addressInfo);
        StompTest.assertTrue((boolean)addressInfo.getRoutingTypes().contains(RoutingType.MULTICAST));
        StompTest.assertFalse((boolean)addressInfo.getRoutingTypes().contains(RoutingType.ANYCAST));
        frame = StompTest.send(conn, "/queue/" + ADDRESS, null, "Hello World 1", true);
        StompTest.assertFalse((boolean)frame.getCommand().equals("ERROR"));
        addressInfo = this.server.getAddressInfo(SimpleString.toSimpleString((String)ADDRESS));
        StompTest.assertTrue((boolean)addressInfo.getRoutingTypes().contains(RoutingType.ANYCAST));
        StompTest.assertTrue((boolean)addressInfo.getRoutingTypes().contains(RoutingType.MULTICAST));
        StompTest.assertNotNull((Object)this.server.locateQueue(SimpleString.toSimpleString((String)ADDRESS)));
        frame = conn.receiveFrame(100L);
        Assert.assertNull((Object)frame);
        frame = StompTest.send(conn, "/topic/" + ADDRESS, null, "Hello World 2", true);
        StompTest.assertFalse((boolean)frame.getCommand().equals("ERROR"));
        frame = conn.receiveFrame(2000L);
        Assert.assertEquals((Object)"MESSAGE", (Object)frame.getCommand());
        Assert.assertEquals((Object)"Hello World 2", (Object)frame.getBody());
        Assert.assertEquals((Object)RoutingType.MULTICAST.toString(), (Object)frame.getHeader("destination-type"));
        Assert.assertEquals((Object)("/topic/" + ADDRESS), (Object)frame.getHeader("destination"));
        frame = conn.receiveFrame(100L);
        Assert.assertNull((Object)frame);
        frame = StompTest.unsubscribe(conn, null, "/topic/" + ADDRESS, true, false);
        StompTest.assertFalse((boolean)frame.getCommand().equals("ERROR"));
        uuid = UUID.randomUUID().toString();
        frame = conn.createFrame("SUBSCRIBE").addHeader("destination", "/queue/" + ADDRESS).addHeader("receipt", uuid);
        frame = conn.sendFrame(frame);
        StompTest.assertEquals((Object)uuid, (Object)frame.getHeader("receipt-id"));
        frame = conn.receiveFrame(1000L);
        Assert.assertEquals((Object)"MESSAGE", (Object)frame.getCommand());
        Assert.assertEquals((Object)"Hello World 1", (Object)frame.getBody());
        Assert.assertEquals((Object)RoutingType.ANYCAST.toString(), (Object)frame.getHeader("destination-type"));
        Assert.assertEquals((Object)("/queue/" + ADDRESS), (Object)frame.getHeader("destination"));
        frame = conn.receiveFrame(100L);
        Assert.assertNull((Object)frame);
        StompTest.unsubscribe(conn, null, "/queue/" + ADDRESS, true, false);
        conn.disconnect();
    }

    @Test
    public void testDotPrefixedSendAndRecieveAnycast() throws Exception {
        this.testPrefixedSendAndRecieve("jms.queue.", RoutingType.ANYCAST);
    }

    @Test
    public void testDotPrefixedSendAndRecieveMulticast() throws Exception {
        this.testPrefixedSendAndRecieve("jms.topic.", RoutingType.MULTICAST);
    }

    @Test
    public void testSlashPrefixedSendAndRecieveAnycast() throws Exception {
        this.testPrefixedSendAndRecieve("/queue/", RoutingType.ANYCAST);
    }

    @Test
    public void testSlashPrefixedSendAndRecieveMulticast() throws Exception {
        this.testPrefixedSendAndRecieve("/topic/", RoutingType.MULTICAST);
    }

    public void testPrefixedSendAndRecieve(String prefix, RoutingType routingType) throws Exception {
        int port = 61614;
        URI uri = StompTest.createStompClientUri(this.scheme, this.hostname, port);
        String ADDRESS = UUID.randomUUID().toString();
        String PREFIXED_ADDRESS = prefix + ADDRESS;
        String urlParam = routingType.toString().toLowerCase() + "Prefix";
        this.server.getRemotingService().createAcceptor("test", "tcp://" + this.hostname + ":" + port + "?protocols=" + "STOMP" + "&" + urlParam + "=" + prefix).start();
        StompClientConnection conn = StompClientConnectionFactory.createClientConnection(uri);
        conn.connect(this.defUser, this.defPass);
        String uuid = UUID.randomUUID().toString();
        ClientStompFrame frame = conn.createFrame("SUBSCRIBE").addHeader("destination", PREFIXED_ADDRESS).addHeader("receipt", uuid);
        frame = conn.sendFrame(frame);
        StompTest.assertEquals((Object)uuid, (Object)frame.getHeader("receipt-id"));
        StompTest.send(conn, ADDRESS, null, "Hello World", true);
        frame = conn.receiveFrame(10000L);
        Assert.assertNotNull((String)"Should have received a message", (Object)frame);
        Assert.assertEquals((Object)"MESSAGE", (Object)frame.getCommand());
        Assert.assertEquals((Object)ADDRESS, (Object)frame.getHeader("destination"));
        Assert.assertEquals((Object)"Hello World", (Object)frame.getBody());
        conn.disconnect();
    }

    @Test
    public void testMulticastOperationsOnAnycastAddress() throws Exception {
        this.server.getAddressSettingsRepository().addMatch("#", (Object)new AddressSettings().setAutoCreateAddresses(Boolean.valueOf(false)).setAutoCreateQueues(Boolean.valueOf(false)));
        this.testRoutingSemantics(RoutingType.MULTICAST.toString(), StompTest.getQueuePrefix() + StompTest.getQueueName());
    }

    @Test
    public void testAnycastOperationsOnMulticastAddress() throws Exception {
        this.server.getAddressSettingsRepository().addMatch("#", (Object)new AddressSettings().setAutoCreateAddresses(Boolean.valueOf(false)).setAutoCreateQueues(Boolean.valueOf(false)));
        this.testRoutingSemantics(RoutingType.ANYCAST.toString(), StompTest.getTopicPrefix() + StompTest.getTopicName());
    }

    public void testRoutingSemantics(String routingType, String destination) throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        String uuid = UUID.randomUUID().toString();
        ClientStompFrame frame = this.conn.createFrame("SUBSCRIBE").addHeader("subscription-type", routingType).addHeader("destination", destination).addHeader("receipt", uuid);
        frame = this.conn.sendFrame(frame);
        StompTest.assertEquals((Object)"ERROR", (Object)frame.getCommand());
        uuid = UUID.randomUUID().toString();
        frame = this.conn.createFrame("SEND").addHeader("destination-type", RoutingType.MULTICAST.toString()).addHeader("destination", StompTest.getQueuePrefix() + StompTest.getQueueName()).addHeader("receipt", uuid);
        frame = this.conn.sendFrame(frame);
        StompTest.assertEquals((Object)"ERROR", (Object)frame.getCommand());
    }

    @Test
    public void testGetManagementAttributeFromStomp() throws Exception {
        this.server.getAddressSettingsRepository().addMatch("#", (Object)new AddressSettings().setAutoCreateAddresses(Boolean.valueOf(false)).setAutoCreateQueues(Boolean.valueOf(false)));
        this.conn.connect(this.defUser, this.defPass);
        StompTest.subscribe(this.conn, null);
        ClientStompFrame frame = this.conn.createFrame("SEND").addHeader("destination", ActiveMQDefaultConfiguration.getDefaultManagementAddress().toString()).addHeader("reply-to", StompTest.getQueuePrefix() + StompTest.getQueueName()).addHeader(ManagementHelper.HDR_RESOURCE_NAME.toString(), "queue." + StompTest.getQueuePrefix() + StompTest.getQueueName()).addHeader(ManagementHelper.HDR_ATTRIBUTE.toString(), "Address");
        this.conn.sendFrame(frame);
        frame = this.conn.receiveFrame(10000L);
        log.debug((Object)("Received: " + frame));
        Assert.assertEquals((Object)Boolean.TRUE.toString(), (Object)frame.getHeader(ManagementHelper.HDR_OPERATION_SUCCEEDED.toString()));
        Assert.assertEquals((Object)("[\"" + StompTest.getQueuePrefix() + StompTest.getQueueName() + "\"]"), (Object)frame.getBody());
        StompTest.unsubscribe(this.conn, null);
        this.conn.disconnect();
    }

    @Test
    public void testInvokeOperationFromStomp() throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        StompTest.subscribe(this.conn, null);
        ClientStompFrame frame = this.conn.createFrame("SEND").addHeader("destination", ActiveMQDefaultConfiguration.getDefaultManagementAddress().toString()).addHeader("reply-to", StompTest.getQueuePrefix() + StompTest.getQueueName()).addHeader(ManagementHelper.HDR_RESOURCE_NAME.toString(), "queue." + StompTest.getQueuePrefix() + StompTest.getQueueName()).addHeader(ManagementHelper.HDR_OPERATION_NAME.toString(), "countMessages").setBody("[\"color = 'blue'\"]");
        this.conn.sendFrame(frame);
        frame = this.conn.receiveFrame(10000L);
        log.debug((Object)("Received: " + frame));
        Assert.assertEquals((Object)Boolean.TRUE.toString(), (Object)frame.getHeader(ManagementHelper.HDR_OPERATION_SUCCEEDED.toString()));
        Assert.assertEquals((Object)"[0]", (Object)frame.getBody());
        StompTest.unsubscribe(this.conn, null);
        this.conn.disconnect();
    }

    @Test
    public void testAnycastMessageRoutingExclusivity() throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        String addressA = "addressA";
        String queueA = "queueA";
        String queueB = "queueB";
        String queueC = "queueC";
        ActiveMQServerControlImpl serverControl = this.server.getActiveMQServerControl();
        serverControl.createAddress("addressA", RoutingType.ANYCAST.toString() + "," + RoutingType.MULTICAST.toString());
        serverControl.createQueue(new QueueConfiguration("queueA").setAddress("addressA").setRoutingType(RoutingType.ANYCAST).toJSON());
        serverControl.createQueue(new QueueConfiguration("queueB").setAddress("addressA").setRoutingType(RoutingType.ANYCAST).toJSON());
        serverControl.createQueue(new QueueConfiguration("queueC").setAddress("addressA").setRoutingType(RoutingType.MULTICAST).toJSON());
        StompTest.send(this.conn, "addressA", null, "Hello World!", true, RoutingType.ANYCAST);
        Wait.assertTrue(() -> this.server.locateQueue(SimpleString.toSimpleString((String)"queueA")).getMessageCount() + this.server.locateQueue(SimpleString.toSimpleString((String)"queueB")).getMessageCount() == 1L);
        Wait.assertTrue(() -> this.server.locateQueue(SimpleString.toSimpleString((String)"queueC")).getMessageCount() == 0L);
    }

    @Test
    public void testMulticastMessageRoutingExclusivity() throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        String addressA = "addressA";
        String queueA = "queueA";
        String queueB = "queueB";
        String queueC = "queueC";
        ActiveMQServerControlImpl serverControl = this.server.getActiveMQServerControl();
        serverControl.createAddress("addressA", RoutingType.ANYCAST.toString() + "," + RoutingType.MULTICAST.toString());
        serverControl.createQueue(new QueueConfiguration("queueA").setAddress("addressA").setRoutingType(RoutingType.ANYCAST).toJSON());
        serverControl.createQueue(new QueueConfiguration("queueB").setAddress("addressA").setRoutingType(RoutingType.MULTICAST).toJSON());
        serverControl.createQueue(new QueueConfiguration("queueC").setAddress("addressA").setRoutingType(RoutingType.MULTICAST).toJSON());
        StompTest.send(this.conn, "addressA", null, "Hello World!", true, RoutingType.MULTICAST);
        Wait.assertTrue(() -> this.server.locateQueue(SimpleString.toSimpleString((String)"queueA")).getMessageCount() == 0L);
        Wait.assertTrue(() -> this.server.locateQueue(SimpleString.toSimpleString((String)"queueC")).getMessageCount() + this.server.locateQueue(SimpleString.toSimpleString((String)"queueB")).getMessageCount() == 2L);
    }

    @Test
    public void testAmbiguousMessageRouting() throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        String addressA = "addressA";
        String queueA = "queueA";
        String queueB = "queueB";
        String queueC = "queueC";
        String queueD = "queueD";
        ActiveMQServerControlImpl serverControl = this.server.getActiveMQServerControl();
        serverControl.createAddress("addressA", RoutingType.ANYCAST.toString() + "," + RoutingType.MULTICAST.toString());
        serverControl.createQueue(new QueueConfiguration("queueA").setAddress("addressA").setRoutingType(RoutingType.ANYCAST).toJSON());
        serverControl.createQueue(new QueueConfiguration("queueB").setAddress("addressA").setRoutingType(RoutingType.ANYCAST).toJSON());
        serverControl.createQueue(new QueueConfiguration("queueC").setAddress("addressA").setRoutingType(RoutingType.MULTICAST).toJSON());
        serverControl.createQueue(new QueueConfiguration("queueD").setAddress("addressA").setRoutingType(RoutingType.MULTICAST).toJSON());
        StompTest.send(this.conn, "addressA", null, "Hello World!", true);
        Wait.assertTrue(() -> this.server.locateQueue(SimpleString.toSimpleString((String)"queueA")).getMessageCount() + this.server.locateQueue(SimpleString.toSimpleString((String)"queueB")).getMessageCount() == 1L);
        Wait.assertTrue(() -> this.server.locateQueue(SimpleString.toSimpleString((String)"queueC")).getMessageCount() + this.server.locateQueue(SimpleString.toSimpleString((String)"queueD")).getMessageCount() == 2L);
    }

    @Test
    public void testAutoCreatedAnycastAddress() throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        String queueName = UUID.randomUUID().toString();
        SimpleString simpleQueueName = SimpleString.toSimpleString((String)queueName);
        Assert.assertNull((Object)this.server.getAddressInfo(simpleQueueName));
        Assert.assertNull((Object)this.server.locateQueue(simpleQueueName));
        this.server.getAddressSettingsRepository().addMatch(queueName, (Object)new AddressSettings().setDefaultAddressRoutingType(RoutingType.ANYCAST).setDefaultQueueRoutingType(RoutingType.ANYCAST));
        StompTest.send(this.conn, queueName, null, "Hello ANYCAST");
        Wait.assertTrue((String)"Address and queue should be created now", () -> this.server.getAddressInfo(simpleQueueName) != null && this.server.locateQueue(simpleQueueName) != null);
        StompTest.assertTrue((boolean)this.server.getAddressInfo(simpleQueueName).getRoutingTypes().contains(RoutingType.ANYCAST));
        StompTest.assertEquals((Object)RoutingType.ANYCAST, (Object)this.server.locateQueue(simpleQueueName).getRoutingType());
    }

    @Test
    public void testAutoCreatedMulticastAddress() throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        String queueName = UUID.randomUUID().toString();
        SimpleString simpleQueueName = SimpleString.toSimpleString((String)queueName);
        Assert.assertNull((Object)this.server.getAddressInfo(simpleQueueName));
        Assert.assertNull((Object)this.server.locateQueue(simpleQueueName));
        StompTest.send(this.conn, queueName, null, "Hello MULTICAST");
        Wait.assertTrue((String)"Address should be created now", () -> this.server.getAddressInfo(simpleQueueName) != null);
        StompTest.assertTrue((boolean)this.server.getAddressInfo(simpleQueueName).getRoutingTypes().contains(RoutingType.MULTICAST));
        Assert.assertNull((Object)this.server.locateQueue(simpleQueueName));
    }

    @Test
    public void directDeliverDisabledOnStomp() throws Exception {
        String payload = "This is a test message";
        this.conn.connect(this.defUser, this.defPass);
        StompTest.subscribe(this.conn, null, "auto");
        for (Binding b : IterableStream.iterableOf(this.server.getPostOffice().getAllBindings().filter(QueueBinding.class::isInstance))) {
            Assert.assertFalse((String)("Queue " + ((QueueBinding)b).getQueue().getName()), (boolean)((QueueBinding)b).getQueue().isDirectDeliver());
        }
        FuseMQTTClientProvider clientProvider = new FuseMQTTClientProvider();
        clientProvider.connect("tcp://" + this.hostname + ":" + 61613);
        clientProvider.publish(StompTest.getQueuePrefix() + StompTest.getQueueName(), payload.getBytes(), 0);
        clientProvider.disconnect();
        ClientStompFrame frame = this.conn.receiveFrame();
        StompTest.assertTrue((boolean)frame.getBody().contains(payload));
    }
}

