/*
 * Copyright (C) 2008-2015 by Holger Arndt
 *
 * This file is part of the Universal Java Matrix Package (UJMP).
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership and licensing.
 *
 * UJMP is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * UJMP is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with UJMP; if not, write to the
 * Free Software Foundation, Inc., 51 Franklin St, Fifth Floor,
 * Boston, MA  02110-1301  USA
 */

package org.ujmp.core.util.concurrent;

import java.util.concurrent.BlockingQueue;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

import org.ujmp.core.util.UJMPSettings;

public class UJMPThreadPoolExecutor extends ThreadPoolExecutor {

	private static final ThreadLocal<ThreadPoolExecutor> executors = new ThreadLocal<ThreadPoolExecutor>();

	public UJMPThreadPoolExecutor(String name, int corePoolSize, int maximumPoolSize,
			long keepAliveTime, TimeUnit unit, BlockingQueue<Runnable> workQueue, int priority,
			boolean deamon) {
		super(corePoolSize, maximumPoolSize, keepAliveTime, unit, workQueue, new UJMPThreadFactory(
				name, priority, deamon));
	}

	public UJMPThreadPoolExecutor(String name, int corePoolSize, int maximumPoolSize) {
		this(name, corePoolSize, maximumPoolSize, 500L, TimeUnit.MILLISECONDS,
				new LinkedBlockingQueue<Runnable>());
	}

	public UJMPThreadPoolExecutor(String name, int corePoolSize, int maximumPoolSize,
			long keepAliveTime, TimeUnit unit, BlockingQueue<Runnable> workQueue) {
		this(name, maximumPoolSize, maximumPoolSize, keepAliveTime, unit, workQueue,
				Thread.NORM_PRIORITY, true);
	}

	public static final ThreadPoolExecutor getInstance() {
		return getInstance(UJMPSettings.getInstance().getNumberOfThreads());
	}

	public static final ThreadPoolExecutor getInstance(final int maximumPoolSize) {
		ThreadPoolExecutor es = executors.get();
		if (es == null) {
			synchronized (executors) {
				es = new UJMPThreadPoolExecutor(Thread.currentThread().getName(), 0,
						maximumPoolSize);
				executors.set(es);
			}
		}
		return es;
	}

}
