/*
 * Decompiled with CFR 0.152.
 */
package org.threeten.bp;

import java.io.DataInput;
import java.io.DataOutput;
import java.io.IOException;
import java.io.InvalidObjectException;
import java.io.ObjectStreamException;
import java.io.Serializable;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.math.RoundingMode;
import java.util.Objects;
import org.threeten.bp.DateTimeException;
import org.threeten.bp.Ser;
import org.threeten.bp.format.DateTimeParseException;
import org.threeten.bp.jdk8.Jdk8Methods;
import org.threeten.bp.temporal.ChronoField;
import org.threeten.bp.temporal.ChronoUnit;
import org.threeten.bp.temporal.Temporal;
import org.threeten.bp.temporal.TemporalAccessor;
import org.threeten.bp.temporal.TemporalAdder;
import org.threeten.bp.temporal.TemporalSubtractor;
import org.threeten.bp.temporal.TemporalUnit;

public final class Duration
implements TemporalAdder,
TemporalSubtractor,
Comparable<Duration>,
Serializable {
    public static final Duration ZERO = new Duration(0L, 0);
    private static final long serialVersionUID = 3078945930695997490L;
    private static final int NANOS_PER_SECOND = 1000000000;
    private static final BigInteger BI_NANOS_PER_SECOND = BigInteger.valueOf(1000000000L);
    private final long seconds;
    private final int nanos;

    public static Duration ofSeconds(long seconds) {
        return Duration.create(seconds, 0);
    }

    public static Duration ofSeconds(long seconds, long nanoAdjustment) {
        long secs = Jdk8Methods.safeAdd(seconds, Jdk8Methods.floorDiv(nanoAdjustment, 1000000000L));
        int nos = Jdk8Methods.floorMod(nanoAdjustment, 1000000000);
        return Duration.create(secs, nos);
    }

    public static Duration ofMillis(long millis) {
        long secs = millis / 1000L;
        int mos = (int)(millis % 1000L);
        if (mos < 0) {
            mos += 1000;
            --secs;
        }
        return Duration.create(secs, mos * 1000000);
    }

    public static Duration ofNanos(long nanos) {
        long secs = nanos / 1000000000L;
        int nos = (int)(nanos % 1000000000L);
        if (nos < 0) {
            nos += 1000000000;
            --secs;
        }
        return Duration.create(secs, nos);
    }

    public static Duration ofMinutes(long minutes) {
        return Duration.create(Jdk8Methods.safeMultiply(minutes, 60), 0);
    }

    public static Duration ofHours(long hours) {
        return Duration.create(Jdk8Methods.safeMultiply(hours, 3600), 0);
    }

    public static Duration ofDays(long days) {
        return Duration.create(Jdk8Methods.safeMultiply(days, 86400), 0);
    }

    public static Duration of(long amount, TemporalUnit unit) {
        return ZERO.plus(amount, unit);
    }

    public static Duration between(TemporalAccessor startInclusive, TemporalAccessor endExclusive) {
        long secs = Jdk8Methods.safeSubtract(endExclusive.getLong(ChronoField.INSTANT_SECONDS), startInclusive.getLong(ChronoField.INSTANT_SECONDS));
        long nanos = endExclusive.getLong(ChronoField.NANO_OF_SECOND) - startInclusive.getLong(ChronoField.NANO_OF_SECOND);
        secs = Jdk8Methods.safeAdd(secs, Jdk8Methods.floorDiv(nanos, 1000000000L));
        nanos = Jdk8Methods.floorMod(nanos, 1000000000);
        return Duration.create(secs, (int)nanos);
    }

    public static Duration parse(CharSequence text) {
        Objects.requireNonNull(text, "text");
        int len = text.length();
        if (len < 4 || text.charAt(0) != 'P' && text.charAt(0) != 'p' || text.charAt(1) != 'T' && text.charAt(1) != 't' || text.charAt(len - 1) != 'S' && text.charAt(len - 1) != 's' || len == 5 && text.charAt(2) == '-' && text.charAt(3) == '0') {
            throw new DateTimeParseException("Duration could not be parsed: " + text, text, 0);
        }
        String numberText = text.subSequence(2, len - 1).toString().replace(',', '.');
        if (numberText.charAt(0) == '+') {
            throw new DateTimeParseException("Duration could not be parsed: " + text, text, 2);
        }
        int dot = numberText.indexOf(46);
        try {
            if (dot == -1) {
                if (numberText.startsWith("-0")) {
                    throw new DateTimeParseException("Duration could not be parsed: " + text, text, 2);
                }
                return Duration.create(Long.parseLong(numberText), 0);
            }
            boolean negative = false;
            if (numberText.charAt(0) == '-') {
                negative = true;
            }
            long secs = Long.parseLong(numberText.substring(0, dot));
            len = (numberText = numberText.substring(dot + 1)).length();
            if (len == 0 || len > 9 || numberText.charAt(0) == '-' || numberText.charAt(0) == '+') {
                throw new DateTimeParseException("Duration could not be parsed: " + text, text, 2);
            }
            int nanos = Integer.parseInt(numberText);
            switch (len) {
                case 1: {
                    nanos *= 100000000;
                    break;
                }
                case 2: {
                    nanos *= 10000000;
                    break;
                }
                case 3: {
                    nanos *= 1000000;
                    break;
                }
                case 4: {
                    nanos *= 100000;
                    break;
                }
                case 5: {
                    nanos *= 10000;
                    break;
                }
                case 6: {
                    nanos *= 1000;
                    break;
                }
                case 7: {
                    nanos *= 100;
                    break;
                }
                case 8: {
                    nanos *= 10;
                }
            }
            return negative ? Duration.ofSeconds(secs, -nanos) : Duration.create(secs, nanos);
        }
        catch (ArithmeticException | NumberFormatException ex) {
            throw new DateTimeParseException("Duration could not be parsed: " + text, text, 2, ex);
        }
    }

    private static Duration create(long seconds, int nanoAdjustment) {
        if ((seconds | (long)nanoAdjustment) == 0L) {
            return ZERO;
        }
        return new Duration(seconds, nanoAdjustment);
    }

    private Duration(long seconds, int nanos) {
        this.seconds = seconds;
        this.nanos = nanos;
    }

    public boolean isZero() {
        return (this.seconds | (long)this.nanos) == 0L;
    }

    public boolean isPositive() {
        return this.seconds >= 0L && (this.seconds | (long)this.nanos) != 0L;
    }

    public boolean isNegative() {
        return this.seconds < 0L;
    }

    public long getSeconds() {
        return this.seconds;
    }

    public int getNano() {
        return this.nanos;
    }

    public Duration plus(Duration duration) {
        return this.plus(duration.getSeconds(), duration.getNano());
    }

    public Duration plus(long amountToAdd, TemporalUnit unit) {
        Objects.requireNonNull(unit, "unit");
        if (unit == ChronoUnit.DAYS) {
            return this.plus(Jdk8Methods.safeMultiply(amountToAdd, 86400), 0L);
        }
        if (unit.isDurationEstimated()) {
            throw new DateTimeException("Unit must not have an estimated duration");
        }
        if (amountToAdd == 0L) {
            return this;
        }
        if (unit instanceof ChronoUnit) {
            switch ((ChronoUnit)unit) {
                case NANOS: {
                    return this.plusNanos(amountToAdd);
                }
                case MICROS: {
                    return this.plusSeconds(amountToAdd / 1000000000L * 1000L).plusNanos(amountToAdd % 1000000000L * 1000L);
                }
                case MILLIS: {
                    return this.plusMillis(amountToAdd);
                }
                case SECONDS: {
                    return this.plusSeconds(amountToAdd);
                }
            }
            return this.plusSeconds(Jdk8Methods.safeMultiply(unit.getDuration().seconds, amountToAdd));
        }
        Duration duration = unit.getDuration().multipliedBy(amountToAdd);
        return this.plusSeconds(duration.getSeconds()).plusNanos(duration.getNano());
    }

    public Duration plusSeconds(long secondsToAdd) {
        return this.plus(secondsToAdd, 0L);
    }

    public Duration plusMillis(long millisToAdd) {
        return this.plus(millisToAdd / 1000L, millisToAdd % 1000L * 1000000L);
    }

    public Duration plusNanos(long nanosToAdd) {
        return this.plus(0L, nanosToAdd);
    }

    private Duration plus(long secondsToAdd, long nanosToAdd) {
        if ((secondsToAdd | nanosToAdd) == 0L) {
            return this;
        }
        long epochSec = Jdk8Methods.safeAdd(this.seconds, secondsToAdd);
        epochSec = Jdk8Methods.safeAdd(epochSec, nanosToAdd / 1000000000L);
        long nanoAdjustment = (long)this.nanos + (nanosToAdd %= 1000000000L);
        return Duration.ofSeconds(epochSec, nanoAdjustment);
    }

    public Duration minus(Duration duration) {
        long secsToSubtract = duration.getSeconds();
        int nanosToSubtract = duration.getNano();
        if (secsToSubtract == Long.MIN_VALUE) {
            return this.plus(Long.MAX_VALUE, -nanosToSubtract).plus(1L, 0L);
        }
        return this.plus(-secsToSubtract, -nanosToSubtract);
    }

    public Duration minus(long amountToSubtract, TemporalUnit unit) {
        return amountToSubtract == Long.MIN_VALUE ? this.plus(Long.MAX_VALUE, unit).plus(1L, unit) : this.plus(-amountToSubtract, unit);
    }

    public Duration minusSeconds(long secondsToSubtract) {
        return secondsToSubtract == Long.MIN_VALUE ? this.plusSeconds(Long.MAX_VALUE).plusSeconds(1L) : this.plusSeconds(-secondsToSubtract);
    }

    public Duration minusMillis(long millisToSubtract) {
        return millisToSubtract == Long.MIN_VALUE ? this.plusMillis(Long.MAX_VALUE).plusMillis(1L) : this.plusMillis(-millisToSubtract);
    }

    public Duration minusNanos(long nanosToSubtract) {
        return nanosToSubtract == Long.MIN_VALUE ? this.plusNanos(Long.MAX_VALUE).plusNanos(1L) : this.plusNanos(-nanosToSubtract);
    }

    public Duration multipliedBy(long multiplicand) {
        if (multiplicand == 0L) {
            return ZERO;
        }
        if (multiplicand == 1L) {
            return this;
        }
        return Duration.create(this.toSeconds().multiply(BigDecimal.valueOf(multiplicand)));
    }

    public Duration dividedBy(long divisor) {
        if (divisor == 0L) {
            throw new ArithmeticException("Cannot divide by zero");
        }
        if (divisor == 1L) {
            return this;
        }
        return Duration.create(this.toSeconds().divide(BigDecimal.valueOf(divisor), RoundingMode.DOWN));
    }

    private BigDecimal toSeconds() {
        return BigDecimal.valueOf(this.seconds).add(BigDecimal.valueOf(this.nanos, 9));
    }

    private static Duration create(BigDecimal seconds) {
        BigInteger nanos = seconds.movePointRight(9).toBigIntegerExact();
        BigInteger[] divRem = nanos.divideAndRemainder(BI_NANOS_PER_SECOND);
        if (divRem[0].bitLength() > 63) {
            throw new ArithmeticException("Exceeds capacity of Duration: " + nanos);
        }
        return Duration.ofSeconds(divRem[0].longValue(), divRem[1].intValue());
    }

    public Duration negated() {
        return this.multipliedBy(-1L);
    }

    public Duration abs() {
        return this.isNegative() ? this.negated() : this;
    }

    @Override
    public Temporal addTo(Temporal temporal) {
        long instantSecs = temporal.getLong(ChronoField.INSTANT_SECONDS);
        long instantNanos = temporal.getLong(ChronoField.NANO_OF_SECOND);
        instantSecs = Jdk8Methods.safeAdd(instantSecs, this.seconds);
        instantNanos = Jdk8Methods.safeAdd(instantNanos, (long)this.nanos);
        instantSecs = Jdk8Methods.safeAdd(instantSecs, Jdk8Methods.floorDiv(instantNanos, 1000000000L));
        instantNanos = Jdk8Methods.floorMod(instantNanos, 1000000000);
        return temporal.with(ChronoField.INSTANT_SECONDS, instantSecs).with(ChronoField.NANO_OF_SECOND, instantNanos);
    }

    @Override
    public Temporal subtractFrom(Temporal temporal) {
        long instantSecs = temporal.getLong(ChronoField.INSTANT_SECONDS);
        long instantNanos = temporal.getLong(ChronoField.NANO_OF_SECOND);
        instantSecs = Jdk8Methods.safeSubtract(instantSecs, this.seconds);
        instantNanos = Jdk8Methods.safeSubtract(instantNanos, (long)this.nanos);
        instantSecs = Jdk8Methods.safeAdd(instantSecs, Jdk8Methods.floorDiv(instantNanos, 1000000000L));
        instantNanos = Jdk8Methods.floorMod(instantNanos, 1000000000);
        return temporal.with(ChronoField.INSTANT_SECONDS, instantSecs).with(ChronoField.NANO_OF_SECOND, instantNanos);
    }

    public long toMillis() {
        long millis = Jdk8Methods.safeMultiply(this.seconds, 1000);
        millis = Jdk8Methods.safeAdd(millis, (long)(this.nanos / 1000000));
        return millis;
    }

    public long toNanos() {
        long millis = Jdk8Methods.safeMultiply(this.seconds, 1000000000);
        millis = Jdk8Methods.safeAdd(millis, (long)this.nanos);
        return millis;
    }

    @Override
    public int compareTo(Duration otherDuration) {
        int cmp = Long.compare(this.seconds, otherDuration.seconds);
        if (cmp != 0) {
            return cmp;
        }
        return this.nanos - otherDuration.nanos;
    }

    public boolean isGreaterThan(Duration otherDuration) {
        return this.compareTo(otherDuration) > 0;
    }

    public boolean isLessThan(Duration otherDuration) {
        return this.compareTo(otherDuration) < 0;
    }

    public boolean equals(Object otherDuration) {
        if (this == otherDuration) {
            return true;
        }
        if (otherDuration instanceof Duration) {
            Duration other = (Duration)otherDuration;
            return this.seconds == other.seconds && this.nanos == other.nanos;
        }
        return false;
    }

    public int hashCode() {
        return (int)(this.seconds ^ this.seconds >>> 32) + 51 * this.nanos;
    }

    public String toString() {
        StringBuilder buf = new StringBuilder(24);
        buf.append("PT");
        if (this.seconds < 0L && this.nanos > 0) {
            if (this.seconds == -1L) {
                buf.append("-0");
            } else {
                buf.append(this.seconds + 1L);
            }
        } else {
            buf.append(this.seconds);
        }
        if (this.nanos > 0) {
            int pos = buf.length();
            if (this.seconds < 0L) {
                buf.append(2000000000 - this.nanos);
            } else {
                buf.append(this.nanos + 1000000000);
            }
            while (buf.charAt(buf.length() - 1) == '0') {
                buf.setLength(buf.length() - 1);
            }
            buf.setCharAt(pos, '.');
        }
        buf.append('S');
        return buf.toString();
    }

    private Object writeReplace() {
        return new Ser(1, this);
    }

    private Object readResolve() throws ObjectStreamException {
        throw new InvalidObjectException("Deserialization via serialization delegate");
    }

    void writeExternal(DataOutput out) throws IOException {
        out.writeLong(this.seconds);
        out.writeInt(this.nanos);
    }

    static Duration readExternal(DataInput in) throws IOException {
        long seconds = in.readLong();
        int nanos = in.readInt();
        return Duration.ofSeconds(seconds, nanos);
    }
}

