/*
 * All content copyright (c) 2003-2008 Terracotta, Inc., except as may otherwise be noted in a separate copyright
 * notice. All rights reserved.
 */
package com.tc.test.server.appserver.deployment;

import com.tc.config.schema.test.L2ConfigBuilder;
import com.tc.config.schema.test.TerracottaConfigBuilder;
import com.tc.objectserver.control.ExtraProcessServerControl;
import com.tc.util.TcConfigBuilder;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.PrintWriter;
import java.util.ArrayList;
import java.util.List;

public class DSOServer extends AbstractStoppable {

  private static final String       SERVER_TEST_CONFIG = "server-config.xml";

  private ExtraProcessServerControl serverProc         = null;
  private final boolean             withPersistentStore;

  private int                       serverPort         = 9510;
  private int                       adminPort          = 9520;
  private int                       groupPort          = 9530;

  private final List                jvmArgs            = new ArrayList();

  private final File                workingDir;
  private TcConfigBuilder           configBuilder;

  public DSOServer(boolean withPersistentStore, File workingDir) {
    this.withPersistentStore = withPersistentStore;
    this.workingDir = workingDir;
  }

  public DSOServer(boolean withPersistentStore, File workingDir, TcConfigBuilder configBuilder) {
    this.withPersistentStore = withPersistentStore;
    this.workingDir = workingDir;
    this.configBuilder = configBuilder;
    this.serverPort = configBuilder.getDsoPort();
    this.adminPort = configBuilder.getJmxPort();
    this.groupPort = configBuilder.getGroupPort();
  }

  @Override
  protected void doStart() throws Exception {
    File configFile = writeConfig();
    serverProc = new ExtraProcessServerControl("localhost", serverPort, adminPort, configFile.getAbsolutePath(), false);
    serverProc.writeOutputTo(new FileOutputStream(new File(workingDir, "dso-server.log")));
    serverProc.getJvmArgs().addAll(jvmArgs);
    serverProc.start();
  }

  @Override
  protected void doStop() throws Exception {
    logger.debug("Stopping...");
    serverProc.shutdown();
    logger.debug("...stopped");
  }

  private File writeConfig() throws IOException {
    File configFile = new File(workingDir, SERVER_TEST_CONFIG);
    if (configBuilder != null) {
      configBuilder.setTcConfigFile(configFile);
      configBuilder.saveToFile();
    } else {
      TerracottaConfigBuilder builder = TerracottaConfigBuilder.newMinimalInstance();
      builder.getSystem().setConfigurationModel("development");

      L2ConfigBuilder l2 = builder.getServers().getL2s()[0];
      l2.setDSOPort(serverPort);
      l2.setJMXPort(adminPort);
      l2.setL2GroupPort(groupPort);
      l2.setData(workingDir + File.separator + "data");
      l2.setLogs(workingDir + File.separator + "logs");
      if (withPersistentStore) {
        l2.setPersistenceMode("permanent-store"); // XXX make this one configurable
      }

      String configAsString = builder.toString();

      FileOutputStream fileOutputStream = new FileOutputStream(configFile);
      PrintWriter out = new PrintWriter((fileOutputStream));
      out.println(configAsString);
      out.flush();
      out.close();
    }
    return configFile;
  }

  @Override
  public String toString() {
    return "DSO server; serverport:" + serverPort + "; adminPort:" + adminPort;
  }

  public int getServerPort() {
    return serverPort;
  }

  public int getAdminPort() {
    return adminPort;
  }

  public List getJvmArgs() {
    return jvmArgs;
  }

}
