/* $Id: BasicTimInfo.java 3033 2008-03-21 20:51:19Z jgalang $
 * Copyright (c) 2003-2008 Terracotta, Inc. All rights reserved.
 */
package org.terracotta.modules;

import java.net.URL;
import java.util.Properties;

/**
 * Basic implementation of the {@link TimInfo} interface that looks up
 * properties based on the groupId and artifactId.
 *
 * @author Jason Voegele (jvoegele@terracotta.org)
 */
public class BasicTimInfo implements TimInfo {
  private final String groupId;
  private final String artifactId;
  private Properties props;

  /**
   * Creates a new {@link BasicTimInfo} object for the TIM with the given artifactId
   * and the default groupId {@link TimInfo.DEFAULT_GROUP_ID}.
   */
  public BasicTimInfo(String artifactId) {
    this(DEFAULT_GROUP_ID, artifactId);
  }

  /**
   * Creates a new {@link BasicTimInfo} object for the TIM with the given groupId and artifactId.
   */
  public BasicTimInfo(String groupId, String artifactId) {
    this.groupId = groupId;
    this.artifactId = artifactId;
  }

  public String artifactId() {
    return artifactId;
  }

  public String groupId() {
    return groupId;
  }

  public String fullName() {
    return groupId() + "." + artifactId();
  }

  /**
   * Returns the properties associated with the TIM by loading from the CLASSPATH
   * the resource named by concatenating the groupId and artifactId, replacing
   * dots with slashes, and appending ".properties".  For example, given a groupId
   * of "org.terracotta.modules" and an artifactId of "tim-example", this method
   * would return the properties loaded from the CLASSPATH resource
   * "/org/terracotta/modules/tim-example.properties".
   */
  public synchronized Properties properties() {
    if (props == null) {
      props = new Properties();
      String propsPath = propertiesPath();
      URL propsUrl = getClass().getResource(propsPath);
      if (propsUrl == null) {
        throw new PropertiesLoadException("Error loading properties for TIM " +
            fullName() + ".  " + propsPath + " not found in CLASSPATH");
      }
      try {
        props.load(getClass().getResourceAsStream(propsPath));
      }
      catch (Exception e) {
        String message = "Error loading properties for TIM " + fullName();
        throw new PropertiesLoadException(message, e);
      }
    }
    return props;
  }

  /**
   * Returns the path to the properties for this TIM by combining the groupId
   * and the artifactId, replacing periods with slashes.
   */
  protected String propertiesPath() {
    return '/' + groupId().replace('.', '/') + '/' + artifactId() + ".properties";
  }

  /**
   * Returns the version for the TIM, assuming that it is present in the properties
   * under the key "version".
   */
  public String version() {
    return properties().getProperty("version");
  }

}
