/**
 * 
 * All content copyright (c) 2003-2008 Terracotta, Inc.,
 * except as may otherwise be noted in a separate copyright notice.
 * All rights reserved.
 *
 */
package org.terracotta.workmanager.support;

import commonj.work.Work;
import commonj.work.WorkEvent;
import commonj.work.WorkException;
import commonj.work.WorkItem;
import commonj.work.WorkListener;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * {@link commonj.work.WorkItem} implementation, holding the actual {@link commonj.work.Work} to be executed, status of the progress and the
 * future result.<br>
 * This implementation differs from the CommonJ specification regarding the work status tracking, which can be cancelled in order to
 * optimize performances by reducing time spent in status communication.<br>
 * Please note that when ignoring work status, {@link commonj.work.WorkListener}s will not be fired.
 */
public class DefaultWorkItem implements WorkItem {
    
    private transient static final Logger logger = LoggerFactory.getLogger(DefaultWorkItem.class);
    
    public static final int NO_STATUS_TRACKING = -1;

    private int status = NO_STATUS_TRACKING;
    private final Work work;
    private final WorkListener workListener;
    private final boolean cancelWorkStatusTracking;

    /**
     * Construct the work item.
     * 
     * @param work The work object this item holds.
     * @param workListener The work listener to be notified on work status changing.
     * @param cancelWorkStatusTracking True to avoid tracking status, false otherwise.
     */
    public DefaultWorkItem(final Work work, final WorkListener workListener, final boolean cancelWorkStatusTracking) {
        this.work = work;
        this.workListener = workListener;
        this.cancelWorkStatusTracking = cancelWorkStatusTracking;
    }

    /**
     * Get the resulting work.
     * 
     * @return The {@link commonj.work.Work} object.
     */
    public synchronized Work getResult() {
        return work;
    }

    /**
     * Get this work item status.
     * 
     * @return The status (see {@link commonj.work.WorkEvent}).
     */
    public synchronized int getStatus() {
        return status;
    }

    /**
     * Set the status of this work item and notifies the related work listener.
     * 
     * @param status The status to set (see {@link commonj.work.WorkEvent}).
     * @param exception The {@link commonj.work.WorkException}, if any.
     */
    public synchronized void setStatus(final int status, final WorkException exception) {
        if (!cancelWorkStatusTracking) {
            this.status = status;
            if (workListener != null) {
                switch (status) {
                    case WorkEvent.WORK_ACCEPTED:
                        workListener.workAccepted(new DefaultWorkEvent(WorkEvent.WORK_ACCEPTED, this, exception));
                        break;
                    case WorkEvent.WORK_REJECTED:
                        workListener.workRejected(new DefaultWorkEvent(WorkEvent.WORK_REJECTED, this, exception));
                        break;
                    case WorkEvent.WORK_STARTED:
                        workListener.workStarted(new DefaultWorkEvent(WorkEvent.WORK_STARTED, this, exception));
                        break;
                    case WorkEvent.WORK_COMPLETED:
                        workListener.workCompleted(new DefaultWorkEvent(WorkEvent.WORK_COMPLETED, this, exception));
                        break;
                }
            }
        } else {
            logger.warn("Setting status on a work item with disabled status tracking: this will have no effect.");
        }
    }

    @SuppressWarnings("unchecked")
    public int compareTo(Object compareTo) {
        Work otherWork = ((WorkItem) compareTo).getResult();
        if (otherWork instanceof Comparable) {
            Comparable comparableWork1 = (Comparable) otherWork;
            if (work instanceof Comparable) {
                Comparable comparableWork2 = (Comparable) work;
                return comparableWork2.compareTo(comparableWork1);
            }
        }
        return 0;
    }

    public String toString() {
        String out;
        switch (status) {
            case WorkEvent.WORK_ACCEPTED:
                out = "WORK_ACCEPTED";
                break;
            case WorkEvent.WORK_COMPLETED:
                out = "WORK_COMPLETED";
                break;
            case WorkEvent.WORK_REJECTED:
                out = "WORK_REJECTED";
                break;
            case WorkEvent.WORK_STARTED:
                out = "WORK_STARTED";
                break;
            default:
                out = "UNKNOWN";
        }
        return work.toString() + ":" + out;
    }
}
