/**
 * 
 * All content copyright (c) 2003-2008 Terracotta, Inc.,
 * except as may otherwise be noted in a separate copyright notice.
 * All rights reserved.
 *
 */
package org.terracotta.workmanager;

import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import org.terracotta.message.topology.Topology;

/**
 * Distributed CommonJ Worker implementation.<br>
 * <code>DistributedWorkManager</code> and <code>DistributedWorker</code> instances must share the same
 * {@link org.terracotta.message.topology.Topology} in order to collaborate each other.
 */
public class DistributedWorker extends AbstractWorkManagerWorker {

    /**
     * Create a DistributedWorker which executes works sequentially in another worker thread.
     * 
     * @param topologyName The name of the topology holding the pipe to listen to:
     * the actual {@link org.terracotta.message.topology.Topology} implementation is left to the worker.
     */
    public DistributedWorker(final String topologyName) {
        this(null, topologyName, null, Executors.newSingleThreadExecutor());
    }

    /**
     * Create a DistributedWorker which executes works sequentially in another worker thread.
     *
     * @param workerPrefix An optional string value to use as a prefix for this worker id; please note that
     * it is possible to assign the same prefix to more workers.
     * @param topologyName The name of the topology holding the pipe to listen to:
     * the actual {@link org.terracotta.message.topology.Topology} implementation is left to the worker.
     */
    public DistributedWorker(final String workerPrefix, final String topologyName) {
        this(workerPrefix, topologyName, null, Executors.newSingleThreadExecutor());
    }

    /**
     * Create a DistributedWorker which executes works as specified by the passed {@link java.util.concurrent.ExecutorService}.
     *
     * @param workerPrefix An optional string value to use as a prefix for this worker id; please note that
     * it is possible to assign the same prefix to more workers.
     * @param topologyName The name of the topology holding the pipe to listen to: the actual {@link org.terracotta.message.topology.Topology}
     * implementation is left to the worker.
     * @param executorService The {@link java.util.concurrent.ExecutorService} used for actually executing works.
     */
    public DistributedWorker(final String workerPrefix, final String topologyName, final ExecutorService executorService) {
        this(workerPrefix, topologyName, null, executorService);
    }

    /**
     * Create a DistributedWorker which executes works sequentially in another worker thread.
     *
     * @param workerPrefix An optional string value to use as a prefix for this worker id; please note that
     * it is possible to assign the same prefix to more workers.
     * @param topologyName The name of the topology holding the pipe to listen to.
     * @param topologyFactory The factory to use for the topology implementation.
     */
    public DistributedWorker(final String workerPrefix, final String topologyName, final Topology.Factory topologyFactory) {
        this(workerPrefix, topologyName, topologyFactory, Executors.newSingleThreadExecutor());
    }

    /**
     * Create a DistributedWorker which executes works as specified by the passed {@link java.util.concurrent.ExecutorService}.
     *
     * @param workerPrefix An optional string value to use as a prefix for this worker id; please note that
     * it is possible to assign the same prefix to more workers.
     * @param topologyName The name of the topology holding the pipe to listen to.
     * @param topologyFactory The factory to use for the topology implementation.
     * @param executorService The {@link java.util.concurrent.ExecutorService} used for actually executing works.
     */
    public DistributedWorker(final String workerPrefix, final String topologyName, final Topology.Factory topologyFactory, final ExecutorService executorService) {
        super(workerPrefix, topologyName, topologyFactory, executorService);
    }
}
