/**
 * 
 * All content copyright (c) 2003-2008 Terracotta, Inc.,
 * except as may otherwise be noted in a separate copyright notice.
 * All rights reserved.
 *
 */
package org.terracotta.workmanager;

import java.util.Collection;
import commonj.work.Work;
import commonj.work.WorkEvent;
import commonj.work.WorkItem;
import commonj.work.WorkListener;
import commonj.work.WorkManager;
import java.util.ArrayList;
import java.util.LinkedList;
import java.util.List;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.terracotta.masterworker.AbstractMaster;
import org.terracotta.masterworker.Master;
import org.terracotta.masterworker.WorkMessage;
import org.terracotta.message.routing.Router;
import org.terracotta.message.topology.Topology;
import org.terracotta.message.topology.TopologyManager;
import org.terracotta.workmanager.support.DefaultWorkItem;

/**
 * Abstract {@link commonj.work.WorkManager} implementation based on {@link org.terracotta.masterworker.AbstractMaster}
 * and {@link DefaultWorkItem}.  
 */
public abstract class AbstractWorkManager implements WorkManager {

    private transient static final Logger logger = LoggerFactory.getLogger(AbstractWorkManager.class);
    private final Master<DefaultWorkItem> masterDelegate;
    private final List<DefaultWorkItem> completedWorks = new LinkedList<DefaultWorkItem>();

    public AbstractWorkManager(final String topologyName, final Router router) {
        this(topologyName, null, router);
    }

    public AbstractWorkManager(final String topologyName, final Topology.Factory topologyFactory, final Router router) {
        this.masterDelegate = new MasterDelegate(TopologyManager.getInstance().<WorkMessage<DefaultWorkItem>, String>getOrCreateTopology(topologyName, topologyFactory), router);
        this.masterDelegate.start();
    }

    public WorkItem schedule(final Work work) {
        return schedule(work, null);
    }

    public WorkItem schedule(final Work work, final WorkListener listener) {
        return schedule(work, listener, false);
    }

    public WorkItem schedule(final Work work, final WorkListener listener, final boolean cancelWorkStatusTracking) {
        DefaultWorkItem workItem = new DefaultWorkItem(work, listener, cancelWorkStatusTracking);
        return schedule(workItem);
    }

    public boolean waitForAll(final Collection workItems, final long timeout) throws InterruptedException {
        synchronized (completedWorks) {
            long waitingTime = timeout;
            long startTime = 0;
            int numberOfCurrentlyCompletedWorks = 0;
            while ((numberOfCurrentlyCompletedWorks = getCurrentlyCompletedWorksFor(workItems).size()) < workItems.size() && waitingTime > 0) {
                startTime = System.currentTimeMillis();
                logger.info("Waiting for all {} work items with timeout {}", workItems.size(), timeout);
                completedWorks.wait(waitingTime);
                waitingTime = waitingTime - (System.currentTimeMillis() - startTime);
            }
            return numberOfCurrentlyCompletedWorks == workItems.size();
        }
    }

    public Collection waitForAny(final Collection workItems, final long timeout) throws InterruptedException {
        synchronized (completedWorks) {
            long waitingTime = timeout;
            long startTime = 0;
            List currentlyCompletedWorks = null;
            while ((currentlyCompletedWorks = getCurrentlyCompletedWorksFor(workItems)).size() == 0 && waitingTime > 0) {
                startTime = System.currentTimeMillis();
                logger.info("Waiting for all {} work items with timeout {}", workItems.size(), timeout);
                completedWorks.wait(waitingTime);
                waitingTime = waitingTime - (System.currentTimeMillis() - startTime);
            }
            return currentlyCompletedWorks;
        }
    }

    /**
     * Remove from the list of stored completed works, the given collection of works.<br>
     * This is a method outside of the standard {@link commonj.work.WorkManager} interface,
     * to avoid having completed works accumulated into the WorkManager itself.<br>
     * Please note that you will not be able to wait for works removed by this method.
     * @param workItems 
     */
    public void removeCompletedWorks(final Collection workItems) {
        synchronized (completedWorks) {
            for (Object obj : workItems) {
                WorkItem item = (WorkItem) obj;
                completedWorks.remove(item);
            }
        }
    }

    /**
     * Remove all stored completed works.<br>
     * This is a method outside of the standard {@link commonj.work.WorkManager} interface,
     * to avoid having completed works accumulated into the WorkManager itself.<br>
     * Please note that you will not be able to wait for all works completed before this method call.
     */
    public void clearCompletedWorks() {
        synchronized (completedWorks) {
            completedWorks.clear();
        }
    }

    /**
     * Shutdown this work manager instance.<br>
     * This is a method outside of the standard {@link commonj.work.WorkManager} interface,
     * used to stop the work manager instance and cleanup resources.<br>
     * Please note that you will not be able to schedule works after this method call.
     */
    public final void shutdown() {
        masterDelegate.shutdown();
    }

    private List getCurrentlyCompletedWorksFor(final Collection workItems) {
        List currentlyCompletedWorks = new ArrayList(workItems.size());
        for (Object obj : workItems) {
            WorkItem item = (WorkItem) obj;
            if (completedWorks.contains(item)) {
                currentlyCompletedWorks.add(item);
                logger.debug("Found completed work item: {}", item);
            }
        }
        return currentlyCompletedWorks;
    }

    private WorkItem schedule(DefaultWorkItem workItem) {
        workItem.setStatus(WorkEvent.WORK_ACCEPTED, null);
        masterDelegate.submit(workItem);
        return workItem;
    }

    private class MasterDelegate extends AbstractMaster<DefaultWorkItem> {

        public MasterDelegate(final Topology<WorkMessage<DefaultWorkItem>, String> topology, final Router router) {
            super(topology, router);
        }

        @Override
        protected void onComplete(WorkMessage<DefaultWorkItem> workMessage) {
            DefaultWorkItem workItem = workMessage.getWorkObject();
            logger.debug("Completed work item {} with state: {}", workItem, workItem.getStatus());
            synchronized (completedWorks) {
                completedWorks.add(workItem);
                completedWorks.notifyAll();
            }
        }

        @Override
        protected void onShutdown() {
            synchronized (completedWorks) {
                completedWorks.clear();
            }
        }
    }
}
