/**
 *
 * All content copyright (c) 2003-2008 Terracotta, Inc.,
 * except as may otherwise be noted in a separate copyright notice.
 * All rights reserved.
 *
 */
package org.terracotta.masterworker.cluster;

import com.tc.cluster.DsoCluster;
import com.tc.cluster.DsoClusterListener;
import com.tc.cluster.DsoClusterTopology;
import com.tc.cluster.exceptions.UnclusteredObjectException;
import com.tcclient.cluster.DsoNode;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Queue;
import java.util.Set;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.terracotta.modules.concurrent.collections.ConcurrentStringMap;

/**
 * Manage master/worker clustered state, more specifically:
 * <ul>
 * <li>Membership of master and worker instances per cluster node.</li>
 * <li>Tracking of submitted/unsubmitted works for each master instance.</li>
 * </ul>
 * This is a singleton object, providing different ClusterState instances depending
 * on the assigned name: that is, you can have different ClusterState instances, each one
 * with its unique name.
 */
public class ClusterState {

    private transient static final Logger logger = LoggerFactory.getLogger(ClusterState.class);
    // ClusterState instances:
    private static final Map<String, ClusterState> instances = new HashMap<String, ClusterState>();
    // Terracotta support for cluster management:
    private DsoCluster dsoCluster;
    // Transient list of cluster listeners:
    private transient final List<ClusterStateListener> clusterListeners = new CopyOnWriteArrayList<ClusterStateListener>();
    // Clustered lock guarding changes to the master/worker topology and related state:
    private final Lock stateLock = new ReentrantLock();
    // Master/Worker instances per node:
    private final Map<String, List<String>> mastersPerNode = new HashMap<String, List<String>>();
    private final Map<String, List<String>> workersPerNode = new HashMap<String, List<String>>();
    // Clustered locks, each one guarding access to the master it has been assigned to, so that each master lock
    // doesn't affect other locks:
    private final Map<String, Lock> locksPerMaster = new ConcurrentStringMap();
    // Submitted/unsubmitted works for each master:
    private final Map<String, Map<String, Queue>> submittedPendingWorksPerMaster = new ConcurrentStringMap();
    private final Map<String, Queue> unsubmittedPendingWorksPerMaster = new ConcurrentStringMap();
    // The use of a single, different, lock per master, and of ConsurrentStringMaps, should lead to lower
    // lock contention and no cluster-wide broadcasting: that is, while a master accesses its own works, other masters
    // can access their own works too, because protected by different locks.

    private ClusterState() {
    }

    public static ClusterState getOrCreateInstance(String name) {
        synchronized (instances) {
            if (instances.containsKey(name)) {
                return instances.get(name);
            } else {
                ClusterState created = new ClusterState();
                instances.put(name, created);
                return created;
            }
        }
    }

    public static ClusterState removeInstance(String name) {
        synchronized (instances) {
            if (instances.containsKey(name)) {
                return instances.remove(name);
            } else {
                return null;
            }
        }
    }

    public void registerClusterStateListener(ClusterStateListener listener) {
        getDsoCluster().addClusterListener(listener);
        clusterListeners.add(listener);
    }

    public void unregisterClusterStateListener(ClusterStateListener listener) {
        getDsoCluster().removeClusterListener(listener);
        clusterListeners.remove(listener);
    }

    public String getCurrentNodeId() {
        return getDsoCluster().getCurrentNode().getId();
    }

    public void addMasterForNode(String master, String node) {
        try {
            stateLock.lock();
            List<String> masters = mastersPerNode.get(node);
            if (masters == null) {
                masters = new LinkedList<String>();
                mastersPerNode.put(node, masters);
            }
            masters.add(master);
            locksPerMaster.put(master, new ReentrantLock());
            submittedPendingWorksPerMaster.put(master, new HashMap<String, Queue>());
            unsubmittedPendingWorksPerMaster.put(master, new LinkedList());
            logger.info("Added master {} for node {}", master, node);
        } finally {
            stateLock.unlock();
        }
        notifyOnMasterAdded(master, node);
    }

    public List<String> getMastersForNode(String node) {
        try {
            stateLock.lock();
            return mastersPerNode.get(node) != null ? Collections.unmodifiableList(mastersPerNode.get(node)) : new ArrayList<String>(0);
        } finally {
            stateLock.unlock();
        }
    }

    public void removeMasterForNode(String master, String node) {
        try {
            stateLock.lock();
            List<String> masters = mastersPerNode.get(node);
            if (masters != null) {
                Lock masterLock = locksPerMaster.get(master);
                try {
                    masterLock.lock();
                    masters.remove(master);
                    if (masters.isEmpty()) {
                        mastersPerNode.remove(node);
                    }
                    locksPerMaster.remove(master);
                    submittedPendingWorksPerMaster.remove(master);
                    unsubmittedPendingWorksPerMaster.remove(master);
                    logger.info("Removed master {} for node {}", master, node);
                } finally {
                    masterLock.unlock();
                }
            }
        } finally {
            stateLock.unlock();
        }
        notifyOnMasterRemoved(master, node);
    }

    public void addWorkerForNode(String worker, String node) {
        try {
            stateLock.lock();
            List<String> workers = workersPerNode.get(node);
            if (workers == null) {
                workers = new LinkedList<String>();
                workersPerNode.put(node, workers);
            }
            workers.add(worker);
            logger.info("Added worker {} for node {}", worker, node);
        } finally {
            stateLock.unlock();
        }
        notifyOnWorkerAdded(worker, node);
    }

    public void removeWorkerForNode(String worker, String node) {
        try {
            stateLock.lock();
            List<String> workers = workersPerNode.get(node);
            if (workers != null) {
                workers.remove(worker);
                if (workers.isEmpty()) {
                    workersPerNode.remove(node);
                }
            }
            logger.info("Removed worker {} for node {}", worker, node);
        } finally {
            stateLock.unlock();
        }
        notifyOnWorkerRemoved(worker, node);
    }

    public List<String> getWorkersForNode(String node) {
        try {
            stateLock.lock();
            List allWorkers = new LinkedList();
            if (workersPerNode.get(node) != null) {
                allWorkers.addAll(workersPerNode.get(node));
            }
            return allWorkers;
        } finally {
            stateLock.unlock();
        }
    }

    public List<String> getAllWorkers() {
        try {
            stateLock.lock();
            Set<String> allNodes = new HashSet<String>();
            List allWorkers = new LinkedList();
            allNodes.addAll(workersPerNode.keySet());
            for (String node : allNodes) {
                if (workersPerNode.get(node) != null) {
                    allWorkers.addAll(workersPerNode.get(node));
                }
            }
            return allWorkers;
        } finally {
            stateLock.unlock();
        }
    }

    public void clearDisconnectedState() {
        try {
            stateLock.lock();
            Collection<String> connectedNodes = new LinkedList<String>();
            for (DsoNode dsoNode : getDsoCluster().getClusterTopology().getNodes()) {
                connectedNodes.add(dsoNode.getId());
            }
            clearDisconnectedMasters(connectedNodes);
            clearDisconnectedWorkers(connectedNodes);
        } finally {
            stateLock.unlock();
        }
    }

    public Map<String, Queue> acquireSubmittedPendingWorksForMaster(String master) {
        Lock masterLock = locksPerMaster.get(master);
        if (masterLock != null) {
            masterLock.lock();
            Map<String, Queue> submittedPendingWorks = submittedPendingWorksPerMaster.get(master);
            return submittedPendingWorks;
        } else {
            throw new IllegalStateException("No master: " + master);
        }
    }

    public Queue acquireUnsubmittedPendingWorksForMaster(String master) {
        Lock masterLock = locksPerMaster.get(master);
        if (masterLock != null) {
            masterLock.lock();
            Queue unsubmittedPendingWorks = unsubmittedPendingWorksPerMaster.get(master);
            return unsubmittedPendingWorks;
        } else {
            throw new IllegalStateException("No master: " + master);
        }
    }

    public void releaseSubmittedPendingWorksForMaster(String master) {
        Lock masterLock = locksPerMaster.get(master);
        if (masterLock != null) {
            masterLock.unlock();
        } else {
            throw new IllegalStateException("No master: " + master);
        }
    }

    public void releaseUnsubmittedPendingWorksForMaster(String master) {
        Lock masterLock = locksPerMaster.get(master);
        if (masterLock != null) {
            masterLock.unlock();
        } else {
            throw new IllegalStateException("No master: " + master);
        }
    }

    public void copySubmittedPendingWorksForMasterTo(String master, List destination) {
        Lock masterLock = locksPerMaster.get(master);
        if (masterLock != null) {
            try {
                masterLock.lock();
                Map<String, Queue> submittedPendingWorks = submittedPendingWorksPerMaster.get(master);
                for (Collection works : submittedPendingWorks.values()) {
                    for (Object work : works) {
                        destination.add(work);
                    }
                }
            } finally {
                locksPerMaster.get(master).unlock();
            }
        } else {
            throw new IllegalStateException("No master: " + master);
        }
    }

    public void copyUnsubmittedPendingWorksForMasterTo(String master, List destination) {
        Lock masterLock = locksPerMaster.get(master);
        if (masterLock != null) {
            try {
                masterLock.lock();
                Queue unsubmittedPendingWorks = unsubmittedPendingWorksPerMaster.get(master);
                for (Object work : unsubmittedPendingWorks) {
                    destination.add(work);
                }
            } finally {
                locksPerMaster.get(master).unlock();
            }
        } else {
            throw new IllegalStateException("No master: " + master);
        }
    }

    private DsoCluster getDsoCluster() {
        if (dsoCluster != null) {
            return dsoCluster;
        } else {
            return new NoDsoCluster();
        }
    }

    private void clearDisconnectedMasters(Collection<String> connectedNodes) {
        Iterator<String> masterNodesIterator = mastersPerNode.keySet().iterator();
        while (masterNodesIterator.hasNext()) {
            String masterNode = masterNodesIterator.next();
            if (!connectedNodes.contains(masterNode)) {
                masterNodesIterator.remove();
                locksPerMaster.remove(masterNode);
                submittedPendingWorksPerMaster.remove(masterNode);
                unsubmittedPendingWorksPerMaster.remove(masterNode);
            }
        }
    }

    private void clearDisconnectedWorkers(Collection<String> connectedNodes) {
        Iterator<String> workerNodesIterator = workersPerNode.keySet().iterator();
        while (workerNodesIterator.hasNext()) {
            String workerNode = workerNodesIterator.next();
            if (!connectedNodes.contains(workerNode)) {
                workerNodesIterator.remove();
            }
        }
    }

    private void notifyOnMasterAdded(String master, String node) {
        for (ClusterStateListener listener : clusterListeners) {
            listener.masteAdded(new ClusterStateEvent(master, null, node));
        }
    }

    private void notifyOnMasterRemoved(String master, String node) {
        for (ClusterStateListener listener : clusterListeners) {
            listener.masterRemoved(new ClusterStateEvent(master, null, node));
        }
    }

    private void notifyOnWorkerAdded(String worker, String node) {
        for (ClusterStateListener listener : clusterListeners) {
            listener.workerAdded(new ClusterStateEvent(null, worker, node));
        }
    }

    private void notifyOnWorkerRemoved(String worker, String node) {
        for (ClusterStateListener listener : clusterListeners) {
            listener.workerRemoved(new ClusterStateEvent(null, worker, node));
        }
    }

    private class NoDsoCluster implements DsoCluster {

        private final DsoNode CURRENT_NODE = new DummyDsoNode("CURRENT_NODE");
        private final DsoClusterTopology dsoClusterTopology = new DummyDsoClusterTopology();

        public DsoNode getCurrentNode() {
            return CURRENT_NODE;
        }

        public DsoClusterTopology getClusterTopology() {
            return dsoClusterTopology;
        }

        public void addClusterListener(DsoClusterListener listener) {
        }

        public void removeClusterListener(DsoClusterListener listener) {
        }

        public boolean isNodeJoined() {
            throw new UnsupportedOperationException("Not supported yet.");
        }

        public boolean areOperationsEnabled() {
            throw new UnsupportedOperationException("Not supported yet.");
        }

        public Set<DsoNode> getNodesWithObject(Object arg0) throws UnclusteredObjectException {
            throw new UnsupportedOperationException("Not supported yet.");
        }

        public Map<?, Set<DsoNode>> getNodesWithObjects(Object... arg0) throws UnclusteredObjectException {
            throw new UnsupportedOperationException("Not supported yet.");
        }

        public Map<?, Set<DsoNode>> getNodesWithObjects(Collection<?> arg0) throws UnclusteredObjectException {
            throw new UnsupportedOperationException("Not supported yet.");
        }

        public <K> Set<K> getKeysForOrphanedValues(Map<K, ?> arg0) throws UnclusteredObjectException {
            throw new UnsupportedOperationException("Not supported yet.");
        }

        public <K> Set<K> getKeysForLocalValues(Map<K, ?> arg0) throws UnclusteredObjectException {
            throw new UnsupportedOperationException("Not supported yet.");
        }

        private class DummyDsoClusterTopology implements DsoClusterTopology {

            public Collection<DsoNode> getNodes() {
                Set<String> nodes = new HashSet<String>();
                for (String master : mastersPerNode.keySet()) {
                    nodes.add(master);
                }
                for (String worker : workersPerNode.keySet()) {
                    nodes.add(worker);
                }
                Collection<DsoNode> result = new ArrayList<DsoNode>(nodes.size());
                for (String node : nodes) {
                    result.add(new DummyDsoNode(node));
                }
                return result;
            }
        }

        private class DummyDsoNode implements DsoNode {

            private final String id;

            public DummyDsoNode(String id) {
                this.id = id;
            }

            public String getId() {
                return id;
            }

            public String getIp() {
                throw new UnsupportedOperationException("Not supported yet.");
            }

            public String getHostname() {
                throw new UnsupportedOperationException("Not supported yet.");
            }
        }
    }
}
