/**
 *
 * All content copyright (c) 2003-2008 Terracotta, Inc.,
 * except as may otherwise be noted in a separate copyright notice.
 * All rights reserved.
 *
 */
package org.terracotta.masterworker;

import java.util.LinkedList;
import java.util.List;
import org.terracotta.masterworker.cluster.ClusterState;
import org.terracotta.message.topology.Topology;

/**
 * Access to the clustered master/worker data, related to a specific {@link org.terracotta.message.topology.Topology}.<br>
 * This class may be used to implement data-affinity routing logic, as well as any other kind of logic which requires the knowledge
 * of the clustered master/worker topology and other related data.
 * <br><br>
 * This class is thread safe.
 */
public class ClusterData {

    private final ClusterState clusterState;

    /**
     * Create a ClusteData object to use for accessing information about a master/worker cluster.<br>
     * Please note that the ClusteData, masters and workers instances must share the same
     * {@link org.terracotta.message.topology.Topology} in order to communicate and actually share the same data.
     *
     * @param topology The topology to use for accessing master/worker clustered data.
     */
    public ClusterData(Topology topology) {
        this.clusterState = ClusterState.getOrCreateInstance(topology.getName());
    }

    /**
     * Get the identifiers of all active masters on the given node.
     *
     * @param node The identifier of the cluster node whose active masters you want to know.
     * @return The list of active masters.
     */
    public List<String> getMastersForNode(String node) {
        return clusterState.getMastersForNode(node);
    }

    /**
     * Get the identifiers of all active workers on the given node.
     *
     * @param node The identifier of the cluster node whose active workers you want to know.
     * @return The list of active workers.
     */
    public List<String> getWorkersForNode(String node) {
        return clusterState.getWorkersForNode(node);
    }

    /**
     * Get all pending submitted/unsubmitted works for the given master.
     *
     * @param master The identifier of the master whose pending works you want to know.
     * @return The list of pending works.
     */
    public List getPendingWorksForMaster(String master) {
        List allWorks = new LinkedList();
        clusterState.copySubmittedPendingWorksForMasterTo(master, allWorks);
        clusterState.copyUnsubmittedPendingWorksForMasterTo(master, allWorks);
        return allWorks;
    }
}
