/**
 *
 * All content copyright (c) 2003-2008 Terracotta, Inc.,
 * except as may otherwise be noted in a separate copyright notice.
 * All rights reserved.
 *
 */
package org.terracotta.masterworker;

import org.terracotta.masterworker.cluster.ClusterState;
import java.util.UUID;
import java.util.concurrent.ExecutorService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.terracotta.message.pipe.Pipe;
import org.terracotta.message.pipe.PipeProcessor;
import org.terracotta.message.topology.Topology;

/**
 * Abstract {@link Worker} implementation based on {@link org.terracotta.masterworker.cluster.ClusterState}.<br>
 * Each <code>AbstractWorker</code> instance adds itself to the proper {@link org.terracotta.masterworker.cluster.ClusterState}
 * instance, in order to communicate with other {@link Master}s sharing the same {@link org.terracotta.message.topology.Topology}.
 *
 * @param <T> Type of the works contained into the pipe to listen to.
 */
public abstract class AbstractWorker<T> implements Worker<T> {

    private transient static final Logger logger = LoggerFactory.getLogger(AbstractWorker.class);
    //
    private final ClusterState clusterState;
    //
    private final String workerId;
    //
    private final Topology<WorkMessage<T>, String> topology;
    private final ExecutorService workThreadPool;
    private Pipe<WorkMessage<T>> workPipe;
    private PipeProcessor<WorkMessage<T>> workPipeProcessor;

    /**
     * Abstract {@link Worker} constructor.
     *
     * @param prefix An optional string value to use as a prefix for this worker id; please note that
     * it is possible to assign the same prefix to more workers.
     * @param topology The {@link org.terracotta.message.topology.Topology} hosting the pipe
     * to listen to.
     * @param workThreadPool The thread pool to use for executing works.
     */
    public AbstractWorker(String prefix, Topology<WorkMessage<T>, String> topology, ExecutorService workThreadPool) {
        if (prefix != null && prefix.length() > 0) {
            this.workerId = prefix + "-" + UUID.randomUUID().toString();
        } else {
            this.workerId = UUID.randomUUID().toString();
        }
        this.topology = topology;
        this.workThreadPool = workThreadPool;
        this.clusterState = ClusterState.getOrCreateInstance(topology.getName());
    }

    public final void start() throws Exception {
        logger.info("Starting worker for routing id {}", workerId);
        onStart();
        workPipe = topology.getOrCreatePipeFor(workerId);
        workPipeProcessor = new WorkPipeListener(workPipe);
        workPipeProcessor.start();
        initClusterState();
    }

    public final void stop() {
        logger.info("Stopping worker for routing id {}", workerId);
        clearClusterState();
        workPipeProcessor.stop();
        workThreadPool.shutdown();
        onStop();
    }

    /**
     * Call in subclasses to reply the message to the worker.
     */
    protected final void doReply(WorkMessage<T> workMessage) {
        try {
            String replyPipeID = workMessage.getReplyPipeRoutingID();
            logger.debug("Replying work to pipe {}", replyPipeID);
            Pipe<WorkMessage<T>> replyPipe = topology.getPipeFor(replyPipeID);
            if (replyPipe != null) {
                replyPipe.put(workMessage);
            } else {
                logger.warn("Reply pipe {} not found!", replyPipeID);
            }
        } catch (InterruptedException ex) {
            logger.error("Exception while putting work into reply pipe {}!", workMessage.getReplyPipeRoutingID());
            logger.error(ex.getMessage(), ex);
        }
    }

    /**
     * Override in subclasses to execute custom logic on worker start invoked through {@link #start()}.
     */
    protected void onStart() throws Exception {
    }

    /**
     * Override in subclasses to execute custom logic on worker stop invoked through {@link #stop()}.
     */
    protected void onStop() {
    }

    /**
     * Implement in subclasses for actual work execution.
     */
    protected abstract void doExecute(WorkMessage<T> workMessage);

    private void initClusterState() {
        clusterState.addWorkerForNode(workerId, clusterState.getCurrentNodeId());
    }

    private void clearClusterState() {
        clusterState.removeWorkerForNode(workerId, clusterState.getCurrentNodeId());
    }

    private class WorkPipeListener extends PipeProcessor<WorkMessage<T>> {

        public WorkPipeListener(Pipe<WorkMessage<T>> workPipe) {
            super(workPipe, false);
        }

        public boolean event(final WorkMessage<T> workMessage) throws Exception {
            logger.debug("Executing work {} in pipe {}", workMessage.getWorkObject(), workerId);
            workThreadPool.submit(new Runnable() {

                public void run() {
                    doExecute(workMessage);
                }
            });
            return true;
        }
    }
}
