/**
 *
 * All content copyright (c) 2003-2008 Terracotta, Inc.,
 * except as may otherwise be noted in a separate copyright notice.
 * All rights reserved.
 *
 */
package org.terracotta.masterworker;

import com.tc.cluster.DsoClusterEvent;
import java.util.ArrayList;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Queue;
import java.util.UUID;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.terracotta.masterworker.cluster.ClusterState;
import org.terracotta.masterworker.cluster.ClusterStateEvent;
import org.terracotta.masterworker.cluster.ClusterStateListener;
import org.terracotta.message.pipe.Pipe;
import org.terracotta.message.pipe.PipeProcessor;
import org.terracotta.message.routing.Route;
import org.terracotta.message.routing.Router;
import org.terracotta.message.routing.RoutingFilter;
import org.terracotta.message.topology.Topology;

/**
 * Abstract {@link Master} implementation based on {@link org.terracotta.masterworker.cluster.ClusterState}.<br>
 * Each <code>AbstractMaster</code> instance adds itself to the proper {@link org.terracotta.masterworker.cluster.ClusterState}
 * instance, in order to communicate with other {@link Worker}s sharing the same {@link org.terracotta.message.topology.Topology}.
 *
 * @param <T> Type of the work object to submit.
 */
public abstract class AbstractMaster<T> implements Master<T> {

    private transient static final Logger logger = LoggerFactory.getLogger(AbstractMaster.class);
    //
    private final ClusterState clusterState;
    private final ClusterStateListener masterClusterListener;
    //
    private final String masterId;
    //
    private final Router router;
    private final Topology<WorkMessage<T>, String> topology;
    private final String replyPipeID;
    private final PipeProcessor<WorkMessage<T>> replyPipeProcessor;
    private final List<String> registeredRoutingIDs;
    private final RoutingFilter masterRoutingFilter;
    //
    private volatile boolean started;
    private volatile boolean shutdown;
    private volatile int totalPending;

    /**
     * Abstract {@link Master} constructor.
     *
     * @param topology The {@link org.terracotta.message.topology.Topology} backing this master.
     * @param router The {@link org.terracotta.message.routing.Router} used to route submitted works.
     */
    public AbstractMaster(Topology<WorkMessage<T>, String> topology, Router router) {
        this.masterId = UUID.randomUUID().toString();
        this.router = router;
        this.topology = topology;
        this.replyPipeID = masterId + "-replyPipe";
        this.replyPipeProcessor = new ReplyPipeListener(topology.getOrCreatePipeFor(replyPipeID), replyPipeID);
        this.registeredRoutingIDs = new LinkedList<String>();
        this.masterRoutingFilter = new MasterRoutingFilter();
        this.clusterState = ClusterState.getOrCreateInstance(topology.getName());
        this.masterClusterListener = new MasterClusterListener();
    }

    public final synchronized void start() {
        if (!started && !shutdown) {
            initClusterState();
            List<String> workers = clusterState.getAllWorkers();
            for (String worker : workers) {
                doRegister(worker);
            }
            replyPipeProcessor.start();
            started = true;
        }
    }

    public final synchronized T submit(final T work) {
        if (started && !shutdown) {
            doSubmit(work);
            return work;
        } else {
            return null;
        }
    }

    public final synchronized void register(String routingID) {
        if (started && !shutdown) {
            doRegister(routingID);
            flushUnsubmittedWorks();
        }
    }

    public final synchronized void unregister(String routingID) {
        if (started && !shutdown) {
            doUnregister(routingID);
            reRoutePendingWorks(routingID);
        }
    }

    public final synchronized void shutdown() {
        if (started && !shutdown) {
            started = false;
            shutdown = true;
            onShutdown();
            Thread cleanupThread = new CleanupThread();
            cleanupThread.start();
        }
    }

    public final synchronized List<T> shutdownNow() {
        if (started && !shutdown) {
            started = false;
            shutdown = true;
            List pendingWorks = new LinkedList();
            clusterState.copySubmittedPendingWorksForMasterTo(masterId, pendingWorks);
            clusterState.copyUnsubmittedPendingWorksForMasterTo(masterId, pendingWorks);
            onShutdown();
            cleanupNow();
            return pendingWorks;
        } else {
            return new ArrayList<T>(0);
        }
    }

    public final boolean awaitTermination(long timeout, TimeUnit unit) throws InterruptedException {
        if (shutdown) {
            long waitingTime = TimeUnit.MILLISECONDS.convert(timeout, unit);
            long startTime = 0;
            synchronized (this) {
                while (totalPending > 0 && waitingTime > 0) {
                    startTime = System.currentTimeMillis();
                    logger.debug("Waiting {} millis for termination ...", waitingTime);
                    wait(waitingTime);
                    waitingTime = waitingTime - (System.currentTimeMillis() - startTime);
                }
                return totalPending == 0;
            }
        } else {
            return false;
        }
    }

    public final boolean isShutdown() {
        return shutdown;
    }

    public final boolean isTerminated() {
        return shutdown && totalPending == 0;
    }

    /**
     * Override in subclasses to react everytime the master receives a completed work from a worker on the reply pipe.
     */
    protected void onComplete(WorkMessage<T> workMessage) {
    }

    /**
     * Override in subclasses to execute custom logic on master shutdown invoked through {@link #shutdown()}.
     */
    protected void onShutdown() {
    }

    private void initClusterState() {
        clusterState.clearDisconnectedState();
        clusterState.addMasterForNode(masterId, clusterState.getCurrentNodeId());
        clusterState.registerClusterStateListener(masterClusterListener);
    }

    private void clearClusterState() {
        clusterState.unregisterClusterStateListener(masterClusterListener);
        clusterState.removeMasterForNode(masterId, clusterState.getCurrentNodeId());
    }

    private void doSubmit(final T work) {
        Route<WorkMessage<T>, String> route = topology.getRouteFor(router, masterRoutingFilter, work);
        if (route != null) {
            Pipe<WorkMessage<T>> workPipe = route.getPipe();
            String workPipeID = route.getRoutingID();
            WorkMessage<T> workMessage = new WorkMessage<T>(work, workPipeID, replyPipeID);
            boolean submitted = workPipe.offer(workMessage);
            if (submitted) {
                logger.debug("Submitted work {} on pipe {}", work, workPipeID);
                addToSubmittedPendingWorks(workPipeID, work);
            } else {
                logger.debug("Scheduled work {} for later submission (apparently full queue).", work);
                addToUnsubmittedPendingWorks(work);
            }
        } else {
            logger.debug("Scheduled work {} for later submission (apparently no route).", work);
            addToUnsubmittedPendingWorks(work);
        }
        logger.debug("There are {} total pending works", totalPending);
    }

    private void doRegister(String routingID) {
        if (!registeredRoutingIDs.contains(routingID)) {
            logger.info("Registering pipe for {}", routingID);
            registeredRoutingIDs.add(routingID);
        }
    }

    private void doUnregister(String routingID) {
        if (registeredRoutingIDs.contains(routingID)) {
            logger.info("Unregistering pipe for {}", routingID);
            registeredRoutingIDs.remove(routingID);
        }
    }

    private void reRoutePendingWorks(String routingID) {
        try {
            Map<String, Queue> submittedPendingWorks = clusterState.acquireSubmittedPendingWorksForMaster(masterId);
            Queue pendingQueue = submittedPendingWorks.get(routingID);
            if (pendingQueue != null && !pendingQueue.isEmpty()) {
                logger.info("Rerouting {} pending works for {}", pendingQueue.size(), routingID);
                int pendings = pendingQueue.size();
                for (int i = 0; i < pendings; i++) {
                    T work = (T) pendingQueue.poll();
                    totalPending--;
                    doSubmit(work);
                }
            }
        } finally {
            clusterState.releaseSubmittedPendingWorksForMaster(masterId);
        }
    }

    private void addToSubmittedPendingWorks(String routingID, T work) {
        try {
            Map<String, Queue> submittedPendingWorks = clusterState.acquireSubmittedPendingWorksForMaster(masterId);
            Queue pendingQueue = submittedPendingWorks.get(routingID);
            if (pendingQueue == null) {
                pendingQueue = new LinkedList();
                submittedPendingWorks.put(routingID, pendingQueue);
            }
            pendingQueue.offer(work);
            totalPending++;
            logger.debug("There are {} submitted pending works", pendingQueue.size());
        } finally {
            clusterState.releaseSubmittedPendingWorksForMaster(masterId);
        }
    }

    private void removeFromSubmittedPendingWorks(String routingID, T work) {
        try {
            Map<String, Queue> submittedPendingWorks = clusterState.acquireSubmittedPendingWorksForMaster(masterId);
            Queue pendingQueue = submittedPendingWorks.get(routingID);
            if (pendingQueue != null) {
                pendingQueue.remove(work);
                totalPending--;
                logger.debug("There are {} submitted pending works", pendingQueue.size());
                notifyAll();
            }
        } finally {
            clusterState.releaseSubmittedPendingWorksForMaster(masterId);
        }
    }

    private void addToUnsubmittedPendingWorks(T work) {
        try {
            Queue unsubmittedPendingWorks = clusterState.acquireUnsubmittedPendingWorksForMaster(masterId);
            unsubmittedPendingWorks.offer(work);
            totalPending++;
            logger.debug("There are {} unsubmitted pending works", unsubmittedPendingWorks.size());
        } finally {
            clusterState.releaseUnsubmittedPendingWorksForMaster(masterId);
        }
    }

    private void flushUnsubmittedWorks() {
        try {
            Queue unsubmittedPendingWorks = clusterState.acquireUnsubmittedPendingWorksForMaster(masterId);
            int unsubmitteds = unsubmittedPendingWorks.size();
            for (int i = 0; i < unsubmitteds; i++) {
                T work = (T) unsubmittedPendingWorks.peek();
                if (work != null) {
                    unsubmittedPendingWorks.remove(work);
                    totalPending--;
                    logger.debug("Flushing, there are still {} unsubmitted pending works", unsubmittedPendingWorks.size());
                    doSubmit(work);
                } else {
                    throw new IllegalStateException();
                }
            }
        } finally {
            clusterState.releaseUnsubmittedPendingWorksForMaster(masterId);
        }
    }

    private void waitAndCleanup() throws InterruptedException {
        synchronized (this) {
            while (totalPending > 0) {
                wait();
            }
        }
        replyPipeProcessor.stop();
        topology.removePipeFor(replyPipeID);
        clearClusterState();
    }

    private void cleanupNow() {
        replyPipeProcessor.stop();
        topology.removePipeFor(replyPipeID);
        clearClusterState();
    }

    private class MasterRoutingFilter implements RoutingFilter {

        public <V, ID> boolean accept(Pipe<V> pipe, ID routingID) {
            if (registeredRoutingIDs.contains(routingID)) {
                return true;
            } else {
                return false;
            }
        }
    }

    private class MasterClusterListener implements ClusterStateListener {

        private final Lock singleResourceLock = new ReentrantLock();

        public void workerAdded(ClusterStateEvent event) {
            register(event.getWorker());
        }

        public void workerRemoved(ClusterStateEvent event) {
            unregister(event.getWorker());
        }

        public void nodeLeft(DsoClusterEvent event) {
            String nodeLeftId = event.getNode().getId();
            logger.info("Node left: {}", nodeLeftId);
            try {
                singleResourceLock.lock();
                List<String> masters = clusterState.getMastersForNode(nodeLeftId);
                if (!masters.isEmpty()) {
                    for (String masterId : masters) {
                        logger.info("Resubmitting works for master: {}", masterId);
                        resubmitForMaster(masterId);
                        clusterState.removeMasterForNode(masterId, nodeLeftId);
                    }
                }
                List<String> workers = clusterState.getWorkersForNode(nodeLeftId);
                if (!workers.isEmpty()) {
                    for (String workerId : workers) {
                        clusterState.removeWorkerForNode(workerId, nodeLeftId);
                    }
                }
            } finally {
                singleResourceLock.unlock();
            }
        }

        public void masteAdded(ClusterStateEvent event) {
        }

        public void masterRemoved(ClusterStateEvent event) {
        }

        public void nodeJoined(DsoClusterEvent event) {
        }

        public void operationsEnabled(DsoClusterEvent event) {
        }

        public void operationsDisabled(DsoClusterEvent event) {
        }

        private void resubmitForMaster(String masterId) {
            List pendingWorks = new LinkedList();
            clusterState.copySubmittedPendingWorksForMasterTo(masterId, pendingWorks);
            clusterState.copyUnsubmittedPendingWorksForMasterTo(masterId, pendingWorks);
            logger.info("Works to resubmit: {}", pendingWorks.size());
            for (Object work : pendingWorks) {
                submit((T) work);
            }
        }
    }

    private class ReplyPipeListener extends PipeProcessor<WorkMessage<T>> {

        private final String replyPipeID;

        public ReplyPipeListener(final Pipe<WorkMessage<T>> replyPipe, String replyPipeID) {
            super(replyPipe, true);
            this.replyPipeID = replyPipeID;
        }

        @Override
        public boolean event(WorkMessage<T> workMessage) throws Exception {
            synchronized (AbstractMaster.this) {
                onComplete(workMessage);
                removeFromSubmittedPendingWorks(workMessage.getWorkPipeRoutingID(), workMessage.getWorkObject());
                flushUnsubmittedWorks();
            }
            return true;
        }
    }

    private class CleanupThread extends Thread {

        @Override
        public void run() {
            try {
                waitAndCleanup();
            } catch (InterruptedException ex) {
                logger.warn("Error executing master termination: " + ex.getMessage(), ex);
            }
        }
    }
}
