/**
 * 
 * All content copyright (c) 2003-2008 Terracotta, Inc.,
 * except as may otherwise be noted in a separate copyright notice.
 * All rights reserved.
 *
 */
package org.terracotta.executor;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.concurrent.AbstractExecutorService;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.RejectedExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.terracotta.executor.support.DefaultDistributedWork;
import org.terracotta.executor.support.DistributedWork;
import org.terracotta.masterworker.AbstractMaster;
import org.terracotta.masterworker.Master;
import org.terracotta.masterworker.WorkMessage;
import org.terracotta.message.routing.RoundRobinRouter;
import org.terracotta.message.routing.Router;
import org.terracotta.message.topology.Topology;
import org.terracotta.message.topology.TopologyManager;

/**
 * {@link java.util.concurrent.ExecutorService} implementation based on the master/worker pattern: submitted tasks
 * are actually executed by external {@link DistributedWorkerService} worker objects.<br>
 * <code>DistributedExecutorService</code> and <code>DistributedWorkerService</code> instances must share the same 
 * {@link org.terracotta.message.topology.Topology} in order to collaborate each other.
 */
public class DistributedExecutorService extends AbstractExecutorService {

    private transient static final Logger logger = LoggerFactory.getLogger(DistributedExecutorService.class);    
    private final Master<DistributedWork> masterDelegate;

    public DistributedExecutorService(final String topologyName) {
        this(topologyName, new RoundRobinRouter());
    }

    public DistributedExecutorService(final String topologyName, final Router router) {
        this(topologyName, null, router);

    }

    public DistributedExecutorService(final String topologyName, final Topology.Factory topologyFactory, final Router router) {
        Topology topology = TopologyManager.getInstance().<WorkMessage<DistributedWork>, String>getOrCreateTopology(topologyName, topologyFactory);
        this.masterDelegate = new MasterDelegate(topology, router);
        this.masterDelegate.start();
    }

    @Override
    public List invokeAll(Collection callables) throws InterruptedException {
        checkShutdown();
        return super.invokeAll(callables);
    }

    @Override
    public List invokeAll(Collection callables, long timeout, TimeUnit unit) throws InterruptedException {
        checkShutdown();
        return super.invokeAll(callables, timeout, unit);
    }

    @Override
    public Object invokeAny(Collection callables) throws InterruptedException, ExecutionException {
        checkShutdown();
        return super.invokeAny(callables);
    }

    @Override
    public Object invokeAny(Collection callables, long timeout, TimeUnit unit) throws InterruptedException, ExecutionException, TimeoutException {
        checkShutdown();
        return super.invokeAny(callables, timeout, unit);
    }

    @Override
    public Future submit(Runnable runnable) {
        checkShutdown();
        return super.submit(runnable);
    }

    @Override
    public Future submit(Runnable runnable, Object result) {
        checkShutdown();
        return super.submit(runnable, result);
    }

    @Override
    public Future submit(Callable callable) {
        checkShutdown();
        return super.submit(callable);
    }

    public void execute(Runnable work) {
        executeDistributedWork(new DefaultDistributedWork(work));
    }

    public List<Runnable> shutdownNow() {
        List<DistributedWork> pendingWorks = masterDelegate.shutdownNow();
        List<Runnable> result = new ArrayList<Runnable>(pendingWorks.size());
        for (DistributedWork work : pendingWorks) {
            result.add(work);
        }
        return result;
    }

    public void shutdown() {
        masterDelegate.shutdown();
    }

    public boolean isTerminated() {
        return masterDelegate.isTerminated();
    }

    public boolean isShutdown() {
        return masterDelegate.isShutdown();
    }

    public boolean awaitTermination(long timeout, TimeUnit unit) throws InterruptedException {
        return masterDelegate.awaitTermination(timeout, unit);
    }
    
    /**
     * Package-access to let other executor classes submit {@link org.terracotta.executor.support.DistributedWork}s.
     */
    void executeDistributedWork(DistributedWork work) {
        try {
            masterDelegate.submit(work);
        } catch (Exception ex) {
            logger.error(ex.getMessage(), ex);
            throw new RuntimeException("Unexpected exception!", ex);
        }
    }

    private void checkShutdown() {
        if (masterDelegate.isShutdown()) {
            throw new RejectedExecutionException("This executor service has been shutdown.");
        }
    }

    private class MasterDelegate extends AbstractMaster<DistributedWork> {

        public MasterDelegate(final Topology<WorkMessage<DistributedWork>, String> topology, final Router router) {
            super(topology, router);
        }

        @Override
        protected void onComplete(WorkMessage<DistributedWork> workMessage) {
            try {
                workMessage.getWorkObject().onComplete();
            } catch (InterruptedException ie) {
                logger.warn("Interrupted: " + ie);
            } catch(Exception ex) {
                logger.warn(ex.getMessage(), ex);
            }
        }
    }
}
