/**
 * 
 * All content copyright (c) 2003-2008 Terracotta, Inc.,
 * except as may otherwise be noted in a separate copyright notice.
 * All rights reserved.
 *
 */
package org.terracotta.executor;

import java.util.concurrent.BlockingQueue;
import java.util.concurrent.Callable;
import java.util.concurrent.CompletionService;
import java.util.concurrent.Future;
import java.util.concurrent.FutureTask;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.TimeUnit;
import org.terracotta.executor.support.DefaultDistributedWork;

/**
 * {@link java.util.concurrent.CompletionService} implementation which uses a {@link DistributedExecutorService} to execute works.
 * 
 * @param <T> 
 */
public class DistributedCompletionService<T> implements CompletionService<T> {

    private final DistributedExecutorService executorDelegate;
    private final BlockingQueue<FutureTask<T>> completedWorksQueue;

    public DistributedCompletionService(DistributedExecutorService executor) {
        this.executorDelegate = executor;
        this.completedWorksQueue = new LinkedBlockingQueue<FutureTask<T>>();
    }

    public Future<T> poll() {
        return completedWorksQueue.poll();
    }

    public Future<T> poll(long timeout, TimeUnit unit) throws InterruptedException {
        return completedWorksQueue.poll(timeout, unit);
    }

    public Future<T> take() throws InterruptedException {
        return completedWorksQueue.take();
    }

    public Future<T> submit(Callable<T> callable) {
        if (callable == null) {
            throw new NullPointerException();
        }
        FutureTask<T> f = new FutureTask<T>(callable);
        executorDelegate.executeDistributedWork(new CompletableDistributedWork<T>(completedWorksQueue, f));
        return f;
    }

    public Future<T> submit(Runnable runnable, T result) {
        if (runnable == null) {
            throw new NullPointerException();
        }
        FutureTask<T> f = new FutureTask<T>(runnable, result);
        executorDelegate.executeDistributedWork(new CompletableDistributedWork<T>(completedWorksQueue, f));
        return f;
    }

    private static class CompletableDistributedWork<T> extends DefaultDistributedWork {

        private final BlockingQueue<FutureTask<T>> queue;
        private final FutureTask<T> task;

        public CompletableDistributedWork(BlockingQueue<FutureTask<T>> queue, FutureTask<T> task) {
            super(task);
            this.queue = queue;
            this.task = task;
        }

        @Override
        public void onComplete() throws Exception {
            super.onComplete();
            queue.put(task);
        }
    }
}
