/*
 * All content copyright (c) 2003-2009 Terracotta, Inc., except as may otherwise be noted in a separate copyright
 * notice. All rights reserved.
 */
package org.terracotta.cache.impl;

import org.terracotta.cache.CacheConfig;
import org.terracotta.cache.CacheConfigFactory;
import org.terracotta.cache.DistributedCache;
import org.terracotta.cache.evictor.CapacityEvictionPolicyData;
import org.terracotta.cache.evictor.CapacityEvictionPolicyData.Factory;

/**
 * An immutable configuration that can be handed to a {@link DistributedMap} implementation. It has no predefined
 * configuration - you must pass it whatever it needs.
 */
public final class ImmutableConfig implements CacheConfig {

  // Basic
  private final String                             name;                             // map name for logging

  // Expiration
  private final int                                maxTTISeconds;                    // max Time To Live till expire
  private final int                                maxTTLSeconds;                    // max Time To Idle till expire

  // Orphan eviction
  private final boolean                            orphanEvictionEnabled;            // detect and evict orphaned
  // elements
  private final int                                orphanEvictionPeriod;             // # of times to run local
  // eviction before
  // doing
  // orphan

  // Logging
  private final boolean                            loggingEnabled;                   // Basic map logging

  // Max capacity eviction
  private final int                                targetMaxInMemoryCount;           // target max in-memory count
  private final int                                targetMaxTotalCount;              // target max total count

  private final CapacityEvictionPolicyData.Factory capacityEvictionPolicyDataFactory;

  public ImmutableConfig(String name, boolean loggingEnabled, int maxTTISeconds, int maxTTLSeconds,
                         boolean orphanEvictionEnabled, int orphanEvictionPeriod, int targetMaxInMemoryCount,
                         int targetMaxTotalCount, CapacityEvictionPolicyData.Factory capacityEvictionPolicyDataFactory) {

    this.name = name;
    this.loggingEnabled = loggingEnabled;
    this.maxTTISeconds = maxTTISeconds;
    this.maxTTLSeconds = maxTTLSeconds;
    this.orphanEvictionEnabled = orphanEvictionEnabled;
    this.orphanEvictionPeriod = orphanEvictionPeriod;
    this.targetMaxInMemoryCount = targetMaxInMemoryCount;
    this.targetMaxTotalCount = targetMaxTotalCount;
    this.capacityEvictionPolicyDataFactory = capacityEvictionPolicyDataFactory;
  }

  public ImmutableConfig(CacheConfig config) {
    this.name = config.getName();
    this.loggingEnabled = config.isLoggingEnabled();
    this.maxTTISeconds = config.getMaxTTISeconds();
    this.maxTTLSeconds = config.getMaxTTLSeconds();
    this.orphanEvictionEnabled = config.isOrphanEvictionEnabled();
    this.orphanEvictionPeriod = config.getOrphanEvictionPeriod();
    this.targetMaxInMemoryCount = config.getTargetMaxInMemoryCount();
    this.targetMaxTotalCount = config.getTargetMaxTotalCount();
    this.capacityEvictionPolicyDataFactory = config.getCapacityEvictionPolicyDataFactory();
  }

  public String getName() {
    return name;
  }

  public int getMaxTTISeconds() {
    return maxTTISeconds;
  }

  public int getMaxTTLSeconds() {
    return maxTTLSeconds;
  }

  public boolean isOrphanEvictionEnabled() {
    return orphanEvictionEnabled;
  }

  public int getOrphanEvictionPeriod() {
    return orphanEvictionPeriod;
  }

  public boolean isLoggingEnabled() {
    return loggingEnabled;
  }

  public boolean isEvictionEnabled() {
    return getMaxTTISeconds() > 0 || getMaxTTLSeconds() > 0;
  }

  public int getTargetMaxInMemoryCount() {
    return targetMaxInMemoryCount;
  }

  public int getTargetMaxTotalCount() {
    return targetMaxTotalCount;
  }

  public boolean refresh() {
    return false;
  }

  public Factory getCapacityEvictionPolicyDataFactory() {
    return capacityEvictionPolicyDataFactory;
  }

  public void waitForChange(long maxWait) {
    try {
      Thread.sleep(maxWait);
    } catch (InterruptedException ignore) {
      // that's ok
    }
  }

  @Override
  public int hashCode() {
    final int prime = 31;
    int result = 1;
    result = prime * result + (loggingEnabled ? 1231 : 1237);
    result = prime * result + maxTTISeconds;
    result = prime * result + maxTTLSeconds;
    result = prime * result + ((name == null) ? 0 : name.hashCode());
    result = prime * result + (orphanEvictionEnabled ? 1231 : 1237);
    result = prime * result + orphanEvictionPeriod;
    result = prime * result + targetMaxInMemoryCount;
    result = prime * result + targetMaxTotalCount;
    return result;
  }
  
  public boolean equals(Object o) {
    if (!(o instanceof ImmutableConfig)) { return false; }

    ImmutableConfig config = (ImmutableConfig) o;

    return getName().equals(config.getName()) && (getMaxTTISeconds() == config.getMaxTTISeconds())
           && (getMaxTTLSeconds() == config.getMaxTTLSeconds())
           && (isOrphanEvictionEnabled() == config.isOrphanEvictionEnabled())
           && (getOrphanEvictionPeriod() == config.getOrphanEvictionPeriod())
           && (isLoggingEnabled() == config.isLoggingEnabled())
           && (getTargetMaxInMemoryCount() == config.getTargetMaxInMemoryCount())
           && (getTargetMaxTotalCount() == config.getTargetMaxTotalCount());
  }

  public <K, V> DistributedCache<K, V> newCache() {
    MutableConfig copyConfig = new MutableConfig(name, loggingEnabled, maxTTISeconds, maxTTLSeconds,
                                                 orphanEvictionEnabled, orphanEvictionPeriod, targetMaxInMemoryCount,
                                                 targetMaxTotalCount, capacityEvictionPolicyDataFactory);

    if (CacheConfigFactory.DSO_ACTIVE) {
      return new DistributedCacheImpl<K, V>(copyConfig);
    } else {
      return new LocalCache<K, V>(copyConfig);
    }
  }

  public CacheConfig setLoggingEnabled(boolean loggingEnabled) {
    throw new UnsupportedOperationException();
  }

  public CacheConfig setMaxTTISeconds(int maxTTISeconds) {
    throw new UnsupportedOperationException();
  }

  public CacheConfig setMaxTTLSeconds(int maxTTLSeconds) {
    throw new UnsupportedOperationException();
  }

  public CacheConfig setName(String name) {
    throw new UnsupportedOperationException();
  }

  public CacheConfig setOrphanEvictionEnabled(boolean orphanEvictionEnabled) {
    throw new UnsupportedOperationException();
  }

  public CacheConfig setOrphanEvictionPeriod(int orphanEvictionPeriod) {
    throw new UnsupportedOperationException();
  }

  public CacheConfig setTargetMaxInMemoryCount(int targetMaxInMemoryCount) {
    throw new UnsupportedOperationException();
  }

  public CacheConfig setTargetMaxTotalCount(int targetMaxTotalCount) {
    throw new UnsupportedOperationException();
  }

  public CacheConfig setCapacityEvictionPolicyDataFactory(Factory type) {
    throw new UnsupportedOperationException();
  }
}
