/*
 * All content copyright (c) Terracotta, Inc., except as may otherwise be noted in a separate copyright
 * notice. All rights reserved.
 */
package org.terracotta.cache.impl;

import org.terracotta.cache.DistributedCache;
import org.terracotta.cache.TimestampedValue;

import java.util.AbstractSet;
import java.util.Iterator;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.Set;

class EntrySet<K, V> extends AbstractSet<Map.Entry<K, V>> {

  private final DistributedCache<K, V>                   parent;
  private final Set<Map.Entry<K, TimestampedValue<V>>> dataEntrySet;

  EntrySet(DistributedCache<K, V> parent, Set<Map.Entry<K, TimestampedValue<V>>> dataEntrySet) {
    this.parent = parent;
    this.dataEntrySet = dataEntrySet;
  }

  @Override
  public Iterator<Map.Entry<K, V>> iterator() {
    return new Itr(dataEntrySet.iterator());
  }

  @Override
  public int size() {
    return parent.size();
  }

  private class Itr implements Iterator<Map.Entry<K, V>> {

    private final Iterator<Map.Entry<K, TimestampedValue<V>>> iterator;
    private Map.Entry<K, TimestampedValue<V>>                 current;

    Itr(Iterator<Map.Entry<K, TimestampedValue<V>>> iterator) {
      this.iterator = iterator;
      this.current = null;
    }

    public boolean hasNext() {
      return iterator.hasNext();
    }

    public Map.Entry<K, V> next() {
      try {
        current = iterator.next();
      } catch (NoSuchElementException nse) {
        current = null;
        throw nse;
      }

      return new Entry(current.getKey());
    }

    public void remove() {
      if (current == null) throw new IllegalStateException();
      parent.removeNoReturn(current.getKey());
      current = null;
    }
  }

  private class Entry implements Map.Entry<K, V> {

    private final K key;

    Entry(K key) {
      this.key = key;
    }

    public K getKey() {
      return key;
    }

    public V getValue() {
      // Calling back on the parent to get the isExpired() logic and to obtain locks on read
      return parent.get(key);
    }

    public V setValue(V value) {
      return parent.put(key, value);
    }
  }
}