/*
 * All content copyright (c) 2003-2009 Terracotta, Inc., except as may otherwise be noted in a separate copyright
 * notice. All rights reserved.
 */
package org.terracotta.cache.impl;

import org.terracotta.cache.CacheConfig;
import org.terracotta.cache.DistributedCache;
import org.terracotta.cache.TimestampedValue;
import org.terracotta.cache.evictor.CapacityEvictionPolicyData;
import org.terracotta.cache.evictor.Evictable;
import org.terracotta.cache.evictor.EvictionScheduler;
import org.terracotta.cache.evictor.EvictionStatistics;
import org.terracotta.cache.evictor.Evictor;
import org.terracotta.cache.evictor.EvictorLock;
import org.terracotta.cache.evictor.OrphanEvictionListener;
import org.terracotta.cache.evictor.TargetCapacityMapSizeListener;
import org.terracotta.cache.value.DefaultTimestampedValue;
import org.terracotta.collections.ConcurrentDistributedMap;
import org.terracotta.collections.FinegrainedLock;

import com.tc.cluster.DsoCluster;
import com.tc.logging.TCLogger;
import com.tc.object.bytecode.Manager;
import com.tc.object.bytecode.ManagerUtil;
import com.tc.object.bytecode.NotClearable;
import com.tc.util.Assert;

import java.util.Collection;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;

public class DistributedCacheImpl<K, V> implements DistributedCache<K, V>, Evictable<K>, NotClearable {

  private static final TCLogger                                    logger            = ManagerUtil
                                                                                         .getLogger(DistributedCacheImpl.class
                                                                                             .getName());

  // Provided configuration
  private final CacheConfig                                        config;

  // The actual data
  protected final ConcurrentDistributedMap<K, TimestampedValue<V>> data;

  // Shared orphan evictor lock
  private final EvictorLock                                        orphanEvictorLock;

  // Boolean indicating whether statistics are enabled
  private boolean                                                  statisticsEnabled = false;

  // Shared orphan eviction listener
  private OrphanEvictionListener<K>                                orphanEvictor;

  // Eviction thread
  private transient EvictionScheduler                              evictionScheduler;

  // A source for the current timestamp
  private transient TimeSource                                     timeSource;

  // JVM-local statistics instance
  private transient EvictionStatistics                             statistics;

  public DistributedCacheImpl(final CacheConfig config) {
    this(config, new ConcurrentDistributedMap<K, TimestampedValue<V>>());
  }

  protected DistributedCacheImpl(final CacheConfig config,
                                 final ConcurrentDistributedMap<K, TimestampedValue<V>> concurrentDistributedMap) {
    this.config = config;
    this.data = concurrentDistributedMap;
    this.orphanEvictorLock = new EvictorLock();
    initializeOnLoad(false);
  }

  /**
   * Terracotta <on-load> method
   */
  public void initializeOnLoad() {
    // XXX: this can potentially start another evictor if one is already running?
    initializeOnLoad(true);
  }

  protected void initializeOnLoad(final boolean startEviction) {
    if (config.isLoggingEnabled()) {
      logger.info("Initializing CHMDistributedMap, starting eviction thread");
    }

    this.timeSource = new SystemTimeSource();

    this.orphanEvictor = new OrphanEvictionListener<K>(config, this, orphanEvictorLock);
    this.evictionScheduler = new EvictionScheduler(config, new Evictor<K>(this, orphanEvictor));
    this.statistics = new EvictionStatistics();

    this.data.registerMapSizeListener(new TargetCapacityMapSizeListener(data, getConfig()));

    if (startEviction) {
      startEviction();
    }
  }

  private void startEviction() {
    // This method is also directly invoked as a "postCreate" method (ie. it will be invoked when a non-shared instance
    // of this class becomes shared)

    evictionScheduler.start();
  }

  public boolean containsKey(final Object key) {
    TimestampedValue<V> entry = getNonExpiredEntryCoherent((K) key, false);
    return entry != null;
  }

  private TimestampedValue<V> getNonExpiredEntryCoherent(K key, boolean markUsed) {
    return getNonExpiredEntry(key, markUsed, true);
  }
  
  private TimestampedValue<V> getNonExpiredEntryUnlocked(K key, boolean markUsed) {
    return getNonExpiredEntry(key, markUsed, false);
  }

  /**
   * Take an entry (which may be null) and return that entry only if not expired. If expired, return a null. If the
   * entry is expired, it is NOT evicted from the map. This method is suitable for calling by methods that return an old
   * version of an entry being modified (put, replace, remove) - these methods should not return an old expired entry.
   *
   * @param entry A possibly null, possibly expired entry
   * @return A possibly null, never expired entry
   */
  private TimestampedValue<V> filterExpired(final TimestampedValue<V> entry) {
    if (entry != null && entry.isExpired(getTime(), config)) {
      return null;
    } else {
      return entry;
    }
  }

  /**
   * Return the value from a possibly null entry.
   *
   * @param entry An entry, which may be null
   * @return The value for entry or null if the entry is null
   */
  private V getValueSafe(final TimestampedValue<V> entry) {
    return entry == null ? null : entry.getValue();
  }

  public V get(final Object key) {
    return getValueSafe(getNonExpiredEntryCoherent((K) key, true));
  }

  public TimestampedValue<V> getTimestampedValue(final K key) {
    return getNonExpiredEntryCoherent(key, true);
  }

  public TimestampedValue<V> getTimestampedValueQuiet(final K key) {
    return getNonExpiredEntryCoherent(key, false);
  }

  public TimestampedValue<V> removeTimestampedValue(final K key) {
    return filterExpired(this.data.remove(key));
  }

  /**
   * Get an entry for the specified key. If the entry is expired, it will be evicted from the cache,
   * except if a read lock is currently held by the current thread on the key (otherwise, get may
   * incur a write lock during removal). If markUsed is set, the entry's last accessed timestamp will be updated.
   *
   * @param key The key for the entry to get
   * @param markUsed True to mark this entry as "used" at the current time
   * @return The entry, or null if not in the map or in the map but expired
   */
  private TimestampedValue<V> getNonExpiredEntry(final K key, final boolean markUsed, boolean coherent) {
    TimestampedValue<V> entry;
    if (coherent) entry = this.data.get(key);
    else entry = this.data.unlockedGet(key);
    if (null == entry) return null;

    if (isEvictionEnabled() || isCapacityEvictionEnabled()) {
      final int now = getTime();

      String lockId = getLockIdForKey(key);
      if (entry.isExpired(now, config)) {
        if (!ManagerUtil.isLockHeldByCurrentThread(lockId, Manager.LOCK_TYPE_READ)) {
          evict(key, entry, now);
        }
        return null;
      }

      if (markUsed) {
        entry.markUsed(now, lockId, config);
      }
    }

    return entry;
  }

  public String getLockIdForKey(final K key) {
    return data.getLockIdForKey(key);
  }

  public Set<K> keySet() {
    return this.data.keySet();
  }

  public V put(final K key, final V value) {
    return getValueSafe(filterExpired(this.data.put(key, wrapValue(value))));
  }

  public V putIfAbsent(final K key, final V value) {
    return getValueSafe(filterExpired(data.putIfAbsent(key, wrapValue(value))));
  }

  public V remove(final Object key) {
    return getValueSafe(filterExpired(this.data.remove(key)));
  }

  public V replace(final K key, final V value) {
    return getValueSafe(filterExpired(this.data.replace(key, wrapValue(value))));
  }

  private TimestampedValue<V> wrapValue(final V value) {
    if (value instanceof TimestampedValue) {
      // If application already supplies a TimestampedEntry, don't wrap it again
      initCapacityEvictionPolicyFromConfig((TimestampedValue) value);
      return (TimestampedValue<V>) value;
    }
    DefaultTimestampedValue<V> defaultTimestampedValue = new DefaultTimestampedValue<V>(value, getTime());
    initCapacityEvictionPolicyFromConfig(defaultTimestampedValue);
    return defaultTimestampedValue;
  }

  private void initCapacityEvictionPolicyFromConfig(final TimestampedValue timestampedValue) {
    if (isCapacityEvictionEnabled()) {
      // use CapacityEvictionPolicyData.Factory in config to create a new one if its not an instance of the config's
      // factory
      CapacityEvictionPolicyData current = timestampedValue.getCapacityEvictionPolicyData();
      if (!getConfig().getCapacityEvictionPolicyDataFactory().isProductOfFactory(current)) {
        CapacityEvictionPolicyData cep = getConfig().getCapacityEvictionPolicyDataFactory()
            .newCapacityEvictionPolicyData();
        timestampedValue.setCapacityEvictionPolicyData(cep);
      }
    }
  }

  public void clear() {
    this.data.clear();
  }

  public int size() {
    return this.data.size();
  }

  public int localSize() {
    return data.localSize();
  }

  public void shutdown() {
    this.evictionScheduler.stop();
  }

  public void evictExpiredLocalElements() {
    if (isEvictionEnabled()) {
      Collection<Map.Entry<K, TimestampedValue<V>>> localEntries = data.getAllLocalEntriesSnapshot();
      invalidateCacheEntries(new EntrySnapshotIterator(localEntries));
    }
  }

  /**
   * This class is used to wrap the local entry snapshot and iterate through the keys without converting the collection
   * into some other collection.
   */
  private static class EntrySnapshotIterator<K> implements Iterator<K> {
    private final Iterator<Map.Entry<K, ?>> localEntryIter;

    EntrySnapshotIterator(final Collection<Map.Entry<K, ?>> localEntries) {
      this.localEntryIter = localEntries.iterator();
    }

    public boolean hasNext() {
      return this.localEntryIter.hasNext();
    }

    public K next() {
      Map.Entry<K, ?> nextEntry = this.localEntryIter.next();
      return nextEntry.getKey();
    }

    public void remove() {
      throw new UnsupportedOperationException();
    }
  }

  public void evictOrphanElements(final DsoCluster clusterInfo) {
    if (isEvictionEnabled() && getConfig().isOrphanEvictionEnabled()) {
      for (Map m : data.getConstituentMaps()) {
        Set<K> keys = clusterInfo.getKeysForOrphanedValues(m);
        invalidateCacheEntries(keys.iterator());
      }
    }
  }

  protected void invalidateCacheEntries(final Iterator<K> keys) {
    int examined = 0;
    int evicted = 0;

    while (keys.hasNext()) {
      K key = keys.next();
      try {
        TimestampedValue<V> wrappedValue = data.get(key);
        if (wrappedValue == null) continue;

        examined++;
        int now = getTime();
        if (wrappedValue.isExpired(now, config)) {
          evicted++;
          evict(key, wrappedValue, now);
        }
      } catch (Throwable t) {
        logger.error("Unhandled exception inspecting item for [" + key + "] for expiration", t);
      }
    }

    if (isStatisticsEnabled()) {
      statistics.increment(examined, evicted);
    }
  }

  /**
   * Evict an item from the cache due to expiration. The caller is responsible for determining whether the entry has
   * expired. This method will log the eviction (if logging is enabled) and remove the entry.
   */
  private void evict(final K key, final TimestampedValue<V> entry, final int now) {
    Assert.pre(key != null);
    if (this.data.remove(key, entry)) {
      onEvict(key, entry.getValue());
      logEviction(key, entry, now);
    }
  }

  private void logEviction(final K key, final TimestampedValue<V> entry, final int now) {
    if (config.isLoggingEnabled()) {
      logger.info(ManagerUtil.getClientID() + " expiring key: " + key + " (expiresAt = " + entry.expiresAt(config)
                  + ", now = " + now + ")");

    }
  }
  
  /**
   * Used by notify listeners of eviction
   * By the time this function is called, the item has already been evicted
   */
  protected void onEvict(final K key, final V value) {
    // to be overridden
  }

  /**
   * This is provided for testing purposes - it lets you override the source of System.currentTimeMillis() so that you
   * can control it yourself in the test. If it's not called, SystemTimeSource is used which just calls
   * System.currentTimeMillis(). Method public for tests in other projects
   *
   * @param timeSource The alternate TimeSource implementation
   */
  public void setTimeSource(final TimeSource timeSource) {
    this.timeSource = timeSource;
  }

  public TimeSource getTimeSource() {
    return this.timeSource;
  }

  /**
   * This method should always be called instead of System.currentTimeMillis() so that time can be controlled by the
   * TimeSource.
   *
   * @return The current time according to the TimeSource
   */
  private int getTime() {
    return this.timeSource.now();
  }

  public Set<Entry<K, V>> entrySet() {
    return new EntrySet<K, V>(this, data.entrySet());
  }

  public void putNoReturn(final K key, final V value) {
    data.putNoReturn(key, wrapValue(value));
  }
  
  public void unlockedPutNoReturn(final K key, final V value) {
    data.unlockedPutNoReturn(key, wrapValue(value));
  }
  
  public V unlockedGet(final K key) {
    return getValueSafe(getNonExpiredEntryUnlocked(key, true));
  }
  
  public TimestampedValue<V> unlockedGetTimestampedValue(final K key) {
    return getNonExpiredEntryUnlocked(key, true);
  }
  
  public TimestampedValue unlockedGetTimestampedValueQuite(final K key) {
    return getNonExpiredEntryUnlocked(key, false);
  }
  
  public boolean unlockedContainsKey(final Object key) {
    TimestampedValue<V> entry = getNonExpiredEntryUnlocked((K) key, false);
    return entry != null;
  }

  public void removeNoReturn(final Object key) {
    data.removeNoReturn((K) key);
  }
  
  public void unlockedRemoveNoReturn(final Object key) {
    data.unlockedRemoveNoReturn((K) key);
  }

  public boolean isStatisticsEnabled() {
    // read <autolock>'d
    synchronized (this) {
      return statisticsEnabled;
    }
  }

  public void setStatisticsEnabled(final boolean enabled) {
    // write <autolock>'d
    synchronized (this) {
      if (statistics != null) {
        if (enabled) {
          statistics.reset();
        } else {
          statistics.shutdown();
        }
        statisticsEnabled = enabled;
      }
    }
  }

  public EvictionStatistics getStatistics() {
    return statistics;
  }

  public CacheConfig getConfig() {
    return config;
  }

  protected boolean isEvictionEnabled() {
    return config.getMaxTTISeconds() > 0 || config.getMaxTTLSeconds() > 0;
  }

  protected boolean isCapacityEvictionEnabled() {
    return config.getTargetMaxInMemoryCount() > 0 || config.getTargetMaxTotalCount() > 0;
  }
  
  public boolean remove(final Object key, final Object value) {
    return this.data.remove(key, wrapValue((V) value));
  }

  public boolean replace(final K key, final V oldValue, final V newValue) {
    return this.data.replace(key, wrapValue(oldValue), wrapValue(newValue));
  }

  public boolean containsValue(final Object value) {
    throw new UnsupportedOperationException();
  }

  public boolean isEmpty() {
    return size() == 0;
  }

  public void putAll(final Map<? extends K, ? extends V> t) {
    throw new UnsupportedOperationException();
  }

  public Collection<V> values() {
    throw new UnsupportedOperationException();
  }

  public FinegrainedLock createFinegrainedLock(final K key) {
    return data.createFinegrainedLock(key);
  }

  public void lockEntry(final K key) {
    this.data.lockEntry(key);
  }

  public void unlockEntry(final K key) {
    this.data.unlockEntry(key);
  }
}
