/*
 * All content copyright (c) 2003-2009 Terracotta, Inc., except as may otherwise be noted in a separate copyright
 * notice. All rights reserved.
 */
package org.terracotta.cache.evictor;

import org.terracotta.cache.TimestampedValue;
import org.terracotta.collections.ConcurrentDistributedMap;

import com.tc.cluster.DsoCluster;
import com.tc.injection.annotations.InjectedDsoInstance;

import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashSet;
import java.util.IdentityHashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.Random;
import java.util.Set;
import java.util.Map.Entry;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicReference;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 * @author Chris Dennis
 */
public class TargetCapacityEvictor {

  private static final Logger                   LOGGER                    = Logger.getLogger(TargetCapacityEvictor.class.getName());
  
  private static final int                      SAMPLE_SIZE               = 12;
  private static final int                      RETAINED_SIZE             = 2;

  private static final int                      LOW_ORPHAN_COUNT          = 100;
  private static final int                      LOW_ORPHAN_COUNT_WAIT     = 10000;

  private static final long                     TRY_REMOVE_TIMEOUT        = 200;

  private static final int                      EVICTION_ATTEMPT_RATIO    = 10;

  private static final Random                   RNDM                      = new Random();
  private static final Set                      ORPHAN_LOOKUP_IN_PROGRESS = Collections.emptySet();

  @InjectedDsoInstance
  private DsoCluster                            clusterInfo;

  private final ConcurrentDistributedMap        map;
  private final Comparator<DetachedEntry>       capacityEvictionComparator;

  private final AtomicReference<Set>            orphanKeys                = new AtomicReference<Set>(new ConcurrentHashSet());
  private final ThreadLocal<Set<DetachedEntry>> localSamples              = new ThreadLocal<Set<DetachedEntry>>() {
                                                                            @Override
                                                                            protected Set<DetachedEntry> initialValue() {
                                                                       return new HashSet<DetachedEntry>(SAMPLE_SIZE);
                                                                            }
                                                                          };

  public TargetCapacityEvictor(ConcurrentDistributedMap map) {
    this.map = map;
    this.capacityEvictionComparator = new CapacityEvictionPolicyComparator();
  }

  public void evictLocalElements(int maxEvict) {
    Set<DetachedEntry> sample = localSamples.get();
    int attempts = maxEvict * EVICTION_ATTEMPT_RATIO;
    int evicted = 0;
    for (int j = 0; evicted < maxEvict && j < maxEvict * 10; j++) {
      if (evictLocalElement(sample, false)) {
        evicted++;
      }
    }
    
    if (evicted < maxEvict) {
      LOGGER.log(Level.WARNING, "Attempted to evict {0} local elements: aborted after {1} attempts - evicted {2}",
                 new Object[] {maxEvict, attempts, evicted});
    }
  }

  public void evictOrphanElements(int maxEvict) {
    int attempts = maxEvict * EVICTION_ATTEMPT_RATIO;
    int evicted = 0;
    for (int j = 0; evicted < maxEvict && j < attempts * 10; j++) {
      if (evictOrphanElement()) {
        evicted++;
      }
    }
    
    if (evicted < maxEvict) {
      LOGGER.log(Level.WARNING, "Attempted to evict {0} orphan elements: aborted after {1} attempts - evicted {2}",
                 new Object[] {maxEvict, attempts, evicted});
    }
  }

  /**
   * The eviction algorithm used here is statistical. Rather than maintaining a strict LRU data structure (adds cost on
   * read) or doing a full sweep of the cache (too slow) we sample a small selection of the cache elements and evict the
   * LRU element from the sample.
   * <p>
   * In order to minimize the sample size needed to maintain a certain level of correctness we skew the sample by
   * retaining the next 2 least useful elements for consideration in the next cycle. This allows us to have correctness
   * as good as a regular 30 element sample despite only retrieving 10 new elements per cycle (plus 2 retained).
   * <p>
   * A more detailed explanation and analysis of the algorithm can be found here: <a
   * href="http://www-rcf.usc.edu/~kpsounis/thesis.html"> http://www-rcf.usc.edu/~kpsounis/thesis.html</a>
   * 
   * @param sample set containing held over elements
   * @param remove <code>true</code> = remove, <code>false</code> = flush
   */
  private boolean evictLocalElement(Set<DetachedEntry> sample, boolean remove) {
    // select new random keys until we have a full set
    // cap the number of iterations here to 10*SAMPLE_SIZE
    for (int i = 0; (i < (SAMPLE_SIZE * 10)) && (sample.size() < Math.min(SAMPLE_SIZE, map.localSize())); i++) {
      Entry e = map.getRandomLocalEntry();
      if (e == null) {
        break;
      } else {
        /*
         * This wrapping is not strictly necessary for NBHM as it already detaches its entries. However I left this in
         * as it is good insurance against a change in the underlying map implementation. (Most maps do not detach
         * their entries e.g. CHM).
         */
        sample.add(new DetachedEntry(e));
      }
    }

    if (sample.isEmpty()) { return false; }

    // sort the list of entries
    DetachedEntry[] entries = sample.toArray(new DetachedEntry[sample.size()]);
    Arrays.sort(entries, capacityEvictionComparator);

    // attempt to evict the least useful entry
    DetachedEntry evictee = entries[0];
    boolean success;
    if (remove) {
      /*
       * Here we do a tryRemove - which will only remove if a tryLock on the associated entry succeeds. This prevents
       * coincidental dead-locks in total eviction (entries trying to evict each other while both are externally write
       * locked).
       */
      success = map.tryRemove(evictee.getKey(), TRY_REMOVE_TIMEOUT, TimeUnit.MILLISECONDS);
    } else {
      success = map.flush(evictee.getKey(), evictee.getValue());
    }

    // keep the next RETAINED_SIZE least useful entries
    sample.clear();
    for (int i = 1; (i < entries.length) && (i < (RETAINED_SIZE + 1)); i++) {
      sample.add(entries[i]);
    }

    return success;
  }

  private boolean evictOrphanElement() {
    boolean success = false;

    Set orphans = orphanKeys.get();
    if (orphans.isEmpty()) {
      /*
       * We could potentially delay evicting local elements until things get serious - e.g. 10% over target size. This
       * would give the thread getting the orphan list time to return before we start kicking out local values.
       */
      success = evictLocalElement(localSamples.get(), true);

      /*
       * If there are no local entries and we're over the total capacity limit then remove a random entry.
       */
      if (!success) {
        success = map.tryRemove(map.getRandomEntry().getKey(), TRY_REMOVE_TIMEOUT, TimeUnit.SECONDS);
      }

      if (orphans != ORPHAN_LOOKUP_IN_PROGRESS) {
        /*
         * XXX: This is an ugly hack This looks really odd but it serves to kick the ClientObjectManager to tell the
         * server about all the objects it has evicted. ObjectID.MAX_ID is a valid object-id but is highly unlikely to
         * map to a real object - hence we're unlikely to incur unnecessary overhead.
         */
        // ManagerUtil.preFetchObject(new ObjectID(ObjectID.MAX_ID));
        if (orphanKeys.compareAndSet(orphans, ORPHAN_LOOKUP_IN_PROGRESS)) {
          new Thread(new Runnable() {
            public void run() {
              /*
               * Suggest the winning node do a gc. This will clear all the weak refs associated with the orphan values.
               */
              // System.gc();
              List<Map> maps = map.getConstituentMaps();
              Set newOrphansSet = new ConcurrentHashSet();
              Map<Map, Boolean> usedMaps = new IdentityHashMap();

              /*
               * Sample 10% of the maps (in a random order) to build our orphan set. This prevents too many collisions
               * between nodes when trying to evict orphans.
               */
              while (usedMaps.size() <= (0.1 * maps.size())) {
                Map m = maps.get(RNDM.nextInt(maps.size()));
                if (usedMaps.put(m, Boolean.TRUE) == null) {
                  newOrphansSet.addAll(clusterInfo.getKeysForOrphanedValues(m));
                }
              }

              /*
               * We didn't find many orphans this time around. This may be a sign that the user has set up the L1 limits
               * so high that we are evicting elements before any have become orphans. If we just return the small set
               * here then we'll just end up spamming the L2 with requests for orphan keys - not good. Instead lets
               * sleep for a bit here before returning the new orphan set so the L2 can breath a bit. We really need a
               * better tactic here - not that I can think of anything right now.
               */
              if (newOrphansSet.size() < LOW_ORPHAN_COUNT) {
                try {
                  Thread.sleep(LOW_ORPHAN_COUNT_WAIT);
                } catch (InterruptedException e) {
                  // ignore
                }
              }
              orphanKeys.set(newOrphansSet);
            }
          }).start();
        }
      }
    } else {
      try {
        Iterator it = orphans.iterator();
        /*
         * Here we do a tryRemove - which will only remove if a tryLock on the associated entry succeeds. This prevents
         * coincidental dead-locks in total eviction (entries trying to evict each other while both are externally write
         * locked).
         */
        success = map.tryRemove(it.next(), TRY_REMOVE_TIMEOUT, TimeUnit.MILLISECONDS);
        it.remove();
      } catch (NoSuchElementException e) {
        //
      }
    }

    return success;
  }

  static class CapacityEvictionPolicyComparator implements Comparator<DetachedEntry> {

    public int compare(DetachedEntry o1, DetachedEntry o2) {
      Object v1 = o1.getValue();
      Object v2 = o2.getValue();
      if ((v1 instanceof TimestampedValue) && (v2 instanceof TimestampedValue)) {
        CapacityEvictionPolicyData d1 = ((TimestampedValue) v1).getCapacityEvictionPolicyData();
        CapacityEvictionPolicyData d2 = ((TimestampedValue) v2).getCapacityEvictionPolicyData();

        if (d1 == null) {
          if (d2 == null) {
            return 0;
          } else {
            return -1;
          }
        } else {
          return d1.compareTo(d2);
        }
      } else {
        return 0;
      }
    }

  }

  private static class DetachedEntry {
    // this is not a java.util.Map.Entry so that we don't have to implement
    // hashCode()/equals() that considers the value (EHC-557)

    private final Object key;
    private final Object value;

    public DetachedEntry(Entry original) {
      key = original.getKey();
      value = original.getValue();
    }

    public Object getKey() {
      return key;
    }

    public Object getValue() {
      return value;
    }

    @Override
    public boolean equals(Object o) {
      if (o == this) { return true; }

      if (o instanceof DetachedEntry) {
        return getKey().equals(((DetachedEntry) o).getKey());
      } else {
        return false;
      }
    }

    @Override
    public int hashCode() {
      Object k = getKey();
      return k == null ? 0 : k.hashCode();
    }
  }
}
