/*
 * All content copyright (c) 2003-2008 Terracotta, Inc., except as may otherwise be noted in a separate copyright
 * notice. All rights reserved.
 */
package org.terracotta.cache.evictor;

import org.terracotta.cache.CacheConfig;

import com.tc.cluster.DsoCluster;
import com.tc.injection.annotations.InjectedDsoInstance;
import com.tc.logging.TCLogger;
import com.tc.object.bytecode.ManagerUtil;
import com.tc.util.Assert;

/**
 * <p>
 * At the beginning of each run, the cache attempts to become the "orphan evictor". For each chunk of the cache, only
 * one of the local evictors in the cluster will become the orphan evictor. That thread is responsible for both its own
 * local eviction and also the orphan eviction. Orphan eviction entails checking "orphan" keys (those not currently
 * faulted into any node's cache) for eviction. The process of doing this will cause those keys to be loaded into this
 * node such that they are no longer orphans.
 * </p>
 */
public class OrphanEvictionListener<K> implements EvictionListener {

  // Local cached resources
  private static volatile TCLogger logger;

  // Configuration
  private final CacheConfig             config;

  // Clustered resources
  private final Evictable<K>       store;
  private final EvictorLock        orphanEvictorLock;

  // Orphan eviction state
  private transient boolean        isOrphanEvictor;
  // when this == orphanEvictionPeriod, do orphan eviction
  private transient int            evictionCount;

  // Injected access to DSO cluster topology and locality information
  @InjectedDsoInstance
  private DsoCluster               clusterInfo;

  /**
   * Construct an orphan eviction listener with clustered state
   * 
   * @param config The clustered configuration of the orphan distributed map and orphan evictor
   * @param store The clustered store to evict
   * @param orphanEvictorLock The clustered lock that is used as a token to indicate the elected orphan evictor
   */
  public OrphanEvictionListener(CacheConfig config, Evictable<K> store, EvictorLock orphanEvictorLock) {

    // Copy configuration
    this.config = config;
    if (config.isOrphanEvictionEnabled()) {
      Assert.eval(config.getOrphanEvictionPeriod() > 0);
    }

    // Copy clustered resources
    this.store = store;
    this.orphanEvictorLock = orphanEvictorLock;
  }

  private static TCLogger getLogger() {
    if (logger == null) {
      logger = ManagerUtil.getLogger("com.tc.dmap.eviction");
    }
    return logger;
  }

  private void log(String message) {
    if (config.isLoggingEnabled()) {
      getLogger().info("[" + ManagerUtil.getClientID() + "] " + message);
    }
  }

  /**
   * Attempt to obtain orphan evictor write lock. If successful, set isOrphanEvictor to true
   */
  public void startLocalEviction() {
    if (config.isOrphanEvictionEnabled() && !isOrphanEvictor) {
      if (orphanEvictorLock.tryLock()) {
        isOrphanEvictor = true;
        log("Elected to be orphan evictor");
      } else {
        isOrphanEvictor = false;
      }
    }

    log("Local eviction started");
  }

  /**
   * Perform orphan eviction
   */
  public void endLocalEviction() {
    log("Local eviction finished");

    if (!config.isOrphanEvictionEnabled() || !isOrphanEvictor || !(config.getMaxTTISeconds() > 0 || config.getMaxTTLSeconds() > 0)) { return; }

    boolean isTimeForOrphanEviction = incrementEvictionCounter();
    if (isTimeForOrphanEviction) {
      log("Running orphan eviction (evictionCount = " + evictionCount + ", orphanEvictionPeriod = "
          + config.getOrphanEvictionPeriod() + ")");
      // when the cluster information has been injected, there's also no sense to evict
      if (clusterInfo != null) {
        store.evictOrphanElements(clusterInfo);
      }
    } else {
      log("Not running orphan eviction (evictionCount = " + evictionCount + ", orphanEvictionPeriod = "
          + config.getOrphanEvictionPeriod() + ")");
    }

  }

  private boolean incrementEvictionCounter() {
    evictionCount++;
    boolean isTimeForOrphanEviction = evictionCount >= config.getOrphanEvictionPeriod();
    if (isTimeForOrphanEviction) {
      evictionCount = 0;
    }
    return isTimeForOrphanEviction;
  }

  /**
   * On shutdown, release the orphan evictor lock so it can be reobtained by a later evictor.
   */
  public void onShutdown() {
    if (isOrphanEvictor) {
      isOrphanEvictor = false;
      orphanEvictorLock.unlock();
    }
  }

}
