/*
 * All content copyright (c) 2003-2008 Terracotta, Inc., except as may otherwise be noted in a separate copyright
 * notice. All rights reserved.
 */
package org.terracotta.cache.evictor;

import org.terracotta.cache.CacheConfig;
import org.terracotta.cache.CacheConfigFactory;

import com.tc.object.bytecode.ManagerUtil;

/**
 * The scheduler is responsible for creating a background eviction thread and kicking it off where the thread will call
 * the Evictor after a specified delay in a loop. The thread will run forever unless stopped. The start() method can be
 * called again after stop.
 */
public class EvictionScheduler {

  private final CacheConfig     config;
  private final Evictor<?> evictor;
  private EvictionRunner   runner;

  /**
   * Construct a scheduler that pauses for the delay and whose background thread name is based on the timerName.
   * 
   * @param config The eviction configuration
   * @param evictor The evictor to callback on
   */
  public EvictionScheduler(final CacheConfig config, final Evictor<?> evictor) {
    this.config = config;
    this.evictor = evictor;
  }

  /**
   * Actually kick off the thread.
   * 
   * @return True if started, false if already running
   */
  public synchronized void start() {
    if (this.runner == null) {
      this.runner = new EvictionRunner(config, evictor);

      Thread t = new Thread(runner, config.getName() + " Evictor");
      t.setDaemon(true);
      t.start();
    }
  }

  /**
   * Tells background thread to stop at next opportunity and kills the reference to it
   */
  public synchronized void stop() {
    EvictionRunner oldRunner = this.runner;
    runner = null;

    if (oldRunner != null) {
      oldRunner.cancel();
    }
  }

  private static class EvictionRunner implements Runnable {
    private final CacheConfig   config;
    private final Evictor<?>    evictor;
    private final int           evictor_max_sleep_secs;
    private final int           evictor_min_sleep_secs;
    private volatile boolean    running                   = true;

    private static final String EVICTOR_MAX_SLEEP_PROP    = "dmap.evictor-maximum-sleep-seconds";
    private static final String EVICTOR_MIN_SLEEP_PROP    = "dmap.evictor-minimum-sleep-seconds";
    // 1 hour
    private static final int    DEFAULT_MAX_EVICTOR_SLEEP = 60 * 60;
    // 1 second
    private static final int    DEFAULT_MIN_EVICTOR_SLEEP = 1;

    public EvictionRunner(final CacheConfig config, final Evictor<?> evictor) {
      this.evictor = evictor;
      this.config = config;
      this.evictor_max_sleep_secs = getSleepTime(EVICTOR_MAX_SLEEP_PROP, DEFAULT_MAX_EVICTOR_SLEEP);
      this.evictor_min_sleep_secs = getSleepTime(EVICTOR_MIN_SLEEP_PROP, DEFAULT_MIN_EVICTOR_SLEEP);
    }

    private static int getSleepTime(String prop, int defaultVal) {
      if (! CacheConfigFactory.DSO_ACTIVE) { return defaultVal; }
      return ManagerUtil.getTCProperties().getInt(prop, defaultVal);
    }
    
    public void cancel() {
      running = false;
    }

    public void run() {
      try {
        do {
          sleep();
          evictor.run();
        } while (running);
      } finally {
        evictor.postRun();
      }
    }

    /**
     * Calculates sleep time from the config based on TTI and TTL.
     * <p />
     * If tti=ttl=0, returns EVICTOR_MAX_SLEEP_SECS (whose value can be changed using the EVICTOR_MAX_SLEEP_PROP tc
     * property, defaults to 1 second) otherwise returns 50% of (minimum non-0 value of either tti or ttl)
     * <p />
     * The returned value is always >= EVICTOR_MIN_SLEEP_SECS and <= EVICTOR_MAX_SLEEP_SECS
     */
    private int getSleepTimeSeconds() {
      int sleepSeconds = checkBounds(calculateSleepTimeFromConfig(), evictor_min_sleep_secs, evictor_max_sleep_secs);
      if (sleepSeconds <= 0) { throw new AssertionError("Invalid sleepTimeInSeconds: " + sleepSeconds); }
      return sleepSeconds;
    }

    private int checkBounds(final int value, final int min, final int max) {
      if (value < min) return min;
      if (value > max) return max;
      return value;
    }

    private int calculateSleepTimeFromConfig() {
      int tti = config.getMaxTTISeconds();
      int ttl = config.getMaxTTLSeconds();
      if (tti == 0 && ttl == 0) { return Integer.MAX_VALUE; }
      if (tti <= 0) {
        tti = Integer.MAX_VALUE;
      }
      if (ttl <= 0) {
        ttl = Integer.MAX_VALUE;
      }
      int rv = Math.min(tti, ttl) / 2;
      return rv;
    }

    private void sleep() {
      long start = System.currentTimeMillis();

      long wakeup = calculateWakeup(start);
      while (System.currentTimeMillis() < wakeup) {
        config.waitForChange(wakeup - System.currentTimeMillis());
        wakeup = calculateWakeup(start);
      }
    }

    private long calculateWakeup(long start) {
      return start + (getSleepTimeSeconds() * 1000);
    }
  }
}