/*
 * All content copyright Terracotta, Inc., unless otherwise indicated. All rights reserved.
 */
package org.terracotta.collections;

import java.util.Collection;
import java.util.Iterator;
import java.util.Set;

public class ConcurrentDistributedSet<E> implements Set<E> {

  // XXX: When this type is used in express we can't use AbstractSet as super class for some reason

  private static final Object                       PRESENT = Boolean.TRUE;

  private final ConcurrentDistributedMap<E, Object> map;

  public ConcurrentDistributedSet() {
    map = new ConcurrentDistributedMap<E, Object>();
  }

  public ConcurrentDistributedSet(LockType lockType, LockStrategy<? super E> lockStrategy) {
    map = new ConcurrentDistributedMap<E, Object>(lockType, lockStrategy);
  }

  public ConcurrentDistributedSet(LockType lockType, LockStrategy<? super E> lockStrategy, int concurrency) {
    map = new ConcurrentDistributedMap<E, Object>(lockType, lockStrategy, concurrency);
  }

  public boolean add(E o) {
    return map.put(o, PRESENT) == null;
  }

  public boolean addAll(Collection<? extends E> c) {
    boolean rv = false;
    for (E e : c) {
      rv |= add(e);
    }
    return rv;
  }

  public void clear() {
    map.clear();
  }

  public boolean contains(Object o) {
    return map.containsKey(o);
  }

  public boolean containsAll(Collection<?> c) {
    for (Object o : c) {
      if (!contains(o)) { return false; }
    }

    return true;

  }

  public boolean isEmpty() {
    return map.isEmpty();
  }

  public Iterator<E> iterator() {
    return map.keySet().iterator();
  }

  public boolean remove(Object o) {
    return map.remove(o) == PRESENT;
  }

  public boolean removeAll(Collection<?> c) {
    boolean rv = false;

    if (size() > c.size()) {
      for (Object o : c) {
        rv |= remove(o);
      }
    } else {
      for (Iterator<?> iter = iterator(); iter.hasNext();) {
        if (c.contains(iter.next())) {
          iter.remove();
          rv = true;
        }
      }
    }
    return rv;

  }

  public boolean retainAll(Collection<?> c) {
    boolean rv = false;
    for (Iterator<E> iter = iterator(); iter.hasNext();) {
      if (!c.contains(iter.next())) {
        iter.remove();
        rv = true;
      }
    }
    return rv;
  }

  public int size() {
    return map.size();
  }

  public Object[] toArray() {
    return map.keySet().toArray();
  }

  public <T> T[] toArray(T[] a) {
    return map.keySet().toArray(a);
  }

}
