/*
 * All content copyright (c) 2003-2008 Terracotta, Inc., except as may otherwise be noted in a separate copyright
 * notice. All rights reserved.
 */
package org.terracotta.collections;

import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.TimeUnit;

/**
 * A {@link ConcurrentMap} that automatically switches its implementation to one that is highly concurrent in a
 * Terracotta DSO cluster. When Terracotta DSO isn't active, a traditional {@code ConcurrentMap} implementation like
 * {@link ConcurrentHashMap} is used within a single JVM.
 * <p>
 * Map entries can't use {@code null} for keys or values.
 * <p>
 * One of the principal aspects of the Terracotta DSO optimized implementation is that locking is done on several
 * levels.
 * <p>
 * First of all, each local instance of the map is always correctly synchronized locally to ensure consistency and
 * coherency within a single JVM.
 * <p>
 * Secondly, cluster-wide locking is split up in two levels:
 * <ul>
 * <li>Methods that operate on individual entries with identifiable keys will take fine-grained cluster-wide locks for
 * each entry</li>
 * <li>Other methods that operate on the entire map are locked with their own map-specific cluster-wide lock.</li>
 * </ul>
 * <p>
 * This cluster-wide locking scheme provides very high concurrency at the cost of unbalanced coherency. This means that
 * the visibility of changes made through the key-based methods will not necessarily be in order with map-based locking
 * methods and vice-versa.
 * 
 * @author Geert Bevin, Chris Dennis
 * @param <K> type of mapped keys
 * @param <V> type of mapped values
 */
public class ConcurrentDistributedMap<K, V> implements ConcurrentMap<K, V>, LockableMap<K> {
  // Note: We add NotClearable and TCMap interfaces to this type dynamically with instrumentation

  static final int                                    DEFAULT_CONCURRENCY = 128;

  private final ConcurrentDistributedMapBackend<K, V> map;

  /**
   * Creates a new {@code ConcurrentDistributedMap} instance with a default lock type of {@link LockType#WRITE WRITE}
   * and the default lock strategy of {@link HashcodeLockStrategy}.
   * 
   * @see #ConcurrentDistributedMap(LockType, LockStrategy)
   */
  public ConcurrentDistributedMap() {
    this(LockType.WRITE, new HashcodeLockStrategy());
  }

  /**
   * Creates a new {@code ConcurrentDistributedMap} instance that allows the specification of the internal lock type
   * that should be used when Terracotta DSO is active.
   * 
   * @param lockType the lock type that should be used internally when Terracotta DSO is active
   * @param lockStrategy the lock strategy to use for this map
   * @see LockType
   * @see LockStrategy
   */
  public ConcurrentDistributedMap(final LockType lockType, final LockStrategy<? super K> lockStrategy) {
    this(lockType, lockStrategy, DEFAULT_CONCURRENCY);
  }

  /**
   * Creates a new {@code ConcurrentDistributedMap} instance that allows the specification of the internal lock type
   * that should be used when Terracotta DSO is active and the concurrency of internal data structures.
   * 
   * @param lockType the lock type that should be used internally when Terracotta DSO is active
   * @param lockStrategy the lock strategy to use for this map
   * @param concurrency the estimated number of concurrently updating threads
   * @see LockType
   * @see LockStrategy
   */
  public ConcurrentDistributedMap(final LockType lockType, final LockStrategy<? super K> lockStrategy,
                                  final int concurrency) {
    if (null == lockType) { throw new IllegalArgumentException("lockType can't be null"); }
    if (null == lockStrategy) { throw new IllegalArgumentException("lockStrategy can't be null"); }

    if (isDsoActive()) {
      map = new ConcurrentDistributedMapDsoArray<K, V>(lockType, lockStrategy, concurrency);
    } else {
      map = new ConcurrentDistributedMapNoDso<K, V>(concurrency);
    }
  }

  private boolean isDsoActive() {
    return Boolean.getBoolean("tc.active");
  }

  public V putIfAbsent(final K key, final V value) {
    return map.putIfAbsent(key, value);
  }

  public boolean remove(final Object key, final Object value) {
    return map.remove(key, value);
  }

  public V replace(final K key, final V value) {
    return map.replace(key, value);
  }

  /**
   * Behaves the same as the standard {@link #remove(Object)} method except that the previous value is not returned.
   * This can provide significant performance improvements when used in a Terracotta DSO cluster.
   * 
   * @param key the key of the entry to remove
   */
  public void removeNoReturn(final K key) {
    map.removeNoReturn(key);
  }
  
  public void unlockedRemoveNoReturn(final K key) {
    map.unlockedRemoveNoReturn(key);
  }

  public boolean replace(final K key, final V oldValue, final V newValue) {
    return map.replace(key, oldValue, newValue);
  }

  /**
   * This method clears the map locally on the node that is resides on but doesn't guarantee when this is done on other
   * nodes in the cluster. It is possible that methods that operate on specific keys are actually reordered and execute
   * in unexpected ways across the cluster. {@inheritDoc}
   */
  public void clear() {
    map.clear();
  }

  public boolean containsKey(final Object key) {
    return map.containsKey(key);
  }

  /**
   * Perform a raw/unlocked read for the given key. It performs no locking and thus provides absolutely no visibility
   * semantics. If the value is not local it will NOT be faulted and null is returned instead -- thus one cannot
   * distinguish between a non-existent mapping from an un-faulted value. USE WITH CAUTION
   * 
   * @param key The key to lookup
   * @return A value mapped to the given key (but not necessarily the most current) value or null if the value is
   *         missing or not local
   */
  public V unsafeGet(final K key) {
    return map.unsafeGet(key);
  }

  public boolean containsValue(final Object value) {
    return map.containsValue(value);
  }

  public Set<Map.Entry<K, V>> entrySet() {
    return map.entrySet();
  }

  public V get(final Object key) {
    return map.get(key);
  }
  
  public V unlockedGet(final Object key) {
    return map.unlockedGet((K) key);
  }

  /**
   * This method reflects the state of the map instance that is local on the node. Any changes that are done through
   * methods that operate on specific keys are not guaranteed to have been propagated throughout the cluster.
   * {@inheritDoc}
   */
  public boolean isEmpty() {
    return map.isEmpty();
  }

  public Set<K> keySet() {
    return map.keySet();
  }

  public V put(final K key, final V value) {
    return map.put(key, value);
  }

  /**
   * Behaves the same as the standard {@link #put(Object, Object)}�method except that the previous value is not
   * returned. This can provide significant performance improvements when used in a Terracotta DSO cluster.
   * 
   * @param key the key of the entry to put
   * @param value the value of the entry to put
   */
  public void putNoReturn(final K key, final V value) {
    map.putNoReturn(key, value);
  }
  
  public void unlockedPutNoReturn(final K key, final V value) {
    map.unlockedPutNoReturn(key, value);
  }

  public void putAll(final Map<? extends K, ? extends V> t) {
    map.putAll(t);
  }

  public V remove(final Object key) {
    return map.remove(key);
  }

  /**
   * This method reflects the state of the map instance that is local on the node. Any changes that are done through
   * methods that operate on specific keys are not guaranteed to have been propagated throughout the cluster.
   * {@inheritDoc}
   */
  public int size() {
    return map.size();
  }

  public int localSize() {
    return map.localSize();
  }

  public Collection<V> values() {
    return map.values();
  }

  public Collection<Entry<K, V>> getAllLocalEntriesSnapshot() {
    return map.getAllLocalEntriesSnapshot();
  }

  public Collection<Entry<K, V>> getAllEntriesSnapshot() {
    return map.getAllEntriesSnapshot();
  }

  public FinegrainedLock createFinegrainedLock(final K key) {
    return map.createFinegrainedLock(key);
  }

  /**
   * In a shared instance of a concurrent distributed map this will lock the associated entry. In an unshared instance
   * it is a no-op.
   * <p>
   * ConcurrentDistributedMap instances allow non existent entries to be locked.
   */
  public void lockEntry(final K key) {
    map.lockEntry(key);
  }

  /**
   * In a shared instance of a concurrent distributed map this will unlock the associated entry. In an unshared instance
   * it is a no-op.
   */
  public void unlockEntry(final K key) {
    map.unlockEntry(key);
  }

  public String getLockIdForKey(final K key) {
    return map.getLockIdForKey(key);
  }

  /* TCMap interface method */
  public void __tc_applicator_clear() {
    // This method should not be called on the top level CSM instance
    throw new AssertionError();
  }

  /* TCMap interface method */
  public void __tc_applicator_put(final Object obj, final Object obj1) {
    // This method should not be called on the top level CSM instance
    throw new AssertionError();
  }

  /* TCMap interface method */
  public void __tc_applicator_remove(final Object obj) {
    // This method should not be called on the top level CSM instance
    throw new AssertionError();
  }

  /* TCMap interface method */
  public Collection __tc_getAllEntriesSnapshot() {
    return getAllEntriesSnapshot();
  }

  /* TCMap interface method */
  public Collection __tc_getAllLocalEntriesSnapshot() {
    /*
     * This null check prevents needless faulting-in of the map reference. If the reference is null then we have no
     * resident values as the map itself isn't resident. This prevents the needless faulting of the entire key-set.
     */
    if (map == null) {
      return Collections.emptyList();
    } else {
      return map.getAllLocalEntriesSnapshot();
    }
  }

  /* TCMap interface method */
  public void __tc_put_logical(final Object key, final Object value) {
    putNoReturn((K) key, (V) value);
  }

  /* TCMap interface method */
  public void __tc_remove_logical(final Object key) {
    removeNoReturn((K) key);
  }

  public List<Map<K, ?>> getConstituentMaps() {
    return map.getConstituentMaps();
  }

  public Map.Entry<K, V> getRandomEntry() {
    return map.getRandomEntry();
  }

  public Map.Entry<K, V> getRandomLocalEntry() {
    return map.getRandomLocalEntry();
  }

  public boolean flush(final Object key, final Object value) {
    return map.flush(key, value);
  }

  public boolean tryRemove(final Object key, final long time, final TimeUnit unit) {
    return map.tryRemove(key, time, unit);
  }

  public MapSizeListener registerMapSizeListener(final MapSizeListener newListener) {
    return map.registerMapSizeListener(newListener);
  }
}
