/*
 * All content copyright (c) 2003-2008 Terracotta, Inc., except as may otherwise be noted in a separate copyright
 * notice. All rights reserved.
 */
package org.terracotta.collections;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Random;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;

class ConcurrentDistributedMapNoDso<K, V> extends ConcurrentHashMap<K, V> implements ConcurrentDistributedMapBackend<K, V> {

  private final Random    rndm            = new Random();

  private MapSizeListener mapSizeListener = null;

  ConcurrentDistributedMapNoDso() {
    super();
  }

  public int localSize() {
    return size();
  }
  
  public void removeNoReturn(final K key) {
    remove(key);
  }

  @Override
  public V remove(Object key) {
    V val = super.remove(key);
    if (val != null) {
      sizeDecrement();
    }
    return val;
  }

  @Override
  // overriding this method to throw a NPE when the value is null,
  // since ConcurrentHashMap doesn't and NBHM does.
  public boolean remove(Object key, Object value) {
    if (null == value) { throw new NullPointerException(); }
    boolean removed = super.remove(key, value);
    if (removed) {
      sizeDecrement();
    }
    return removed;
  }
  
  public Collection<java.util.Map.Entry<K, V>> getAllEntriesSnapshot() {
    return new ArrayList<Map.Entry<K, V>>(entrySet());
  }

  public Collection<java.util.Map.Entry<K, V>> getAllLocalEntriesSnapshot() {
    return new ArrayList<Map.Entry<K, V>>(entrySet());
  }

  public void putNoReturn(K key, V value) {
    put(key, value);
  }

  @Override
  public V put(K key, V value) {
    V val = super.put(key, value);
    if (val == null) {
      sizeIncrement();
    }
    return val;
  }

  @Override
  public V putIfAbsent(K key, V value) {
    V val = super.putIfAbsent(key, value);
    if (val == null) {
      sizeIncrement();
    }
    return val;
  }

  public void lockEntry(K key) {
    // nop
  }

  public void unlockEntry(K key) {
    // nop
  }

  public V unsafeGet(K key) {
    // no need to do anything different in the local case
    return get(key);
  }

  public String getLockIdForKey(K key) {
    return "";
  }

  public List<Map<K, ?>> getConstituentMaps() {
    return Collections.<Map<K, ?>>singletonList(this);
  }

  public Map.Entry<K, V> getRandomEntry() {
    int index = rndm.nextInt(size());
    Iterator<Map.Entry<K, V>> it = entrySet().iterator();
    Map.Entry<K, V> entry = null;
    do {
      entry = it.next();
    } while (it.hasNext() && (index-- > 0));

    return entry;
  }

  public Map.Entry<K, V> getRandomLocalEntry() {
    return getRandomEntry();
  }

  public boolean flush(Object key, Object value) {
    boolean success = remove(key, value);
    if (success) {
      sizeDecrement();
    }
    return success;
  }

  public boolean tryRemove(Object key, long time, TimeUnit unit) {
    return remove(key) != null;
  }
  
  private void sizeIncrement() {
    MapSizeListener l = mapSizeListener;
    if (l != null) {
      l.sizeChanged(1);
      l.localSizeChanged(1);
    }
  }

  private void sizeDecrement() {
    MapSizeListener l = mapSizeListener;
    if (l != null) {
      l.sizeChanged(-1);
      l.localSizeChanged(-1);
    }
  }

  public MapSizeListener registerMapSizeListener(MapSizeListener newListener) {
    MapSizeListener old = mapSizeListener;
    mapSizeListener = newListener;
    return old;
  }
}