/*
 * All content copyright (c) Terracotta, Inc., except as may otherwise be noted in a separate copyright
 * notice. All rights reserved.
 */
package org.terracotta.collections;

import java.util.AbstractCollection;
import java.util.AbstractSet;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Random;
import java.util.Set;
import java.util.concurrent.TimeUnit;

import com.tc.object.bytecode.Clearable;
import com.tc.object.bytecode.ManagerUtil;
import com.tc.object.bytecode.NotClearable;
import com.tc.object.bytecode.NullManager;
import com.tc.util.Assert;

public class ConcurrentDistributedMapDsoArray<K, V> implements ConcurrentDistributedMapBackend<K, V>, NotClearable {

  private final int                                         concurrency;
  // using ArrayList because it's faster then using an Array.
  private final List<ConcurrentDistributedMapDso<K, V>>     storeList;
  private final int                                         segmentShift;
  private final int                                         segmentMask;

  private transient Random                                  rndm;

  public ConcurrentDistributedMapDsoArray(final LockType lockType, final ConcurrentDistributedMapDsoConfig config,
                                         final LockStrategy<? super K> lockStrategy) {
    // ensure that DSO is active at construction time, ie. fail fastconcurrency
    Assert.pre(!(ManagerUtil.getManager() instanceof NullManager));
    final int c = config.getConcurrency();

    int sshift = 0;
    int ssize = 1;
    while (ssize < c) {
      ++sshift;
      ssize <<= 1;
    }
    segmentShift = 32 - sshift;
    segmentMask = ssize - 1;
    this.concurrency = ssize;

    this.storeList = new ArrayList<ConcurrentDistributedMapDso<K, V>>(ssize);

    if (storeList instanceof Clearable) {
      // ArrayList isn't currently clearable but it might be someday
      ((Clearable)storeList).setEvictionEnabled(false);
    }

    for (int i = 0; i < ssize; i++) {
      this.storeList.add(i, new ConcurrentDistributedMapDso<K, V>(lockType, lockStrategy));
    }

    onLoad();
  }

  private void onLoad() {
    rndm = new Random();
  }
  /**
   * Applies a supplemental hash function to a given hashCode, which defends against poor quality hash functions.
   */
  private static int hash(int h) {
    h += (h << 15) ^ 0xffffcd7d;
    h ^= (h >>> 10);
    h += (h << 3);
    h ^= (h >>> 6);
    h += (h << 2) + (h << 14);
    return h ^ (h >>> 16);
  }

  /*
   * This method is used to read the storeList instance variable.  As its
   * name starts with __tc_ it avoids the TC instrumentation - it then
   * directly reads storeList (bypassing the resolve lock).  As storeList
   * is final we know if it has a non-null value that should be the correct
   * one.  If the value is null we do a normal locking get (via a regular
   * method which will ge instrumented to use the __tc_ accessor.
   */
  private List<ConcurrentDistributedMapDso<K, V>> __tc_dirtyGetStoreList() {
    if (storeList != null) {
      return storeList;
    } else {
      return cleanGetStoreList();
    }
  }

  private List<ConcurrentDistributedMapDso<K, V>> cleanGetStoreList() {
    return storeList;
  }
  
  private int getIndexFromKey(Object key) {
    int hash = hash(key.hashCode());
    return (hash >>> segmentShift) & segmentMask;
  }

  private ConcurrentDistributedMapDso<K, V> getMapFromKey(Object key) {
    int index = getIndexFromKey(key);
    return __tc_dirtyGetStoreList().get(index);
  }

  public Collection<java.util.Map.Entry<K, V>> getAllEntriesSnapshot() {
    return new AggregateEntriesSnapshotCollection();
  }

  public Collection<java.util.Map.Entry<K, V>> getAllLocalEntriesSnapshot() {
    return new AggregateLocalEntriesSnapshotCollection();
  }

  public void putNoReturn(K key, V value) {
    getMapFromKey(key).putNoReturn(key, value);
  }

  public void removeNoReturn(K key) {
    getMapFromKey(key).removeNoReturn(key);
  }

  public V putIfAbsent(K key, V value) {
    return getMapFromKey(key).putIfAbsent(key, value);
  }

  public boolean remove(Object key, Object value) {
    return getMapFromKey(key).remove(key, value);
  }

  public V replace(K key, V value) {
    return getMapFromKey(key).replace(key, value);
  }

  public boolean replace(K key, V oldValue, V newValue) {
    return getMapFromKey(key).replace(key, oldValue, newValue);
  }

  public void clear() {
    for (int i = 0; i < this.concurrency; i++) {
      this.__tc_dirtyGetStoreList().get(i).clear();
    }
  }

  public boolean containsKey(Object key) {
    return getMapFromKey(key).containsKey(key);
  }

  public boolean containsValue(Object value) {
    for (int i = 0; i < this.concurrency; i++) {
      if (this.__tc_dirtyGetStoreList().get(i).containsValue(value)) { return true; }
    }
    return false;
  }

  public Set<java.util.Map.Entry<K, V>> entrySet() {
    return new AggregateEntrySet();
  }

  public V get(Object key) {
    return getMapFromKey(key).get(key);
  }

  public V unsafeGet(K key) {
    return getMapFromKey(key).unsafeGet(key);
  }

  public boolean isEmpty() {
    for (int i = 0; i < this.concurrency; i++) {
      if (!this.__tc_dirtyGetStoreList().get(i).isEmpty()) { return false; }
    }
    return true;
  }

  public Set<K> keySet() {
    return new AggregateKeySet();
  }

  public V put(K key, V value) {
    return getMapFromKey(key).put(key, value);
  }

  public void putAll(Map<? extends K, ? extends V> m) {
    for (Object element : m.entrySet()) {
      Map.Entry<? extends K, ? extends V> entry = (java.util.Map.Entry<? extends K, ? extends V>) element;
      getMapFromKey(entry.getKey()).put(entry.getKey(), entry.getValue());
    }
  }

  public V remove(Object key) {
    return getMapFromKey(key).remove(key);
  }

  public int size() {
    int size = 0;
    for (int i = 0; i < this.concurrency; i++) {
      size += this.__tc_dirtyGetStoreList().get(i).size();
    }
    return size;
  }

  public int localSize() {
    int localSize = 0;
    for (int i = 0; i < concurrency; i++) {
      localSize += __tc_dirtyGetStoreList().get(i).localSize();
    }
    return localSize;
  }

  public Collection<V> values() {
    return new AggregatedValuesCollection();
  }

  public void lockEntry(K key) {
    getMapFromKey(key).lockEntry(key);
  }

  public void unlockEntry(K key) {
    getMapFromKey(key).unlockEntry(key);
  }

  public String getLockIdForKey(K key) {
    return getMapFromKey(key).getLockIdForKey(key);
  }
  
  public List<Map<K, ?>> getConstituentMaps() {
    return Collections.<Map<K, ?>>unmodifiableList(__tc_dirtyGetStoreList());
  }

  public Map.Entry<K, V> getRandomEntry() {
    ConcurrentDistributedMapDso<K, V> random = __tc_dirtyGetStoreList().get(rndm.nextInt(__tc_dirtyGetStoreList().size()));
    if (!random.isEmpty()) {
      Map.Entry<K, V> entry = random.getRandomEntry();
      if (entry != null) {
        return entry;
      }
    }

    for (ConcurrentDistributedMapDso<K, V> e : __tc_dirtyGetStoreList()) {
      if (!e.isEmpty()) {
        Map.Entry<K, V> entry = e.getRandomEntry();
        if (entry != null) {
          return entry;
        }
      }
    }
    return null;
  }

  public Map.Entry<K, V> getRandomLocalEntry() {
    ConcurrentDistributedMapDso<K, V> random = __tc_dirtyGetStoreList().get(rndm.nextInt(__tc_dirtyGetStoreList().size()));
    if (random.localSize() != 0) {
      Map.Entry<K, V> entry = random.getRandomLocalEntry();
      if (entry != null) {
        return entry;
      }
    }

    for (ConcurrentDistributedMapDso<K, V> e : __tc_dirtyGetStoreList()) {
      if (e.localSize() != 0) {
        Map.Entry<K, V> entry = e.getRandomLocalEntry();
        if (entry != null) {
          return entry;
        }
      }
    }
    
    return null;
  }
  
  public boolean flush(Object key, Object value) {
    return getMapFromKey(key).flush(key, value);
  }

  public boolean tryRemove(Object key, long time, TimeUnit unit) {
    return getMapFromKey(key).tryRemove(key, time, unit);
  }
  
  public MapSizeListener registerMapSizeListener(MapSizeListener newListener) {
    MapSizeListener old = null;
    for (ConcurrentDistributedMapBackend e : __tc_dirtyGetStoreList()) {
      old = e.registerMapSizeListener(newListener);
    }
    return old;
  }
  
  private class AggregateKeySet extends BaseAggregateSet<K> {

    @Override
    public boolean contains(Object o) {
      return ConcurrentDistributedMapDsoArray.this.containsKey(o);
    }

    @Override
    public boolean remove(Object o) {
      return ConcurrentDistributedMapDsoArray.this.remove(o) != null;
    }

    @Override
    public Iterator<K> iterator() {
      return new AggregateSetIterator<K>() {

        @Override
        protected Iterator<K> getNextIterator() {
          return this.listIterator.next().keySet().iterator();
        }

      };
    }

  }

  private class AggregateEntrySet extends BaseAggregateSet<Map.Entry<K, V>> {

    @Override
    public Iterator<Map.Entry<K, V>> iterator() {
      return new AggregateSetIterator<Map.Entry<K, V>>() {

        @Override
        protected Iterator<java.util.Map.Entry<K, V>> getNextIterator() {
          return this.listIterator.next().entrySet().iterator();
        }

      };
    }

    @Override
    public boolean contains(Object o) {
      if (!(o instanceof Map.Entry)) return false;
      Map.Entry<K, V> e = (Map.Entry<K, V>) o;
      V value = ConcurrentDistributedMapDsoArray.this.get(e.getKey());
      return value != null && value.equals(e.getValue());
    }

    @Override
    public boolean remove(Object o) {
      if (!(o instanceof Map.Entry)) return false;
      Map.Entry<K, V> e = (Map.Entry<K, V>) o;
      V value = ConcurrentDistributedMapDsoArray.this.get(e.getKey());
      if (value != null && value.equals(e.getValue())) {
        return ConcurrentDistributedMapDsoArray.this.remove(e.getKey()) != null;
      } else {
        return false;
      }
    }

  }

  private class AggregatedValuesCollection extends AbstractCollection<V> {

    @Override
    public Iterator<V> iterator() {
      return new AggregateSetIterator<V>() {

        @Override
        protected Iterator<V> getNextIterator() {
          return this.listIterator.next().values().iterator();
        }

      };
    }

    @Override
    public int size() {
      return ConcurrentDistributedMapDsoArray.this.size();
    }

  }

  private abstract class BaseAggregateSet<T> extends AbstractSet<T> {

    @Override
    public int size() {
      return ConcurrentDistributedMapDsoArray.this.size();
    }

    @Override
    public void clear() {
      ConcurrentDistributedMapDsoArray.this.clear();
    }
  }

  private abstract class AggregateSetIterator<T> implements Iterator<T> {

    protected final Iterator<ConcurrentDistributedMapDso<K, V>> listIterator;
    protected Iterator<T>                                           currentIterator;

    protected abstract Iterator<T> getNextIterator();

    public AggregateSetIterator() {
      this.listIterator = __tc_dirtyGetStoreList().iterator();
      while (listIterator.hasNext()) {
        currentIterator = getNextIterator();
        if (currentIterator.hasNext()) { return; }
      }
    }

    public boolean hasNext() {

      if (this.currentIterator == null) return false;
      boolean hasNext = false;

      if (this.currentIterator.hasNext()) {
        hasNext = true;
      } else {
        while (this.listIterator.hasNext()) {
          this.currentIterator = getNextIterator();
          if (this.currentIterator.hasNext()) { return true; }
        }
      }
      return hasNext;
    }

    public T next() {
      T item = null;

      if (currentIterator == null) return null;

      if (currentIterator.hasNext()) {
        item = currentIterator.next();

      } else {
        while (listIterator.hasNext()) {
          currentIterator = getNextIterator();

          if (currentIterator.hasNext()) {
            item = currentIterator.next();
            break;
          }
        }
      }

      return item;
    }

    public void remove() {
      currentIterator.remove();
    }

  }
  
  private class AggregateLocalEntriesSnapshotCollection extends AbstractCollection<Map.Entry<K, V>> {

    public Iterator<java.util.Map.Entry<K, V>> iterator() {
      return new AggregateSetIterator<Entry<K,V>>() {
        protected Iterator<java.util.Map.Entry<K, V>> getNextIterator() {
          return this.listIterator.next().getAllLocalEntriesSnapshot().iterator();
        }
      };
    }

    public int size() {
      long size = 0;
      for (ConcurrentDistributedMapDso map : ConcurrentDistributedMapDsoArray.this.__tc_dirtyGetStoreList()) {
        size += map.getAllLocalEntriesSnapshot().size();
        if (size > Integer.MAX_VALUE) {
          return Integer.MAX_VALUE;
        }
      }
      
      return (int) size;
    }
    
    @Override
    public boolean contains(Object o) {
      if (!(o instanceof Map.Entry)) return false;
      Map.Entry<K, V> e = (Map.Entry<K, V>) o;
      ConcurrentDistributedMapDso map = getMapFromKey(e.getKey());
      return map.getAllLocalEntriesSnapshot().contains(o);
    }
    
    @Override
    public boolean remove(Object o) {
      throw new UnsupportedOperationException();
    }
  }
  
  private class AggregateEntriesSnapshotCollection extends AbstractCollection<Map.Entry<K, V>> {

    public Iterator<java.util.Map.Entry<K, V>> iterator() {
      return new AggregateSetIterator<Entry<K,V>>() {
        protected Iterator<java.util.Map.Entry<K, V>> getNextIterator() {
          return this.listIterator.next().getAllEntriesSnapshot().iterator();
        }
      };
    }

    public int size() {
      return ConcurrentDistributedMapDsoArray.this.size();
    }
    
    @Override
    public boolean contains(Object o) {
      if (!(o instanceof Map.Entry)) return false;
      Map.Entry<K, V> e = (Map.Entry<K, V>) o;
      ConcurrentDistributedMapDso map = getMapFromKey(e.getKey());
      return map.getAllEntriesSnapshot().contains(o);
    }
    
    @Override
    public boolean remove(Object o) {
      throw new UnsupportedOperationException();
    }
  }
}
