/*
 * All content copyright (c) 2003-2008 Terracotta, Inc., except as may otherwise be noted in a separate copyright
 * notice. All rights reserved.
 */
package org.terracotta.collections;

import java.util.AbstractCollection;
import java.util.AbstractMap;
import java.util.AbstractSet;
import java.util.Collection;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;

import org.terracotta.repackaged.org.cliffc.high_scale_lib.NonBlockingHashMap;

import com.tc.exception.TCObjectNotFoundException;
import com.tc.logging.TCLogger;
import com.tc.object.ObjectID;
import com.tc.object.SerializationUtil;
import com.tc.object.TCObject;
import com.tc.object.bytecode.AAFairDistributionPolicyMarker;
import com.tc.object.bytecode.Clearable;
import com.tc.object.bytecode.Manageable;
import com.tc.object.bytecode.Manager;
import com.tc.object.bytecode.ManagerUtil;
import com.tc.object.bytecode.NullManager;
import com.tc.object.bytecode.TCMap;
import com.tc.util.Assert;

class ConcurrentDistributedMapDso<K, V> extends AbstractMap<K, V> implements ConcurrentDistributedMapBackend<K, V>,
    TCMap, Manageable, Clearable, AAFairDistributionPolicyMarker {
  private static final TCLogger              LOGGER              = ManagerUtil
                                                                     .getLogger(ConcurrentDistributedMapDso.class
                                                                         .getName());
  private static final Object[]              NO_ARGS             = new Object[0];

  private volatile transient TCObject        $__tc_MANAGED;
  private volatile boolean                   evictionEnabled     = true;

  private final int                          dsoLockType;
  private final NonBlockingHashMap           store;
  private final LockStrategy<? super K>      lockStrategy;

  private volatile transient MapSizeListener listener;
  private transient AtomicInteger            localSize;
  private volatile transient String          instanceDsoLockName = null;

  ConcurrentDistributedMapDso(final LockType lockType, LockStrategy<? super K> lockingStrategy) {
    this(lockType.getDsoLockType(), lockingStrategy);
  }

  /*
   * This factory method is included to make for easier reflective invocation from the applicator. Otherwise the
   * applicator would have to worry about loading the LockType enum in the correct class loader and then go about
   * finding the matching enum etc...
   */
  public static <K, V> ConcurrentDistributedMapDso<K, V> newCDM(int lockLevel, Object strategy) {
    return new ConcurrentDistributedMapDso<K, V>(lockLevel, (LockStrategy<? super K>) strategy);
  }

  private ConcurrentDistributedMapDso(final int lockLevel, LockStrategy<? super K> lockStrategy) {
    // ensure that DSO is active at construction time, ie. fail fast
    Assert.pre(!(ManagerUtil.getManager() instanceof NullManager));

    this.dsoLockType = lockLevel;
    this.store = new NonBlockingHashMap();
    this.lockStrategy = lockStrategy;
    this.localSize = new AtomicInteger(0);
    Assert.post(store != null);
  }

  private String getInstanceDsoLockName() {
    if (instanceDsoLockName != null) { return instanceDsoLockName; }

    instanceDsoLockName = "@" + "CSM" + ((Manageable) this).__tc_managed().getObjectID().toLong() + ":";
    return instanceDsoLockName;
  }

  private String generateLockIdForKey(final K key) {
    if (null == key) { throw new NullPointerException(); }

    return lockStrategy.generateLockIdForKey(getInstanceDsoLockName(), key);
  }

  @Override
  public V put(final K key, final V value) {
    if (__tc_isManaged()) {
      if (null == value) { throw new NullPointerException(); }

      final String lockID = generateLockIdForKey(key);

      Object old;
      boolean sizeChanged = false;

      prefetch(key);
      ManagerUtil.beginLock(lockID, this.dsoLockType);
      try {
        pinLock(lockID);
        old = store.put(key, value);

        if (old != value) {
          ManagerUtil.logicalInvoke(this, SerializationUtil.PUT_SIGNATURE, new Object[] { key, value });

          if (old == null) {
            localSizeIncrement();
            sizeChanged = true;
          } else if (old instanceof ObjectID) {
            localSizeIncrement();
          }
        }

        old = lookup(old);
      } finally {
        ManagerUtil.commitLock(lockID);
      }

      if (sizeChanged) sizeIncrement();

      return (V) old;
    } else {
      V old = (V) store.put(key, value);
      if (old == null) {
        localSizeIncrement();
        sizeIncrement();
      }
      return old;
    }
  }

  public V unsafeGet(K key) {
    Object obj = store.get(key);

    // Performing a lookup here without any locks would be a dirty read and could request a non-existent object
    // (read: Don't change this!)
    if (obj instanceof ObjectID) return null;

    return (V) obj;
  }

  @Override
  public V get(final Object key) {
    if (__tc_isManaged()) {
      Object val = prefetch(key);

      if (val == null && lockStrategy.allowNonCoherentReadsForNonExistentMapping()) {
        // Mapping not present and we are allowed to perform fast non-coherent reads for non-existent mapping.
        return null;
      } else if (! (val instanceof ObjectID) && lockStrategy.allowNonCoherentReadsForLocalEntries()) {
        // Mapping present and is faulted in and we are allowed to perform fast non-coherent reads for locally present mapping.
        return (V) val;
      }

      final String lockID = generateLockIdForKey((K) key);

      ManagerUtil.beginLock(lockID, Manager.LOCK_TYPE_READ);
      try {
        V result = lookupAndFaultIn(key, store.get(key));
        if (result != null) {
          pinLock(lockID);
        }
        return result;
      } finally {
        ManagerUtil.commitLock(lockID);
      }
    } else {
      return (V) store.get(key);
    }
  }

  /**
   * Pre-fetches the value associated with the key, if it is not already faulted into the VM.
   * 
   * @param key
   * @return the current mapping, ObjectID if it is not faulted, Object if faulted, null if no mapping present.
   */
  private Object prefetch(Object key) {
    Object obj = store.get(key);
    if (obj instanceof ObjectID) {
      // XXX::Note since we are reading outside the lock scope this might result in an ObjectNotFound Message sent from
      // the server, but we ignore it since it a result of pre-fetch request.
      ManagerUtil.preFetchObject((ObjectID) obj);
    }
    return obj;
  }

  @Override
  public V remove(final Object key) {
    if (__tc_isManaged()) {
      final String lockID = generateLockIdForKey((K) key);

      Object old;
      boolean sizeChanged = false;

      prefetch(key);
      ManagerUtil.beginLock(lockID, this.dsoLockType);
      try {
        old = store.remove(key);

        if (old != null) {
          ManagerUtil.logicalInvoke(this, SerializationUtil.REMOVE_KEY_SIGNATURE, new Object[] { key });
          if (!(old instanceof ObjectID)) {
            localSizeDecrement();
          }
          sizeChanged = true;
        }
        old = lookup(old);
      } finally {
        ManagerUtil.commitLock(lockID);
        evictLock(lockID);
      }

      if (sizeChanged) sizeDecrement();

      return (V) old;
    } else {
      V old = (V) store.remove(key);
      if (old != null) {
        localSizeDecrement();
        sizeDecrement();
      }
      return old;
    }
  }

  public void removeNoReturn(final K key) {
    if (__tc_isManaged()) {
      final String lockID = generateLockIdForKey(key);

      boolean sizeChanged = false;

      ManagerUtil.beginLock(lockID, this.dsoLockType);
      try {
        Object old = store.remove(key);
        if (old != null) {
          ManagerUtil.logicalInvoke(this, SerializationUtil.REMOVE_KEY_SIGNATURE, new Object[] { key });
          if (!(old instanceof ObjectID)) {
            localSizeDecrement();
          }
          sizeChanged = true;
        }
      } finally {
        ManagerUtil.commitLock(lockID);
        evictLock(lockID);
      }

      if (sizeChanged) sizeDecrement();
    } else {
      if (store.remove(key) != null) {
        localSizeDecrement();
        sizeDecrement();
      }
    }
  }

  public void putNoReturn(K key, V value) {
    if (__tc_isManaged()) {
      final String lockID = generateLockIdForKey(key);

      boolean sizeChanged = false;

      ManagerUtil.beginLock(lockID, this.dsoLockType);
      try {
        pinLock(lockID);
        Object old = store.put(key, value);

        if (old != value) {
          ManagerUtil.logicalInvoke(this, SerializationUtil.PUT_SIGNATURE, new Object[] { key, value });

          if (old == null) {
            localSizeIncrement();
            sizeChanged = true;
          } else if (old instanceof ObjectID) {
            localSizeIncrement();
          }
        }
      } finally {
        ManagerUtil.commitLock(lockID);
      }

      if (sizeChanged) sizeIncrement();
    } else {
      if (store.put(key, value) == null) {
        localSizeIncrement();
        sizeIncrement();
      }
    }
  }

  @Override
  public boolean containsKey(final Object key) {
    if (__tc_isManaged()) {
      final String lockID = generateLockIdForKey((K) key);

      ManagerUtil.beginLock(lockID, Manager.LOCK_TYPE_READ);
      try {
        return store.containsKey(key);
      } finally {
        ManagerUtil.commitLock(lockID);
      }
    } else {
      return store.containsKey(key);
    }
  }

  @Override
  public boolean containsValue(final Object value) {
    if (__tc_isManaged()) {
      if (null == value) { throw new NullPointerException(); }

      for (Entry<K, V> entry : entrySet()) {
        final String lockID = generateLockIdForKey(entry.getKey());

        ManagerUtil.beginLock(lockID, Manager.LOCK_TYPE_READ);
        try {
          if (value.equals(lookupAndFaultIn(entry.getKey(), entry.getValue()))) { return true; }
        } finally {
          ManagerUtil.commitLock(lockID);
        }
      }

      return false;
    } else {
      return store.containsValue(value);
    }
  }

  @Override
  public int size() {
    if (__tc_isManaged()) {
      ManagerUtil.beginLock(getInstanceDsoLockName(), Manager.LOCK_TYPE_READ);
      try {
        return store.size();
      } finally {
        ManagerUtil.commitLock(getInstanceDsoLockName());
      }
    } else {
      return store.size();
    }
  }

  public int localSize() {
    if (__tc_isManaged()) {
      return localSize.get();
    } else {
      return size();
    }
  }

  @Override
  public void clear() {
    if (__tc_isManaged()) {
      ManagerUtil.beginLock(getInstanceDsoLockName(), this.dsoLockType);
      try {
        evictAllLocks();
        int size = store.size();
        store.clear();
        int local = localSize.getAndSet(0);
        if (listener != null) {
          listener.localSizeChanged(-local);
          listener.sizeChanged(-size);
        }
        ManagerUtil.logicalInvoke(this, SerializationUtil.CLEAR_SIGNATURE, NO_ARGS);
      } finally {
        ManagerUtil.commitLock(getInstanceDsoLockName());
      }
    } else {
      int local = localSize.getAndSet(0);
      store.clear();
      int size = store.size();
      if (listener != null) {
        listener.localSizeChanged(-local);
        listener.sizeChanged(-size);
      }
    }
  }

  @Override
  public Set<K> keySet() {
    return new KeySet(store.keySet());
  }

  @Override
  public Set<Entry<K, V>> entrySet() {
    return new EntrySet(store.entrySet());
  }

  private String generateLockIdForEntry(final Object o) {
    if (!(o instanceof Entry)) { return null; }
    Entry<K, V> entry = (Entry<K, V>) o;
    final K key = entry.getKey();
    final String lockID = generateLockIdForKey(key);
    return lockID;
  }

  /*
   * ConcurrentMap methods
   */

  public V putIfAbsent(final K key, final V value) {
    if (__tc_isManaged()) {
      if (null == value) { throw new NullPointerException(); }

      final String lockID = generateLockIdForKey(key);

      Object old;
      boolean sizeChanged = false;

      prefetch(key);
      ManagerUtil.beginLock(lockID, this.dsoLockType);
      try {
        pinLock(lockID);
        old = store.putIfAbsent(key, value);
        if (old == null) {
          ManagerUtil.logicalInvoke(this, SerializationUtil.PUT_SIGNATURE, new Object[] { key, value });
          localSizeIncrement();
          sizeChanged = true;
        }

        old = lookupAndFaultIn(key, old);
      } finally {
        ManagerUtil.commitLock(lockID);
      }

      if (sizeChanged) sizeIncrement();

      return (V) old;
    } else {
      V old = (V) store.putIfAbsent(key, value);
      if (old == null) {
        localSizeIncrement();
        sizeIncrement();
      }
      return old;
    }
  }

  public boolean remove(final Object key, final Object value) {
    if (__tc_isManaged()) {
      if (null == value) { throw new NullPointerException(); }

      final String lockID = generateLockIdForKey((K) key);

      boolean success = false;

      ManagerUtil.beginLock(lockID, dsoLockType);
      try {
        fault(key);
        if (store.remove(key, value)) {
          ManagerUtil.logicalInvoke(this, SerializationUtil.REMOVE_KEY_SIGNATURE, new Object[] { key });
          localSizeDecrement();
          success = true;
        } else {
          success = false;
        }
      } finally {
        ManagerUtil.commitLock(lockID);
        if (success) evictLock(lockID);
      }

      if (success) sizeDecrement();

      return success;
    } else {
      if (store.remove(key, value)) {
        localSizeDecrement();
        sizeDecrement();
        return true;
      } else {
        return false;
      }
    }
  }

  public boolean replace(final K key, final V oldValue, final V newValue) {
    if (__tc_isManaged()) {
      if (null == oldValue) { throw new NullPointerException(); }
      if (null == newValue) { throw new NullPointerException(); }

      final String lockID = generateLockIdForKey(key);

      prefetch(key);
      ManagerUtil.beginLock(lockID, dsoLockType);
      try {
        pinLock(lockID);
        fault(key);
        if (store.replace(key, oldValue, newValue)) {
          ManagerUtil.logicalInvoke(this, SerializationUtil.PUT_SIGNATURE, new Object[] { key, newValue });
          return true;
        } else {
          return false;
        }
      } finally {
        ManagerUtil.commitLock(lockID);
      }
    } else {
      return store.replace(key, oldValue, newValue);
    }
  }

  public V replace(final K key, final V value) {
    if (__tc_isManaged()) {
      if (null == value) { throw new NullPointerException(); }

      final String lockID = generateLockIdForKey(key);

      prefetch(key);
      ManagerUtil.beginLock(lockID, dsoLockType);
      try {
        Object old = store.replace(key, value);
        if (old != null) {
          ManagerUtil.logicalInvoke(this, SerializationUtil.PUT_SIGNATURE, new Object[] { key, value });
          if (old instanceof ObjectID) {
            localSizeIncrement();
          }
        }
        return lookup(old);
      } finally {
        ManagerUtil.commitLock(lockID);
      }
    } else {
      return (V) store.replace(key, value);
    }
  }

  public Collection<Entry<K, V>> getAllEntriesSnapshot() {
    return Collections.unmodifiableCollection(store.entrySet());
  }

  public Collection<Entry<K, V>> getAllLocalEntriesSnapshot() {
    return new LocalEntriesCollection();
  }

  private class KeySet extends AbstractSet<K> {

    private final Set<K> delegate;

    private KeySet(final Set<K> delegate) {
      this.delegate = delegate;
    }

    @Override
    public int size() {
      return ConcurrentDistributedMapDso.this.size();
    }

    @Override
    public boolean contains(Object key) {
      return ConcurrentDistributedMapDso.this.containsKey(key);
    }

    @Override
    public void clear() {
      ConcurrentDistributedMapDso.this.clear();
    }

    @Override
    public boolean remove(Object key) {
      return ConcurrentDistributedMapDso.this.remove(key) != null;
    }

    @Override
    public Iterator<K> iterator() {
      return new KeyIterator(delegate.iterator());
    }

  }

  private class KeyIterator implements Iterator<K> {

    private final Iterator<K> delegate;
    private K                 lastKey;

    public KeyIterator(Iterator<K> delegate) {
      this.delegate = delegate;
    }

    public boolean hasNext() {
      return delegate.hasNext();
    }

    public K next() {
      final K result = delegate.next();
      lastKey = result;
      return result;
    }

    public void remove() {
      if (null == lastKey) { throw new IllegalStateException("next needs to be called before calling remove"); }

      if (ManagerUtil.isManaged(ConcurrentDistributedMapDso.this)) {
        final String lockID = generateLockIdForKey(lastKey);
        if (null == lockID) { throw new IllegalStateException("can't obtain lockId for entry"); }

        ConcurrentDistributedMapDso.this.remove(lastKey);
      } else {
        delegate.remove();
      }

      lastKey = null;
    }

  }

  private class EntrySet extends AbstractSet<Entry<K, V>> {

    private final Set<Entry<K, V>> delegate;

    private EntrySet(final Set<Entry<K, V>> delegate) {
      this.delegate = delegate;
    }

    @Override
    public Iterator<Entry<K, V>> iterator() {
      return new EntryIterator(delegate.iterator());
    }

    @Override
    public int size() {
      return ConcurrentDistributedMapDso.this.size();
    }

    @Override
    public boolean contains(final Object o) {
      if (ManagerUtil.isManaged(ConcurrentDistributedMapDso.this)) {
        final String lockID = generateLockIdForEntry(o);
        if (null == lockID) { return false; }

        ManagerUtil.beginLock(lockID, Manager.LOCK_TYPE_READ);
        try {
          fault(((Entry) o).getKey());
          return delegate.contains(o);
        } finally {
          ManagerUtil.commitLock(lockID);
        }
      } else {
        return delegate.contains(o);
      }
    }

    @Override
    public boolean remove(final Object o) {
      if (ManagerUtil.isManaged(ConcurrentDistributedMapDso.this)) {
        final String lockID = generateLockIdForEntry(o);
        if (null == lockID) { return false; }

        boolean success = false;
        Entry e = (Entry) o;
        prefetch(e.getKey());
        ManagerUtil.beginLock(lockID, dsoLockType);
        try {
          /*
           * This works because the NBHMs entry set is backed directly by the map. The fault in below will affect the
           * comparison done when remove(Object o) is called on the entry set. The remove will see the value faulted
           * into the map and not some ObjectID cached from when the set was created.
           */
          fault(e.getKey());
          if (delegate.remove(o)) {
            ManagerUtil.logicalInvoke(ConcurrentDistributedMapDso.this, SerializationUtil.REMOVE_KEY_SIGNATURE,
                                      new Object[] { e.getKey() });
            localSizeDecrement();
            success = true;
          } else {
            success = false;
          }
        } finally {
          ManagerUtil.commitLock(lockID);
          if (success) evictLock(lockID);
        }

        if (success) sizeDecrement();

        return success;
      } else {
        if (delegate.remove(o)) {
          localSizeDecrement();
          sizeDecrement();
          return true;
        } else {
          return false;
        }
      }
    }

    @Override
    public void clear() {
      ConcurrentDistributedMapDso.this.clear();
    }
  }

  private class EntryIterator implements Iterator<Entry<K, V>> {

    private final Iterator<Entry<K, V>> delegate;
    private Entry<K, V>                 lastEntry;

    private EntryIterator(final Iterator<Entry<K, V>> delegate) {
      this.delegate = delegate;
    }

    public boolean hasNext() {
      return delegate.hasNext();
    }

    public synchronized Entry<K, V> next() {
      final Entry<K, V> result = delegate.next();
      if (null == result) {
        lastEntry = null;
        return null;
      }
      lastEntry = new TcmEntry(result);
      return lastEntry;
    }

    public synchronized void remove() {
      if (null == lastEntry) { throw new IllegalStateException("next needs to be called before calling remove"); }

      if (ManagerUtil.isManaged(ConcurrentDistributedMapDso.this)) {
        final String lockID = generateLockIdForEntry(lastEntry);
        if (null == lockID) { throw new IllegalStateException("can't obtain lockId for entry"); }

        boolean success = false;
        prefetch(lastEntry.getKey());
        ManagerUtil.beginLock(lockID, dsoLockType);
        try {
          /*
           * This is weird. In order to ensure compatibility of all L1 states we have to operate directly on the map.
           * Otherwise mutations done to the map after construction of the iterator (which may not be visible to the
           * iterator) may be overwritten on nodes receiving the remove logical invoke even though the iterator remove
           * failed. Hence we will get divergence in L1 states.
           */
          fault(lastEntry.getKey());
          if (ConcurrentDistributedMapDso.this.store.remove(lastEntry.getKey(), lastEntry.getValue())) {
            ManagerUtil.logicalInvoke(ConcurrentDistributedMapDso.this, SerializationUtil.REMOVE_KEY_SIGNATURE,
                                      new Object[] { lastEntry.getKey() });
            localSizeDecrement();
            success = true;
          }
          delegate.remove();
        } finally {
          ManagerUtil.commitLock(lockID);
          if (success) evictLock(lockID);
        }

        if (success) sizeDecrement();
      } else {
        if (ConcurrentDistributedMapDso.this.store.remove(lastEntry.getKey(), lastEntry.getValue())) {
          delegate.remove();
          localSizeDecrement();
          sizeDecrement();
        }
      }
      lastEntry = null;
    }
  }

  private class TcmEntry implements Entry<K, V> {

    private final Entry<K, V> delegate;

    private TcmEntry(final Entry<K, V> delegate) {
      this.delegate = delegate;
    }

    public K getKey() {
      // entry locking here doesn't make sense since
      // 1. Entry keys can't change
      // 2. you're have to get the key anyway to generate the lock ID
      return delegate.getKey();
    }

    /*
     * This does not trigger the faulting of other values within the map. Fixing the implementation to do this is just a
     * matter of changing the ManagerUtil call from lookupObject to lookupObjectWithParentContext.
     */
    public V getValue() {
      if (ManagerUtil.isManaged(ConcurrentDistributedMapDso.this)) {
        final String lockId = generateLockIdForKey(delegate.getKey());
        prefetch(delegate.getKey());
        ManagerUtil.beginLock(lockId, Manager.LOCK_TYPE_READ);
        try {
          return lookupAndFaultIn(delegate.getKey(), delegate.getValue());
        } finally {
          ManagerUtil.commitLock(lockId);
        }
      } else {
        return delegate.getValue();
      }
    }

    public V setValue(final V value) {
      if (ManagerUtil.isManaged(ConcurrentDistributedMapDso.this)) {
        final String lockId = generateLockIdForKey(delegate.getKey());
        prefetch(delegate.getKey());
        ManagerUtil.beginLock(lockId, dsoLockType);
        try {
          pinLock(lockId);
          Object old = delegate.setValue(value);
          if (old != value) {
            ManagerUtil.logicalInvoke(ConcurrentDistributedMapDso.this, SerializationUtil.PUT_SIGNATURE, new Object[] {
                delegate.getKey(), value });
            if (old instanceof ObjectID) {
              localSizeIncrement();
            }
          }
          return lookup(old);
        } finally {
          ManagerUtil.commitLock(lockId);
        }
      } else {
        return delegate.setValue(value);
      }
    }
  }

  /**
   * If the value associated with the supplied key is an ObjectID then the value is looked up and faulted into the map.
   * This operation is safe in the absence of all locking, but no guarantees can then be made about subsequent gets of
   * the key.
   */
  private void fault(Object key) {
    Object obj = store.get(key);
    if (obj instanceof ObjectID) {
      try {
        V value = (V) ManagerUtil.lookupObject((ObjectID) obj);
        if (store.replace(key, obj, value)) {
          localSizeIncrement();
        }
      } catch (TCObjectNotFoundException e) {
        LOGGER.info("Missing object caused by concurrent map-level and key-level operations\n"
                    + "Removing local entry for this key [" + key + "] to restore L1-L2 correspondence.", e);
        store.remove(key, obj);
        evictLock(generateLockIdForKey((K) key));
      }
    }
  }

  /**
   * If the passed object is an ObjectID looks it up and returns it, after attempting to replace the original ObjectID
   * value in the map. Safe in the absence of all locking.
   */
  private V lookupAndFaultIn(Object key, Object obj) {
    if (obj instanceof ObjectID) {
      try {
        V value = (V) ManagerUtil.lookupObject((ObjectID) obj);
        // We should really assert that the replace succeeds here
        if (store.replace(key, obj, value)) {
          localSizeIncrement();
        }
        return value;
      } catch (TCObjectNotFoundException e) {
        LOGGER.info("Missing object caused by concurrent map-level and key-level operations\n"
                    + "Removing local entry for this key [" + key + "] to restore L1-L2 correspondence.", e);
        store.remove(key, obj);
        evictLock(generateLockIdForKey((K) key));
        return null;
      }
    } else {
      return (V) obj;
    }
  }

  /**
   * If the passed object is an ObjectID, looks it up and returns it.
   */
  private V lookup(Object obj) {
    if (obj instanceof ObjectID) {
      try {
        return (V) ManagerUtil.lookupObject((ObjectID) obj);
      } catch (TCObjectNotFoundException e) {
        LOGGER.info("Missing object caused by concurrent map-level and key-level operations\n"
                    + "Returning null as the only sensible result.  Caller should have removed the local mapping.", e);
        return null;
      }
    } else {
      return (V) obj;
    }
  }

  public void __tc_applicator_clear() {
    store.clear();
  }

  public void __tc_applicator_put(Object key, Object value) {
    Object old = store.put(key, value);
    if (value instanceof ObjectID) {
      if (old != null && !(old instanceof ObjectID)) {
        localSizeDecrement();
      }
    } else {
      if (old == null || old instanceof ObjectID) {
        localSizeIncrement();
      }
    }
  }

  public void __tc_applicator_remove(Object key) {
    Object old = store.remove(key);
    if (old != null && !(old instanceof ObjectID)) {
      localSizeDecrement();
    }
  }

  public Collection __tc_getAllEntriesSnapshot() {
    return getAllEntriesSnapshot();
  }

  public Collection __tc_getAllLocalEntriesSnapshot() {
    return getAllLocalEntriesSnapshot();
  }

  public void __tc_put_logical(Object key, Object value) {
    putNoReturn((K) key, (V) value);
  }

  public void __tc_remove_logical(Object key) {
    removeNoReturn((K) key);
  }

  public boolean __tc_isManaged() {
    return $__tc_MANAGED != null;
  }

  public TCObject __tc_managed() {
    return $__tc_MANAGED;
  }

  public void __tc_managed(TCObject tcObject) {
    $__tc_MANAGED = tcObject;
  }

  public int __tc_clearReferences(int toClear) {
    if (!__tc_isManaged()) { throw new AssertionError("clearReferences() called on Unmanaged Map"); }

    int cleared = 0;
    for (Object key : store.keySet()) {
      Object v = store.get(key);
      if (v instanceof ObjectID) continue;

      TCObject tcObject = ManagerUtil.lookupExistingOrNull(v);
      if (tcObject != null && !tcObject.recentlyAccessed()) {
        ObjectID oid = tcObject.getObjectID();
        if (store.replace(key, v, oid)) {
          localSizeDecrement();
          evictLock(generateLockIdForKey((K) key));
          if (++cleared == toClear) break;
        }
      }
    }
    return cleared;
  }

  public boolean isEvictionEnabled() {
    return evictionEnabled;
  }

  public void setEvictionEnabled(boolean flag) {
    evictionEnabled = flag;
  }

  private static void pinLock(String lock) {
    ManagerUtil.pinLock(lock);
  }

  private static void evictLock(String lock) {
    ManagerUtil.evictLock(lock);
  }

  private void evictAllLocks() {
    for (K key : (Set<K>) store.keySet()) {
      evictLock(generateLockIdForKey(key));
    }
  }

  public void lockEntry(K key) {
    if (__tc_isManaged()) {
      /*
       * Note this allows locking on keys that are not present in the map. Hibernate uses this to lock a key prior to
       * creating a mapping for it.
       */
      String lockId = generateLockIdForKey(key);
      ManagerUtil.beginLock(lockId, dsoLockType);
      pinLock(lockId);
    }
  }

  public void unlockEntry(K key) {
    if (__tc_isManaged()) {
      String lockId = generateLockIdForKey(key);
      boolean validKey = containsKey(key); // should have write lock on this key here
      ManagerUtil.commitLock(lockId);
      if (!validKey) {
        /*
         * If we were locking a key that is not in the map we must do our own cleanup. Cannot rely on the map to ever
         * evict otherwise. Would be a resource leak...
         */
        evictLock(lockId);
      }
    }
  }

  public String getLockIdForKey(K key) {
    if (__tc_isManaged()) { return generateLockIdForKey(key); }

    return "";
  }

  public List<Map<K, ?>> getConstituentMaps() {
    return Collections.<Map<K, ?>> singletonList(this);
  }

  public Map.Entry<K, V> getRandomEntry() {
    if (isEmpty()) {
      return null;
    } else {
      return store.getRandomEntry();
    }
  }

  public Map.Entry<K, V> getRandomLocalEntry() {
    if (localSize() == 0) {
      return null;
    } else {
      return store.getRandomEntryNotTyped(null, ObjectID.class);
    }
  }

  public boolean flush(Object key, Object value) {
    if (__tc_isManaged()) {
      if (value instanceof ObjectID) { return false; }

      TCObject tcObject = ManagerUtil.lookupExistingOrNull(value);
      if (tcObject == null) { return false; }

      ObjectID oid = tcObject.getObjectID();
      boolean success = store.replace(key, value, oid);
      if (success) {
        localSizeDecrement();
        evictLock(generateLockIdForKey((K) key));
      }

      return success;
    } else {
      return remove(key, value);
    }
  }

  public boolean tryRemove(Object key, long time, TimeUnit unit) {
    if (__tc_isManaged()) {
      final String lockID = generateLockIdForKey((K) key);

      if (ManagerUtil.tryBeginLock(lockID, dsoLockType, unit.toNanos(time))) {
        boolean removed = false;

        try {
          Object old = store.remove(key);
          if (old != null) {
            ManagerUtil.logicalInvoke(this, SerializationUtil.REMOVE_KEY_SIGNATURE, new Object[]{key});
            if (!(old instanceof ObjectID)) {
              localSizeDecrement();
            }
            removed = true;
          }
        } finally {
          ManagerUtil.commitLock(lockID);
          evictLock(lockID);
        }
        if (removed) {
          sizeDecrement();
        }

        return removed;
      } else {
        return false;
      }
    } else {
      if (store.remove(key) != null) {
        localSizeDecrement();
        sizeDecrement();
        return true;
      } else {
        return false;
      }
    }
  }

  private class LocalEntriesCollection extends AbstractCollection<Entry<K, V>> {

    private final Collection<Entry<K, V>> delegate = store.entrySet();

    @Override
    public Iterator<java.util.Map.Entry<K, V>> iterator() {
      return new LocalEntriesIterator<K, V>(delegate.iterator());
    }

    @Override
    public int size() {
      int size = 0;
      for (Object o : store.values()) {
        if (!(o instanceof ObjectID)) {
          size++;
        }
      }

      return size;
    }

    @Override
    public boolean contains(Object o) {
      return delegate.contains(o);
    }

    @Override
    public boolean remove(Object o) {
      throw new UnsupportedOperationException();
    }
  }

  private static class LocalEntriesIterator<K, V> implements Iterator<Entry<K, V>> {

    private final Iterator<Entry<K, V>> delegate;

    private Entry<K, V>                 next;

    public LocalEntriesIterator(Iterator<Entry<K, V>> delegate) {
      this.delegate = delegate;
      findNext();
    }

    public boolean hasNext() {
      return next != null;
    }

    public java.util.Map.Entry<K, V> next() {
      Entry result = findNext();
      if (result == null) {
        throw new NoSuchElementException();
      } else {
        return result;
      }
    }

    public void remove() {
      throw new UnsupportedOperationException();
    }

    private Entry<K, V> findNext() {
      Entry<K, V> current = next;

      while (delegate.hasNext()) {
        Entry obj = delegate.next();
        if (!(obj.getValue() instanceof ObjectID)) {
          next = obj;
          return current;
        }
      }

      next = null;
      return current;
    }
  }

  private void localSizeIncrement() {
    localSize.incrementAndGet();
    MapSizeListener l = listener;
    if (l != null) {
      l.localSizeChanged(1);
    }
  }

  private void localSizeDecrement() {
    localSize.decrementAndGet();
    MapSizeListener l = listener;
    if (l != null) {
      l.localSizeChanged(-1);
    }
  }

  private void sizeIncrement() {
    MapSizeListener l = listener;
    if (l != null) {
      l.sizeChanged(1);
    }
  }

  private void sizeDecrement() {
    MapSizeListener l = listener;
    if (l != null) {
      l.sizeChanged(-1);
    }
  }

  public MapSizeListener registerMapSizeListener(MapSizeListener newListener) {
    MapSizeListener old = listener;
    listener = newListener;
    return old;
  }
}
