package org.terracotta.ehcachedx.monitor.probe.counter.sampled.memory;

import java.lang.reflect.Field;

import net.sf.ehcache.Ehcache;
import net.sf.ehcache.Element;

import org.apache.commons.math.stat.descriptive.DescriptiveStatistics;
import org.apache.commons.math.stat.descriptive.SynchronizedDescriptiveStatistics;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.terracotta.ehcachedx.com.javabi.sizeof.ClassDefinitionMap;
import org.terracotta.ehcachedx.com.javabi.sizeof.IdentityHashSet;
import org.terracotta.ehcachedx.com.javabi.sizeof.MemoryUtil;

/**
 * one memorySamplerSupport is one per sampled statistic
 * 
 */
public class MemorySamplerSupport {

	private static final Logger LOG = LoggerFactory
			.getLogger(MemorySamplerSupport.class);

	private static final int WINDOW_SIZE = 100;
	private final DescriptiveStatistics stats;

	private final boolean memoryMeasurement;

	public MemorySamplerSupport(boolean memoryMeasurement) {
		this.memoryMeasurement = memoryMeasurement;
		stats = new SynchronizedDescriptiveStatistics(WINDOW_SIZE);
	}

	/**
	 * Notifies the sampler that a put or an update occurred. The sampler may
	 * choose to sample.
	 */
	public void notifyMemorySampler(Ehcache ehcache, Element element,
			Long counter) {
		if (shouldSample(ehcache, counter)) {
			sample(ehcache, element);
		}
	}

	/**
	 * When first started, sample each put until we get to 15 samples. Then once
	 * every 100.
	 * 
	 * @return true if should sample
	 */
	public boolean shouldSample(Ehcache ehcache, Long putCounter) {

		// int maxMemorySize =
		// ehcache.getCacheConfiguration().getMaxElementsInMemory();
		return ((putCounter.longValue() % 100 == 0) || putCounter.longValue() <= 15);
	}

	public Long getLastAdjustedSample() {
		return java.lang.Math.round(stats.getPercentile(97));
	}

	public DescriptiveStatistics getStatistics() {
		return stats;
	}

	protected void recordValue(long size) {
		if (size != 0) {
			stats.addValue(size);
		} else {
			// If one sampling fails, then the whole thing is suspect. Clear it.
			stats.clear();
		}
	}

	private long sample(Ehcache ehcache, Element obj) {
		long val = 0;
		if (ehcache.getCacheConfiguration().isTerracottaClustered()) {
			val = tcEstimateSize(obj);
		} else {
			val = estimateSize(obj);
		}
		recordValue(val);
		return val;
	}

	/**
	 * TODO: THIS FUNCTION CURRENTLY BREAKS RUNNING CLUSTERED EHCACHE TERRACOTTA
	 * TODO: SEE DEV-4094
	 * 
	 * todo: plugin options, check safety and liveness todo configurable and
	 * queue etc. for heavy writes
	 * 
	 * @return the size in bytes, or 0 if the size could not be calculated
	 */
	protected long estimateSize(Object object) {
		long valueInBytes = 0;
		if (memoryMeasurement == false) {
			return 0;
		}
		try {
			valueInBytes = MemoryUtil.sizeOf(object);
		} catch (Throwable t) {
			LOG.debug("Could not calculate memory for object " + object
					+ ". Returning 0", t);
		}
		return valueInBytes;
	}

	private Field getField(Object o, String name) {
		for (Field f : o.getClass().getDeclaredFields()) {
			if (f.getName().equals(name)) {
				return f;
			}
		}
		return null;
	}

	protected long tcEstimateSize(Element element) {
		long valueInBytes = 0;
		if (memoryMeasurement == false) {
			return 0;
		}
		try {
			ClassDefinitionMap definitionMap = new ClassDefinitionMap() {
				@Override
				public final boolean shouldIgnoreField(Field field) {
					if (field.getName().startsWith("$__tc_")) {
						return true;
					}
					return super.shouldIgnoreField(field);
				}
			};
			Field field = null;
			try {
				Field elementEvictionData = getField(element,
						"elementEvictionData");
				if (elementEvictionData != null) {
					elementEvictionData.setAccessible(true);
					Object evd = elementEvictionData.get(element);
					field = getField(evd, "store");
				}
			} catch (Exception e) {
				/**/
			}
			if (field != null) {
				definitionMap.ignoreField(field);
			}
			valueInBytes += MemoryUtil.sizeOf(element, definitionMap,
					new IdentityHashSet());
		} catch (Throwable t) {
			LOG.debug("Could not calculate memory for Element " + element
					+ ". Returning 0", t);
		}
		return valueInBytes;
	}
}
